/**
* @file UniversalResourceCompiler.cpp
*
* Is used to compile the motion net for the special actions
* and for generating a variety of other files.
*
* @author Uwe Dffert
* @author Martin Ltzsch
*/

#include "Representations/Motion/MotionRequest.h"
#include "Representations/Motion/JointData.h"
#include "Representations/Motion/LEDRequest.h"
#include "Representations/Motion/HeadControlMode.h"
#include "Tools/KickSelectionTable.h"
#include "Representations/Sound/SoundRequest.h"
#include "Tools/StringFunctions.h"
#define USE_GETMESSAGEIDNAME
#include "Tools/MessageQueue/MessageIDs.h"
#include "Platform/File.h"
#include <fstream>
#include <iostream>
#include <string>
#include <io.h> 

#define MAXLAB 5000
int numOfLabels=0;
char* label_motion[MAXLAB];
char* label_name[MAXLAB];
short label_number[MAXLAB];

#define MAXLIN 32000
int numOfLines=0;
char motion[512];
int actMotionID=-1;
char* line_data[MAXLIN];
short line_number[MAXLIN];
short line_motionID[MAXLIN];

#define MAXFIL 500
int numOfFiles=0;
char* file_name[MAXFIL];
short file_startindex[MAXFIL];

int jumpTable[SpecialActionRequest::numOfSpecialAction];

void generateKickSelectionXml()
{
  int i;
  char s[500];
  sprintf(s,"%s/Src/Modules/BehaviorControl/CommonXabsl2Symbols/kick-selection-symbols.xml",File::getGTDir());
  printf("generating %s\n",s);  
  std::ofstream kickSelectionSymbolsXml;
  kickSelectionSymbolsXml.open(s,std::ios::out);
  
  kickSelectionSymbolsXml << "<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?>\n";
  kickSelectionSymbolsXml << "<!--\n";
  kickSelectionSymbolsXml << "This file was generated by \"URC /kickSelectionXml\"\n";
  kickSelectionSymbolsXml << "-->\n";
  kickSelectionSymbolsXml << "<symbols xmlns=\"http://www.ki.informatik.hu-berlin.de/XABSL2.2\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ki.informatik.hu-berlin.de/XABSL2.2\n";
  kickSelectionSymbolsXml << "../../../../Tools/Xabsl2/xabsl-2.2/xabsl-2.2.symbols.xsd\" id=\"kick-selection-symbols\" title=\"Kick Selection Symbols\" description=\"Symbols for kick selection using kick selection tables\">\n";

  kickSelectionSymbolsXml << "\t<decimal-input-function name=\"retrieve-kick\" description=\"Retrieves a good kick from a kick selection table.\" measure=\"int\">\n";
	kickSelectionSymbolsXml << "\t\t<parameter name=\"retrieve-kick.angle\" measure=\"deg\" range=\"-180..180\" description=\"The desired kick direction relative to the robot\"/>\n";
	kickSelectionSymbolsXml << "\t\t<parameter name=\"retrieve-kick.table-id\" measure=\"int\" range=\"0..\" description=\"The id of the kick selection table to be used\"/>\n";
	kickSelectionSymbolsXml << "\t</decimal-input-function>\n";

  for (i=0;i<KickSelectionTable::numberOfActions;i++)
  {
    getXmlString(s,KickSelectionTable::getActionName((KickSelectionTable::ActionID)i));
    kickSelectionSymbolsXml << "\t<constant name=\"action." << s 
      << "\" description=\"The number of the action " 
      << KickSelectionTable::getActionName((KickSelectionTable::ActionID)i) 
      << ".\" measure=\"int\" value=\"" << i << "\"/>\n";
  }
  
  for (i=0;i<KickSelectionTable::numberOfKickSelectionTableIDs;i++)
  {
    getXmlString(s,KickSelectionTable::getKickSelectionTableIDName((KickSelectionTable::KickSelectionTableID)i));
    kickSelectionSymbolsXml << "\t<constant name=\"kick-selection-table." << s 
      << "\" description=\"The id of the kick selection table " 
      << KickSelectionTable::getKickSelectionTableIDName((KickSelectionTable::KickSelectionTableID)i) 
      << ".\" measure=\"int\" value=\"" << i << "\"/>\n";
  }

  kickSelectionSymbolsXml << "</symbols>\n";
  
  kickSelectionSymbolsXml.close();
}

void generateSpecialActionXml()
{
  int i;
  char s[256];
  sprintf(s,"%s/Src/Modules/BehaviorControl/CommonXabsl2Symbols/special-action-symbols.xml",File::getGTDir());
  printf("generating %s\n",s);
  std::ofstream specialActionSymbolsXml;
  specialActionSymbolsXml.open(s,std::ios::out);
  
  specialActionSymbolsXml << "<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?>\n";
  specialActionSymbolsXml << "<!--\n";
  specialActionSymbolsXml << "This file was generated by \"URC /specialActionXml\"\n";
  specialActionSymbolsXml << "-->\n";
  specialActionSymbolsXml << "<symbols xmlns=\"http://www.ki.informatik.hu-berlin.de/XABSL2.2\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ki.informatik.hu-berlin.de/XABSL2.2\n";
  specialActionSymbolsXml << "../../../Tools/Xabsl2/xabsl-2.2/xabsl-2.2.symbols.xsd\" id=\"special-action-symbols\" title=\"Special Action Symbols\" description=\"Symbols for the special actions that can be performed.\">\n";
    
  for (i=0;i<SpecialActionRequest::numOfSpecialAction;i++)
  {
    getXmlString(s,SpecialActionRequest::getSpecialActionIDName((SpecialActionRequest::SpecialActionID)i));
    specialActionSymbolsXml << "\t<constant name=\"special-action." << s << "\" description=\"The number of a special action used by the special-action skill\" measure=\"\" value=\"" << i << "\"/>\n";
  }
  
  specialActionSymbolsXml << "</symbols>\n";
  
  specialActionSymbolsXml.close();
}

void generateHeadTailXml()
{
  int i;
  char s[256];
  sprintf(s,"%s/Src/Modules/BehaviorControl/CommonXabsl2Symbols/head-and-tail-symbols.xml",File::getGTDir());
  printf("generating %s\n",s);
  std::ofstream headAndTailSymbolsXml;
  headAndTailSymbolsXml.open(s,std::ios::out);
  
  headAndTailSymbolsXml << "<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?>\n";
  headAndTailSymbolsXml << "<!--\n";
  headAndTailSymbolsXml << "This file was generated by \"URC /headTailXml\"\n";
  headAndTailSymbolsXml << "-->\n";
  headAndTailSymbolsXml << "<symbols xmlns=\"http://www.ki.informatik.hu-berlin.de/XABSL2.2\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ki.informatik.hu-berlin.de/XABSL2.2\n";
  headAndTailSymbolsXml << "../../../Tools/Xabsl2/xabsl-2.2/xabsl-2.2.symbols.xsd\" id=\"head-and-tail-symbols\" title=\"Head and Tail Symbols\" description=\"Symbols for how to move the head and the tail\">\n";
  
  headAndTailSymbolsXml << "\t<enumerated-output-symbol name=\"head-control-mode\" description=\"a mode for the head control. (controls the priorities where the head looks at)\">\n";
  for (i=0;i<HeadControlMode::numOfHeadControlModes;i++)
  {
    getXmlString(s,HeadControlMode::getHeadControlModeName((HeadControlMode::HeadControlModes)i));
    headAndTailSymbolsXml << "\t\t<enum-element name=\"head-control-mode." << s << "\"/>\n";
  }
  headAndTailSymbolsXml << "\t</enumerated-output-symbol>\n";
  
  headAndTailSymbolsXml << "\t<enumerated-output-symbol name=\"tail-mode\" description=\"A mode for the tail.\">\n";
  for (i=0;i<TailRequest::numOfTailRequests;i++)
  {
    getXmlString(s, TailRequest::getTailRequestIDName((TailRequest::TailRequestID)i));
    headAndTailSymbolsXml << "\t\t<enum-element name=\"tail-mode." << s << "\"/>\n";
  }
  headAndTailSymbolsXml << "\t</enumerated-output-symbol>\n";
  
  headAndTailSymbolsXml << "</symbols>\n";
  
  headAndTailSymbolsXml.close();
}

void generateLEDSoundXml()
{
  int i;
  char s[256];
  sprintf(s,"%s/Src/Modules/BehaviorControl/CommonXabsl2Symbols/led-and-sound-symbols.xml",File::getGTDir());
  printf("generating %s\n",s);
  std::ofstream ledAndSoundSymbolsXml;
  ledAndSoundSymbolsXml.open(s,std::ios::out);
  
  ledAndSoundSymbolsXml << "<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?>\n";
  ledAndSoundSymbolsXml << "<!--\n";
  ledAndSoundSymbolsXml << "This file was generated by \"URC /ledSoundXml\"\n";
  ledAndSoundSymbolsXml << "-->\n";
  ledAndSoundSymbolsXml << "<symbols xmlns=\"http://www.ki.informatik.hu-berlin.de/XABSL2.2\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ki.informatik.hu-berlin.de/XABSL2.2\n";
  ledAndSoundSymbolsXml << "../../../Tools/Xabsl2/xabsl-2.2/xabsl-2.2.symbols.xsd\" id=\"led-and-sound-symbols\" title=\"LED and Sound Symbols\" description=\"Symbols for setting the LEDs and the sound request.\">\n";
  
  ledAndSoundSymbolsXml << "\t<enumerated-output-symbol name=\"leds.top-red\" description=\"a request how the both red top leds are set.\">\n";
  for (i=0;i<LEDRequest::numOfRedLEDPairs;i++)
  {
    getXmlString(s,LEDRequest::getRedLEDPairName((LEDRequest::RedLEDPair)i));
    ledAndSoundSymbolsXml << "\t\t<enum-element name=\"top." << s << "\"/>\n";
  }
  ledAndSoundSymbolsXml << "\t</enumerated-output-symbol>\n";
  
  ledAndSoundSymbolsXml << "\t<enumerated-output-symbol name=\"leds.bottom-red\" description=\"a request how the both red bottom leds are set.\">\n";
  for (i=0;i<LEDRequest::numOfRedLEDPairs;i++)
  {
    getXmlString(s,LEDRequest::getRedLEDPairName((LEDRequest::RedLEDPair)i));
    ledAndSoundSymbolsXml << "\t\t<enum-element name=\"bottom." << s << "\"/>\n";
  }
  ledAndSoundSymbolsXml << "\t</enumerated-output-symbol>\n";

  
  // face leds
  for(int faceLEDNumber = 1; faceLEDNumber <= 14; faceLEDNumber++)
  {
    ledAndSoundSymbolsXml << "\t<enumerated-output-symbol name=\"leds.face" << faceLEDNumber << "\" description=\"a request how the face leds are set.\">\n";
    for (i=0;i<LEDRequest::numOfBlinkModes;i++)
    {
      getXmlString(s,LEDRequest::getBlinkModeName((LEDRequest::BlinkMode)i));
      ledAndSoundSymbolsXml << "\t\t<enum-element name=\"face" << faceLEDNumber << "." << s << "\"/>\n";
    }
    ledAndSoundSymbolsXml << "\t</enumerated-output-symbol>\n";
  }

  // back front white led
  ledAndSoundSymbolsXml << "\t<enumerated-output-symbol name=\"leds.back-front-white\" description=\"a request how the back front white led is set.\">\n";
  for (i=0;i<LEDRequest::numOfBlinkModes;i++)
  {
    getXmlString(s,LEDRequest::getBlinkModeName((LEDRequest::BlinkMode)i));
    ledAndSoundSymbolsXml << "\t\t<enum-element name=\"back-front-white." << s << "\"/>\n";
  }
  ledAndSoundSymbolsXml << "\t</enumerated-output-symbol>\n";

  // back front blue led
  ledAndSoundSymbolsXml << "\t<enumerated-output-symbol name=\"leds.back-front-blue\" description=\"a request how the back front blue led is set.\">\n";
  for (i=0;i<LEDRequest::numOfBlinkModes;i++)
  {
    getXmlString(s,LEDRequest::getBlinkModeName((LEDRequest::BlinkMode)i));
    ledAndSoundSymbolsXml << "\t\t<enum-element name=\"back-front-blue." << s << "\"/>\n";
  }
  ledAndSoundSymbolsXml << "\t</enumerated-output-symbol>\n";

  // back middle white led
  ledAndSoundSymbolsXml << "\t<enumerated-output-symbol name=\"leds.back-middle-white\" description=\"a request how the back middle white led is set.\">\n";
  for (i=0;i<LEDRequest::numOfBlinkModes;i++)
  {
    getXmlString(s,LEDRequest::getBlinkModeName((LEDRequest::BlinkMode)i));
    ledAndSoundSymbolsXml << "\t\t<enum-element name=\"back-middle-white." << s << "\"/>\n";
  }
  ledAndSoundSymbolsXml << "\t</enumerated-output-symbol>\n";

  // back middle orange led
  ledAndSoundSymbolsXml << "\t<enumerated-output-symbol name=\"leds.back-middle-orange\" description=\"a request how the back middle orange led is set.\">\n";
  for (i=0;i<LEDRequest::numOfBlinkModes;i++)
  {
    getXmlString(s,LEDRequest::getBlinkModeName((LEDRequest::BlinkMode)i));
    ledAndSoundSymbolsXml << "\t\t<enum-element name=\"back-middle-orange." << s << "\"/>\n";
  }
  ledAndSoundSymbolsXml << "\t</enumerated-output-symbol>\n";

  // back rear white led
  ledAndSoundSymbolsXml << "\t<enumerated-output-symbol name=\"leds.back-rear-white\" description=\"a request how the back rear white led is set.\">\n";
  for (i=0;i<LEDRequest::numOfBlinkModes;i++)
  {
    getXmlString(s,LEDRequest::getBlinkModeName((LEDRequest::BlinkMode)i));
    ledAndSoundSymbolsXml << "\t\t<enum-element name=\"back-rear-white." << s << "\"/>\n";
  }
  ledAndSoundSymbolsXml << "\t</enumerated-output-symbol>\n";

  // back rear red led
  ledAndSoundSymbolsXml << "\t<enumerated-output-symbol name=\"leds.back-rear-red\" description=\"a request how the back rear red led is set.\">\n";
  for (i=0;i<LEDRequest::numOfBlinkModes;i++)
  {
    getXmlString(s,LEDRequest::getBlinkModeName((LEDRequest::BlinkMode)i));
    ledAndSoundSymbolsXml << "\t\t<enum-element name=\"back-rear-red." << s << "\"/>\n";
  }
  ledAndSoundSymbolsXml << "\t</enumerated-output-symbol>\n";

  // head white led
  ledAndSoundSymbolsXml << "\t<enumerated-output-symbol name=\"leds.head-white\" description=\"a request how the head white led is set.\">\n";
  for (i=0;i<LEDRequest::numOfBlinkModes;i++)
  {
    getXmlString(s,LEDRequest::getBlinkModeName((LEDRequest::BlinkMode)i));
    ledAndSoundSymbolsXml << "\t\t<enum-element name=\"head-white." << s << "\"/>\n";
  }
  ledAndSoundSymbolsXml << "\t</enumerated-output-symbol>\n";

  // head orange led
  ledAndSoundSymbolsXml << "\t<enumerated-output-symbol name=\"leds.head-orange\" description=\"a request how the head orange led is set.\">\n";
  for (i=0;i<LEDRequest::numOfBlinkModes;i++)
  {
    getXmlString(s,LEDRequest::getBlinkModeName((LEDRequest::BlinkMode)i));
    ledAndSoundSymbolsXml << "\t\t<enum-element name=\"head-orange." << s << "\"/>\n";
  }
  ledAndSoundSymbolsXml << "\t</enumerated-output-symbol>\n";


  ledAndSoundSymbolsXml << "\t<enumerated-output-symbol name=\"leds.tail-mode\" description=\"a request how the tail led is set.\">\n";
  for (i=0;i<LEDRequest::numOfTailModes;i++)
  {
    getXmlString(s,LEDRequest::getTailModeName((LEDRequest::TailMode)i));
    ledAndSoundSymbolsXml << "\t\t<enum-element name=\"tail." << s << "\"/>\n";
  }
  ledAndSoundSymbolsXml << "\t</enumerated-output-symbol>\n";
 
  ledAndSoundSymbolsXml << "\t<enumerated-output-symbol name=\"sound-request\" description=\"a request for a certain sound output\">\n";
  for (i=0;i<SoundRequest::numOfSoundIDs;i++)
  {
    getXmlString(s,SoundRequest::getSoundIDName((SoundRequest::SoundID)i));
    ledAndSoundSymbolsXml << "\t\t<enum-element name=\"sound-request." << s << "\"/>\n";
  }
  ledAndSoundSymbolsXml << "\t</enumerated-output-symbol>\n";
  
  ledAndSoundSymbolsXml << "</symbols>\n";
  
  ledAndSoundSymbolsXml.close();
}

void generateMatlabIDs()
{
  int i;
  char s[256];
  sprintf(s,"%s/Config/Matlab/messageIdFromName.m",File::getGTDir());
  printf("generating %s\n",s);
  std::ofstream messageIdFromName;
  messageIdFromName.open(s,std::ios::out);
  
  messageIdFromName << "function b=messageIdFromName(name)\n";
  messageIdFromName << "%MESSAGEIDFROMNAME(NAME)  returns the number\n";
  messageIdFromName << "%   of the MessageID with the provided NAME.\n";
  messageIdFromName << "%   Attention: this file has to be regenerated\n";
  messageIdFromName << "%   with URC /matlabIDs\n";
  messageIdFromName << "%   when MessageID.h changes!\n\n";
  messageIdFromName << "switch name\n";
  for (i=0;i<numOfMessageIDs;i++)
  {
    messageIdFromName << "  case '" << getMessageIDName((MessageID)i) << "', b=" << i << ";\n";
  }
  messageIdFromName << "  otherwise, b=-1;\n";
  messageIdFromName << "end\n";
  messageIdFromName.close();
  
  
  sprintf(s,"%s/Config/Matlab/nameFromMessageId.m",File::getGTDir());
  printf("generating %s\n",s);
  std::ofstream nameFromMessageId;
  nameFromMessageId.open(s,std::ios::out);
  nameFromMessageId << "function b=nameFromMessageId(id)\n";
  nameFromMessageId << "%NAMEFROMMESSAGEID(ID)  returns the name of\n";
  nameFromMessageId << "%   the MessageID with the provided number ID.\n";
  nameFromMessageId << "%   Attention: this file has to be regenerated\n";
  nameFromMessageId << "%   with URC /matlabIDs\n";
  nameFromMessageId << "%   when MessageID.h changes!\n\n";
  nameFromMessageId << "switch id\n";
  for (i=0;i<numOfMessageIDs;i++)
  {
    nameFromMessageId << "  case " << i << ", b='" << getMessageIDName((MessageID)i) << "';\n";
  }
  nameFromMessageId << "  otherwise, b='unknown';\n";
  nameFromMessageId << "end\n";
  nameFromMessageId.close();
}

/** generate MotionNetData.cpp
* @return True if successful
*/
bool generateMotionNet()
{
  char s[256];
  sprintf(s,"%s/Config/spec_act.dat",File::getGTDir());
  std::ofstream MotionNetData_cpp;
  MotionNetData_cpp.open(s,std::ios::out);
  MotionNetData_cpp << "//\n";
  MotionNetData_cpp << "// This file contains the special actions and was\n";
  MotionNetData_cpp << "// generated by the Universal Resource Compiler\n";
  MotionNetData_cpp << "//\n";
  MotionNetData_cpp << "// authors: Uwe Dffert, Martin Ltzsch, Max Risler\n";
  MotionNetData_cpp << "//\n";
  
  int i;
  
  MotionNetData_cpp << "\n// jump table:\n";
  for (i=0;i<SpecialActionRequest::numOfSpecialAction;i++)
  {
    MotionNetData_cpp << jumpTable[i] << " ";
  }
 

  MotionNetData_cpp << "\n\n// number of nodes:\n" << numOfLines << "\n\n";

  MotionNetData_cpp << "// nodes:\n";
  int labelnum=0;
  int filenum=0;
  for (i=0; i<numOfLines;i++)
  {
    if (file_startindex[filenum]==i)
    {
      filenum++;
    }
    if (label_number[labelnum]==i)
    {
      labelnum++;
    }
    if (strncmp(line_data[i],"pid",3)==0)
    {
      MotionNetData_cpp << "4 " << line_data[i]+4;
    }
    else if (strncmp(line_data[i],"transition",10)==0)
    {
      char request[512];
      char viamotion[512];
      char vialabel[512];
      sscanf(line_data[i]+11,"%s %s %s",request,viamotion,vialabel);
      //check if label exists (request was already checked)
      bool found=false;
      int j,k;
      for (j=0;j<numOfLabels;j++)
      {
        if ((strcmp(label_motion[j],viamotion)==0)&&(strcmp(label_name[j],vialabel)==0))
        {
          found=true;
          break;
        }
      }
      if (!found)
      {
        for (k=numOfFiles-1; k>=0;k--)
        {
          if (file_startindex[k]<=i)
          {
            printf("%s(%i) : error: jump label unknown\n",file_name[k],line_number[i]);
            return(false);
          }
        }
      }
      
      if (strcmp(request,"allMotions")!=0)
      {
        MotionNetData_cpp << "1 " << label_number[j] << " " << SpecialActionRequest::getSpecialActionIDFromName(request);
      }
      else
      {
        MotionNetData_cpp << "2 " << label_number[j];
      }
    }
    else
    {
      MotionNetData_cpp << "3 " << line_data[i];
    }
    
    
    if (line_motionID[i]>=0)
    {
      MotionNetData_cpp << " " << line_motionID[i];
    }
    else
    {
      MotionNetData_cpp << " -1";
    }
    
    
    if (i<numOfLines-1)
    {
      MotionNetData_cpp << "\n";
    }
  }
  
  MotionNetData_cpp.close();
  return(true);
}

/** parse all mof files except extern.mof to generate motion net
* @return True if successful
*/
bool parseMofs()
{
  struct _finddata_t ff;
  char ffname[256];
  sprintf(ffname,"%s/Src/Modules/SpecialActions/GT2003MotionNetSpecialActions/mof/*.mof",File::getGTDir());
  long fd=_findfirst(ffname,&ff);
  bool thereAreMore = (fd>0);
  while (thereAreMore)
  {
    if (strcmp(ff.name,"extern.mof")!=0)
    {
      char name[512];
      sprintf(name,"%s/Src/Modules/SpecialActions/GT2003MotionNetSpecialActions/mof/%s",File::getGTDir(),ff.name);
      FILE* f=fopen(name,"r");
      if (f<=0)
      {
        printf("error opening %s. Aborting.\n",name);
        return(false);
      }
      else
      {
        file_name[numOfFiles]=strdup(name);
        file_startindex[numOfFiles++]=numOfLines;
        bool thisMofHasLabels=false;
        strcpy(motion,ff.name);
        if (strcmp(motion+strlen(motion)-4,".mof")==0)
        {
          motion[strlen(motion)-4]=0;
        }
        actMotionID=-1;
        for (int j=0;j<SpecialActionRequest::numOfSpecialAction;j++)
        {
          if (strcmp(SpecialActionRequest::getSpecialActionIDName((SpecialActionRequest::SpecialActionID)j),motion)==0)
          {
            actMotionID=j;
            break;
          }
        }
        
        char s[128000];
        int siz=fread(s,1,128000,f);
        fclose(f);
        if (siz>0)
        {
          s[siz]=0;
          char* t=&s[0];
          int line=1;
          while (*t)
          {
            char* u=strchr(t,'\n');
            if (u>=t)
            {
              *u=0;
              
              char sval[23][256];
              int c=sscanf(t,"%s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s",sval[0],sval[1],sval[2],sval[3],sval[4],sval[5],sval[6],sval[7],sval[8],sval[9],sval[10],sval[11],sval[12],sval[13],sval[14],sval[15],sval[16],sval[17],sval[18],sval[19],sval[20],sval[21],sval[22]);
              if ((c==-1)||(sval[0][0]=='"')||(strncmp(sval[0],"//",2)==0))
              {
                //skip comments and empty lines by doing nothing
              }
              else if (strcmp(sval[0],"motion_id")==0)
              {
                //if there is a motion_id we use it, otherwise filename
                if ((strcmp(sval[1],"=")==0)&&(c==3))
                {
                  strcpy(motion,sval[2]);
                  for (j=0;j<SpecialActionRequest::numOfSpecialAction;j++)
                  {
                    if (strcmp(SpecialActionRequest::getSpecialActionIDName((SpecialActionRequest::SpecialActionID)j),motion)==0)
                    {
                      actMotionID=j;
                      break;
                    }
                  }
                }
                else
                {
                  printf("%s(%i) : error: motion_id format\n",name,line);
                  return(false);
                }
              }
              else if (strcmp(sval[0],"transition")==0)
              {
                bool found=false;
                if (!thisMofHasLabels)
                {
                  printf("%s(%i) : error: this line is unreachable, because there was no label before\n",name,line);
                  return(false);
                }
                if (strcmp(sval[1],"allMotions")==0)
                {
                  found=true; 
                }
                else
                {
                  for (int j=0;j<SpecialActionRequest::numOfSpecialAction;j++)
                  {
                    if (strcmp(SpecialActionRequest::getSpecialActionIDName((SpecialActionRequest::SpecialActionID)j),sval[1])==0)
                    {
                      found=true;
                      break;
                    }
                  }
                }
                
                if (!found)
                {
                  printf("%s(%i) : error: request for transition unknown\n",name,line);
                  return(false);
                }
                if (c==4)
                {
                  line_motionID[numOfLines]=actMotionID;
                  line_data[numOfLines]=strdup(t);
                  line_number[numOfLines++]=line;
                }
                else
                {
                  printf("%s(%i) : error: transition format\n",name,line);
                  return(false);
                }
              }
              else if (strcmp(sval[0],"pid")==0)
              {
                char temp[512];
                if (JointData::getIDFromName(sval[1]) == JointData::numOfJoint)
                {
                  printf("%s(%i) : error: unknown joint \"%s\"\n",name,line,sval[1]);
                  return(false);
                }
                
                bool ok=(c==5);
                int p;
                if (strcmp(sval[2],"~")==0) strcpy(sval[2],"10000"); else { ok &= ((sscanf(sval[2],"%i",&p)==1)&&(p>=0)&&(p<=100)); }
                if (strcmp(sval[3],"~")==0) strcpy(sval[3],"10000"); else { ok &= ((sscanf(sval[3],"%i",&p)==1)&&(p>=0)&&(p<=100)); }
                if (strcmp(sval[4],"~")==0) strcpy(sval[4],"10000"); else { ok &= ((sscanf(sval[4],"%i",&p)==1)&&(p>=0)&&(p<=100)); }
                
                if (ok)
                {
                  sprintf(temp,"pid %i %s %s %s ",JointData::getIDFromName(sval[1]),sval[2],sval[3],sval[4]);
                  line_data[numOfLines]=strdup(temp);
                  line_number[numOfLines++]=line;
                }
                else
                {
                  printf("%s(%i) : error: pid format\n",name,line);
                  return(false);
                }
              }
              else if (strcmp(sval[0],"label")==0)
              {
                if (c==2)
                {
                  label_motion[numOfLabels]=strdup(motion);
                  label_name[numOfLabels]=strdup(sval[1]);
                  label_number[numOfLabels++]=numOfLines;
                  thisMofHasLabels=true;
                }
                else
                {
                  printf("%s(%i) : error: label format\n",name,line);
                  return(false);
                }
              }
              else if (c==20)
              {
                char temp[1024];
                temp[0]=0;
                int val=0;
                if (!thisMofHasLabels)
                {
                  printf("%s(%i) : error: this line is unreachable, because there was no label before\n",name,line);
                  return(false);
                }
                
                for (int j=0;j<18;j++)
                {
                  if (strcmp(sval[j],"~")==0) sprintf(sval[j],"10000");
                  if ((strcmp(sval[j],"10000")==0)||((sscanf(sval[j],"%i",&val)==1)&&(val>-3000)&&(val<3000)))
                  {
                    strcat(temp," ");
                    strcat(temp,sval[j]);
                  }
                  else
                  {
                    printf("%s(%i) : error: joint data format\n",name,line);
                    return(false);
                  }
                }
                strcat(temp," 0 0");
                if ((sscanf(sval[18],"%i",&val)==1)&&(val>=0)&&(val<=1))
                {
                  strcat(temp," ");
                  strcat(temp,sval[18]);
                }
                else
                {
                  printf("%s(%i) : error: interpolation data format\n",name,line);
                  return(false);
                }
                if ((sscanf(sval[19],"%i",&val)==1)&&(val>0))
                {
                  strcat(temp," ");
                  strcat(temp,sval[19]);
                }
                else
                {
                  printf("%s(%i) : error: time data format\n",name,line);
                  return(false);
                }
                line_motionID[numOfLines]=actMotionID;
                line_data[numOfLines]=strdup(temp);
                line_number[numOfLines++]=line;
              }
              else if (c==22)
              {
                char temp[1024];
                temp[0]=0;
                int val=0;
                if (!thisMofHasLabels)
                {
                  printf("%s(%i) : error: this line is unreachable, because there was no label before\n",name,line);
                  return(false);
                }
                for (int j=0;j<20;j++)
                {
                  if (strcmp(sval[j],"~")==0) sprintf(sval[j],"10000");
                  if ((strcmp(sval[j],"10000")==0)||((sscanf(sval[j],"%i",val)==1)&&(val>-3000)&&(val<3000)))
                  {
                    sprintf((char*)&temp[strlen(temp)]," %s",sval[j]);
                  }
                  else
                  {
                    printf("%s(%i) : error: joint data format\n",name,line);
                    return(false);
                  }
                }
                if ((sscanf(sval[20],"%i",&val)==1)&&(val>=0)&&(val<=1))
                {
                  strcat(temp," ");
                  strcat(temp,sval[20]);
                }
                else
                {
                  printf("%s(%i) : error: interpolation data format\n",name,line);
                  return(false);
                }
                if ((sscanf(sval[21],"%i",&val)==1)&&(val>0))
                {
                  strcat(temp," ");
                  strcat(temp,sval[21]);
                }
                else
                {
                  printf("%s(%i) : error: time data format\n",name,line);
                  return(false);
                }
                line_motionID[numOfLines]=actMotionID;
                line_data[numOfLines]=strdup(temp);
                line_number[numOfLines++]=line;
              }
              line++;
              t=u+1;
            }
            else
            {
              t += strlen(t);
            }
          }
        }
        else
        {
          printf("error reading from %s. Aborting.\n",name);
          return(false);
        }
      }
    }
    thereAreMore=(_findnext(fd,&ff)==0);
  }
  return(true);
}

/** parse extern.mof to generate jump table from extern into motion net
* @return True if successful
*/
bool parseExternMof()
{
  int i,j;
  for (i=0;i<SpecialActionRequest::numOfSpecialAction;i++)
  {
    jumpTable[i]=-1;
  }
  
  char name[512];
  sprintf(name,"%s/Src/Modules/SpecialActions/GT2003MotionNetSpecialActions/mof/extern.mof",File::getGTDir());
  FILE* f=fopen(name,"r");
  if (f<=0)
  {
    printf("error opening %s. Aborting.\n",name);
    return(false);
  }
  else
  {
    char s[128000];
    int siz=fread(s,1,128000,f);
    fclose(f);
    if (siz>0)
    {
      s[siz]=0;
      char* t=&s[0];
      int line=1;
      while (*t)
      {
        char* u=strchr(t,'\n');
        if (u>=t)
        {
          *u=0;
          
          char sval[5][256];
          int c=sscanf(t,"%s %s %s %s %s",sval[0],sval[1],sval[2],sval[3],sval[4]);
          
          if ((c==-1)||(sval[0][0]=='"')||(strncmp(sval[0],"//",2)==0))
          {
            //skip comments and empty lines by doing nothing
          }
          else if ((strcmp(sval[0],"motion_id")==0)||(strcmp(sval[0],"label")==0))
          {
            //skip labels and id
          }
          else if ((strcmp(sval[0],"transition")==0)&&(c==4))
          {
            bool found=false;
            if (strcmp(sval[1],"extern")==0) 
            {
              found=true;
              continue;
            }
            else
            {
              for (i=0;i<SpecialActionRequest::numOfSpecialAction;i++)
              {
                if (strcmp(SpecialActionRequest::getSpecialActionIDName((SpecialActionRequest::SpecialActionID)i),sval[1])==0)
                {
                  found=true;
                  break;
                }
              }
              
              if (!found)
              {
                printf("%s(%i) : error: special action '%s' unknown (not in MotionRequest.h)\n",name,line,sval[1]);
                return(false);
              }
              
              for (j=0;j<numOfLabels;j++)
              {
                if ((strcmp(label_motion[j],sval[2])==0)&&(strcmp(label_name[j],sval[3])==0))
                {
                  jumpTable[i]=label_number[j];
                  found=true;
                  break;
                }
              }
              if (!found)
              {
                printf("%s(%i) : error: jump label '%s:%s' unknown (not in *.mof)\n",name,line,sval[2],sval[3]);
                return(false);
              }
            }
          }
          else
          {
            printf("%s(%i) : error: unexpected data\n",name,line);
            return(false);
          }
          line++;
          t=u+1;
        }
        else
        {
          t += strlen(t);
        }
      }
    }
    else
    {
      printf("error reading from %s. Aborting.\n",name);
      return(false);
    }
  }
  for (i=0;i<SpecialActionRequest::numOfSpecialAction;i++)
  {
    if (jumpTable[i]==-1)
    {
      printf("%s/Src/Modules/SpecialActions/GT2003MotionNetSpecialActions/mof/extern.mof(1): error: no motion net entry defined for special action %s\n",File::getGTDir(),SpecialActionRequest::getSpecialActionIDName((SpecialActionRequest::SpecialActionID)i));
      return(false);
    }
  }
  return(true);
}

void usage()
{
  printf("\nusage:\n URC [/specialActionXml | /headTailXml | /ledSoundXml | /matlabIDs]\n\n");
}

/** main routine of URC
* @return 0 if successful
*/
int main(int argc, char* argv[])
{
  if (argc==2)
  {
    if (strcmp(argv[1],"/specialActionXml")==0)
    {
      generateSpecialActionXml();
    }
    else if (strcmp(argv[1],"/kickSelectionXml")==0)
    {
      generateKickSelectionXml();
    }
    else if (strcmp(argv[1],"/headTailXml")==0)
    {
      generateHeadTailXml();
    }
    else if (strcmp(argv[1],"/ledSoundXml")==0)
    {
      generateLEDSoundXml();
    }
    else if (strcmp(argv[1],"/matlabIDs")==0)
    {
      generateMatlabIDs();
    }
    else
    {
      usage();
    }
  }
  else if (argc>2)
  {
    usage();
  }
  else
  {
    numOfLabels=0;
    label_motion[numOfLabels]=strdup("extern");
    label_name[numOfLabels]=strdup("start");
    label_number[numOfLabels++]=numOfLines;
    
    numOfLines=0;
    line_data[numOfLines]=strdup("transition allMotions extern start");
    line_number[numOfLines++]=1;
    
    if (!parseMofs()) return(1);
    if (!parseExternMof()) return(1);
    
    //check whether the last line of every mof is an unconditional jump
    int i;
    for (i=1;i<=numOfFiles;i++)
    {
      int ind=i<numOfFiles?file_startindex[i]-1:numOfLines-1;
      if (strncmp(line_data[ind],"transition allMotions",21)!=0)
      {
        printf("%s(%i) : error: this mof does not end with a transition for allMotions\n",file_name[i-1],line_number[ind]);
        return(1);
      }
    }  
    
    //check whether there are mofs without MotionRequest::SpecialActionID
    for (i=1;i<numOfLabels;i++)
    {
      if (strcmp(label_motion[i],label_motion[i-1])==0) continue;
      bool found=false;
      for (int j=0;j<SpecialActionRequest::numOfSpecialAction;j++)
      {
        if (strcmp(label_motion[i],SpecialActionRequest::getSpecialActionIDName((SpecialActionRequest::SpecialActionID)j))==0)
        {
          found=true;
          break;
        }
      }
      if (!found)
      {
        for (j=numOfFiles-1;j>=0;j--)
        {
          if (file_startindex[j]<=label_number[i])
          {
            printf("%s(%i) : warning: there is no special action id for this mof\n",file_name[j],1);
            break;
          }
        }
      }
    }  
    /** todo warning for motions without labels */
    
    if (!generateMotionNet()) return(1);
    
    for (i=0;i<numOfLines;i++) delete(line_data[i]);
    for (i=0;i<numOfFiles;i++) delete(file_name[i]);
    for (i=0;i<numOfLabels;i++)
    {
      delete(label_name[i]);
      delete(label_motion[i]);
    }
    printf("created Config/spec_act.dat successfully\n");
  }
  return 0;
}

/*
* Change log :
* 
* $Log: UniversalResourceCompiler.cpp,v $
* Revision 1.4  2004/06/15 12:41:22  loetzsch
* changed XABSL version from 2.1 to 2.2 on the occasion of a new XABSL release
*
* Revision 1.3  2004/06/02 17:18:24  spranger
* MotionRequest cleanup
*
* Revision 1.2  2004/05/22 21:31:04  loetzsch
* adopted kick selection table to GT2004
*
* Revision 1.1.1.1  2004/05/22 17:37:59  cvsadm
* created new repository GT2004_WM
*
* Revision 1.11  2004/03/25 21:20:19  juengel
* Added head LEDs.
*
* Revision 1.10  2004/03/25 19:56:28  loetzsch
* again changed the led request
*
* Revision 1.9  2004/03/25 17:36:26  loetzsch
* changed ledrequest
*
* Revision 1.8  2004/03/21 19:08:14  juengel
* Added back*White LED support.
*
* Revision 1.7  2004/03/20 19:24:07  juengel
* Added face LEDs to LEDRequest
*
* Revision 1.6  2004/03/16 14:00:24  juengel
* Integrated Improvments from "Gnne"
* -ATH2004ERS7Behavior
* -ATHHeadControl
* -KickSelectionTable
* -KickEditor
*
* Revision 1.3  2004/03/09 18:42:05  loetzsch
* kick selection table changes
*
* Revision 1.2  2004/03/09 14:45:27  loetzsch
* automated generation of kick-selection-symbols.xml
*
* Revision 1.5  2004/03/16 12:39:19  dueffert
* error output improved
*
* Revision 1.4  2004/03/01 12:37:22  wachter
* Added LED-Request for button-interface
*
* Revision 1.3  2004/02/26 11:10:22  jhoffman
* - enforce recompile
*
* Revision 1.2  2003/12/31 16:58:44  roefer
* Joints and LEDs for ERS-7
*
* Revision 1.1  2003/12/16 18:51:08  loetzsch
* renamed MotionNetCodeGenerator to UniversalSourceCompiler,
*
* the motion net is not compiled into C++ souce code, but into a file that is
* directly parsed on the robot.
*
* Revision 1.3  2003/11/30 01:53:18  loetzsch
* prepared RobotControl port to Visual C++ .Net
*
* Revision 1.2  2003/10/22 22:19:18  loetzsch
* prepared the cloning of the GT2003BehaviorControl
*
* Revision 1.1  2003/10/07 10:06:58  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.7  2003/09/26 15:29:51  juengel
* Renamed DataTypes to representations.
*
* Revision 1.6  2003/09/26 11:41:17  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.5  2003/07/04 10:40:03  dueffert
* bug report fixed
*
* Revision 1.4  2003/07/04 10:38:15  dueffert
* success report added
*
* Revision 1.3  2003/07/03 18:39:30  risler
* do-kick option added
* all kicks use do-kick
*
* Revision 1.2  2003/07/03 16:10:21  dueffert
* error output improved
*
* Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.13  2003/06/25 01:21:06  loetzsch
* GT2003BehaviorControl uses Xabsl 2.1 now
*
* Revision 1.12  2003/05/20 12:27:05  osterhues
* USE_GETMESSAGEIDNAME has to be defined before
* "Tools/MessageQueue/MessageIDs.h" is included
*
* Revision 1.11  2003/05/20 10:35:06  dueffert
* generating matlab symbols added
*
* Revision 1.10  2003/05/08 14:27:33  risler
* added tailMode to LEDRequest
*
* Revision 1.9  2003/05/07 13:55:42  dueffert
* now using getGTDir() instead of absolute path
*
* Revision 1.8  2003/05/06 14:47:08  loetzsch
* added registration of generated GT2003BehaviorControl symbols
*
* Revision 1.7  2003/05/06 13:39:52  loetzsch
* moved xml comments upward
*
* Revision 1.6  2003/05/06 08:00:35  dueffert
* generating XML symbols in C
*
* Revision 1.5  2003/04/14 16:10:40  loetzsch
* ATH after GermanOpen CVS merge
* bugs fixed
*
* Revision 1.3  2003/04/11 17:45:36  dueffert
* unused label bug fixed
*
* Revision 1.2  2003/04/10 14:29:41  dueffert
* support for mofs without labels added
*
* Revision 1.1.1.1  2003/04/09 14:22:36  loetzsch
* started Aibo Team Humboldt's GermanOpen CVS
*
* Revision 1.4  2003/03/25 15:41:02  timrie
* Doxygen-comments corrected (in generated files)
*
* Revision 1.3  2003/03/25 10:55:14  timrie
* Initialisierung korrigiert, = 700 Warnungen weniger
*
* Revision 1.2  2003/03/19 09:21:25  dueffert
* executedMotionRequest support added
*
* Revision 1.1  2003/03/10 14:15:28  dueffert
* optimized MotionNet
*
*
*/
