/**
* @file Xabsl2Option.h
* 
* Definition of class Xabsl2Option and Helper classes
*
* @author Martin Ltzsch
*/

#ifndef __Xabsl2Option_h_
#define __Xabsl2Option_h_

#include "Xabsl2BasicBehavior.h"
#include "Xabsl2BooleanExpression.h"


// class prototype needed for the declaration of Xabsl2Statement
class Xabsl2State;

/** 
* @class Xabsl2Statement 
* 
* An element of a decision that that determines a transition to a state.
* It can be either a transition to a state or a if/else-if/else block containing other statements.
* 
* @author Martin Ltzsch
*/
class Xabsl2Statement
{
public:
  /** Executes the statement and determines the next active state (can be the same). */
  virtual Xabsl2State* getNextState() = 0;
  
  /** 
  * Creates a statement depending on the input.
  * @param input An input source for the intermediate code. It must be opened and read until 
  *              A position where a statement starts.
  * @param subsequentOption The subsequent option of the state. 0 if the subsequent behavior is a basic behavior
  * @param errorHandler A reference to a Xabsl2ErrorHandler instance
  * @param states All states of the option
  * @param parameters The parameters of the option
  * @param symbols All available symbols
  * @param timeOfOptionExecution The time how long the option is already active
  * @param timeOfStateExecution The time how long the state is already active
  */
  static Xabsl2Statement* createStatement(Xabsl2InputSource& input,    
    Xabsl2Option* subsequentOption,
    Xabsl2ErrorHandler& errorHandler,
    Xabsl2Array<Xabsl2State*>& states,
    Xabsl2Array<double>& parameters,
    Xabsl2Symbols& symbols,    
    unsigned long& timeOfOptionExecution,
    unsigned long& timeOfStateExecution);
  
  /** Destructor */
  virtual ~Xabsl2Statement() = 0;
};

/**
* @class Xabsl2IfElseBlock
*
* An element of a decision tree that that contains of an if - (else-if) - else block
* 
* @author Martin Ltzsch.
l*/
class Xabsl2IfElseBlock : public Xabsl2Statement
{
public:
/** 
* Constructor. Creates the if / else statement
* @param input An input source for the intermediate code. It must be opened and read until 
*              A position where a transition starts.
* @param subsequentOption The subsequent option of the state. 0 if the subsequent behavior is a basic behavior
* @param errorHandler A reference to a Xabsl2ErrorHandler instance
* @param states All states of the option
* @param parameters The parameters of the option
* @param symbols All available symbols
* @param timeOfOptionExecution The time how long the option is already active
* @param timeOfStateExecution The time how long the state is already active
  */
  Xabsl2IfElseBlock(Xabsl2InputSource& input, 
    Xabsl2Option* subsequentOption,
    Xabsl2ErrorHandler& errorHandler,
    Xabsl2Array<Xabsl2State*>& states,
    Xabsl2Array<double>& parameters,
    Xabsl2Symbols& symbols,
    unsigned long& timeOfOptionExecution,
    unsigned long& timeOfStateExecution
    );
  
  /** Destructor. Deletes all statements and conditions */
  ~Xabsl2IfElseBlock();
  
  /** Executes the statement and determines the next active state (can be the same). */
  virtual Xabsl2State* getNextState();
  
private:
  /** The boolean expression that is evaluated for the if case */
  Xabsl2BooleanExpression* ifCondition;
  
  /** The statement that is executed if the if condition is true */
  Xabsl2Statement* ifStatement;
  
  /** The boolean expressions for the else-if cases */
  Xabsl2Array<Xabsl2BooleanExpression*> elseIfConditions;
  
  /** The statements that are executed if the else-if condition are true */
  Xabsl2Array<Xabsl2Statement*> elseIfStatements;
  
  /** The statement that is executed if all if and else-if conditions fail */
  Xabsl2Statement* elseStatement;
};


/**
* Represents a transition to a state inside a decision tree
*/
class Xabsl2TransitionToState : public Xabsl2Statement
{
public:
/** 
* Constructor. Creates the transition object
* @param input An input source for the intermediate code. It must be opened and read until 
*              A position where a transition starts.
* @param errorHandler A reference to a Xabsl2ErrorHandler instance
* @param states All states of the option
  */
  Xabsl2TransitionToState(Xabsl2InputSource& input, 
    Xabsl2ErrorHandler& errorHandler,
    Xabsl2Array<Xabsl2State*>& states);
  
  /** Executes the statement and determines the next active state (can be the same). */
  virtual Xabsl2State* getNextState();
  
private:
  /** The state where that transition points to */
  Xabsl2State* nextState;
};

// class prototype needed for declaration of Xabsl2State
class Xabsl2Option;

/**
* @class Xabsl2State
*
* Represents a single state written in Xabsl2
*
* @author Martin Ltzsch
*/
class Xabsl2State : public Xabsl2NamedItem
{
public:
/** 
* Constructor. Does not create the state.
* @param name The name of the state. For debugging purposes.
* @param errorHandler A reference to a Xabsl2ErrorHandler instance
* @param pTimeFunction a pointer to a function that returns the system time in ms.
  */
  Xabsl2State(const char* name, 
    Xabsl2ErrorHandler& errorHandler,
    unsigned long (*pTimeFunction)());
  
  /** Destructor */
  ~Xabsl2State();
  
  /** 
  * Creates the state and it's subelements from the intermediate code.
  * @param input An input source for the intermediate code. It must be opened and read until 
  *              A position where a state starts.
  * @param options All available options
  * @param basicBehaviors All available basicBehaviors
  * @param states All states of the option
  * @param parameters The parameters of the option
  * @param symbols All available symbols
  * @param timeOfOptionExecution The time how long the option is already active
  */
  void create(Xabsl2InputSource& input,    
    Xabsl2Array<Xabsl2Option*>& options,
    Xabsl2Array<Xabsl2BasicBehavior&>& basicBehaviors,
    Xabsl2Array<Xabsl2State*>& states,
    Xabsl2Array<double>& parameters,
    Xabsl2Symbols& symbols,
    unsigned long& timeOfOptionExecution);
  
  /** 
  * Executes the decision tree and determines the next active state (can be the same). 
  */
  Xabsl2State* getNextState();
  
  /** Sets the output symbols of the state */
  void setOutputSymbols();

  /** 
  * The option that is executed after that option if the state is active. 
  * If 0, a basic behavior is executed after that option. 
  */
  Xabsl2Option* subsequentOption;  
  
  /** 
  * The basic behavior that is executed after that option if the state is active. 
  * If 0, an option is executed after that option. 
  */
  Xabsl2BasicBehavior* subsequentBasicBehavior;  
  
  /** the time how long the state is already active */
  unsigned long timeOfStateExecution;
  
  /** Sets the time when the state was activated to 0 */
  void reset();

  /** Pointers to the parameters of a subsequent basic behavior or a subsequent option */
  Xabsl2Array<double*> parametersOfSubsequentBehavior;

  /** Decimal expressions for the parameters of a subsequent basic behavior or a subsequent option */
  Xabsl2Array<Xabsl2DecimalExpression*> parameterValues;

  /** Returns wheter the state is a target state */
  bool isTargetState() const;

private:

  /** The time, when the state was activated */
  unsigned long timeWhenStateWasActivated;

  /** If true, the state is a target state */
  bool targetState;

  /** Used for error handling */
  Xabsl2ErrorHandler& errorHandler;
  
  /** The root element of the decision tree */
  Xabsl2Statement* decisionTree;

  /** The output symbols that are set if the state is active */
  Xabsl2Array<Xabsl2EnumeratedOutputSymbol*> outputSymbols;

  /** The for the output symbols that are set if the state is active */
  Xabsl2Array<int> outputSymbolValues;

  /** A pointer to a function that returns the system time in ms. */
  unsigned long (*pTimeFunction)();
};

/** 
* @class Xabsl2Option
* 
* Represents a single option written in Xabsl2
* 
* @author Martin Ltzsch
*/
class Xabsl2Option : public Xabsl2NamedItem
{
public:
/** 
* Constructor. Does not create the option.
* @param name The name of the option. For debugging purposes.
* @param input An input source for the intermediate code. It must be opened and read until 
*              A position where an option starts.
* @param errorHandler A reference to a Xabsl2ErrorHandler instance
* @param pTimeFunction a pointer to a function that returns the system time in ms.
  */
  Xabsl2Option(const char* name, Xabsl2InputSource& input,
    Xabsl2ErrorHandler& errorHandler,
    unsigned long (*pTimeFunction)());
  
  /** Destructor. Deletes the states */
  ~Xabsl2Option();
  
  /** 
  * Creates the option and it's states from the intermediate code.
  * @param input An input source for the intermediate code. It must be opened and read until 
  *              A position where an option starts.
  * @param options All other options
  * @param basicBehaviors All basic behaviors
  * @param symbols All available symbols
  */
  void create(Xabsl2InputSource& input,    
    Xabsl2Array<Xabsl2Option*>& options,
    Xabsl2Array<Xabsl2BasicBehavior&>& basicBehaviors,
    Xabsl2Symbols& symbols);
  
    /**
    * Executes the option. The state machine is carried out to determine the subsequent 
    * option or the subsequent basic behavior. Output symbols are set.
  */
  void execute();
  
  /** The parameters of the option */
  Xabsl2Array<double> parameters;
  
  /** The states of the option */
  Xabsl2Array<Xabsl2State*> states;
  
  /** A pointer to the active state. If 0, then no state is active */
  Xabsl2State* activeState;
  
  /** the option is activated in the current path through the option graph */
  bool optionIsActive;
  
  /** the option was activated in the last path trough the option graph */
  bool optionWasActive;

  /** the time how long the option is already active */
  unsigned long timeOfOptionExecution;

  /** Returns wheter the option reached a target state */
  bool getOptionReachedATargetState() const;

private:
  /** The time, when the option was activated */
  unsigned long timeWhenOptionWasActivated;

  /** A pointer to the initial state */
  Xabsl2State* initialState;
  
  /** Used for error handling */
  Xabsl2ErrorHandler& errorHandler;

  /** A pointer to a function that returns the system time in ms. */
  unsigned long (*pTimeFunction)();
};


#endif //__Xabsl2Option_h_

/*
* Change Log:
* 
* $Log: Xabsl2Option.h,v $
* Revision 1.1.1.1  2004/05/22 17:37:57  cvsadm
* created new repository GT2004_WM
*
* Revision 1.2  2003/10/08 11:50:09  loetzsch
* made the Xabsl2Engine really platform independent
* (removed inclusion of Platform/SystemCall.h)
* A time function is given to the engine by parameter.
*
* Revision 1.1  2003/10/07 10:13:25  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.4  2003/09/30 10:51:11  dueffert
* typos fixed
*
* Revision 1.3  2003/09/20 16:34:16  loetzsch
* renamed "following-option-..." to "subsequent-option-.." and
* "following-basic-behavior-.." to "subsequent-basic-behavior-.."
* for consistency with publications
*
* Revision 1.2  2003/09/16 13:27:21  loetzsch
* changed all occurrences of "option tree" to "option graph"
*
* Revision 1.1.1.1  2003/07/02 09:40:29  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.11  2003/05/05 17:47:55  loetzsch
* added in Xabsl2:
* - marking of states as target-states with attribute is-target-state="true"
* - boolean expression "subsequent-option-reached-target-state"
* - common-decision-tree
*
* Revision 1.10  2003/02/22 18:09:40  loetzsch
* changed comments
*
* Revision 1.9  2003/01/15 01:19:45  timrie
* corrected doxygen comments
*
* Revision 1.8  2003/01/14 16:28:32  loetzsch
* creation of references to output symbols added
* setting of output symbols added
*
* Revision 1.7  2003/01/13 22:39:38  loetzsch
* implemented the execution of the engine.
*
* Revision 1.6  2003/01/13 13:18:18  loetzsch
* Creation of boolean and decimal expressions finished.
*
* Revision 1.5  2003/01/12 14:54:04  loetzsch
* continued creation of option tree: Xabsl2Statement and derivates added
*
* Revision 1.4  2003/01/11 14:41:42  loetzsch
* continued creation of the option tree
*
* Revision 1.3  2003/01/09 17:28:33  loetzsch
* introduced Xabsl2Agent, continued Xabsl2Option
*
* Revision 1.2  2003/01/08 18:47:17  loetzsch
* first version of state implementation
*
* Revision 1.1  2003/01/08 15:22:33  loetzsch
* - started implementation of the option tree
* - started the reading of the intermediate code
*
*/
