/** 
* @file Xabsl2Option.cpp
*
* Implementation of class Xabsl2Option and helper classes
*
* @author Martin Ltzsch
*/

#include "Xabsl2Option.h"

Xabsl2Statement* Xabsl2Statement::createStatement(Xabsl2InputSource& input,    
                                                  Xabsl2Option* subsequentOption,
                                                  Xabsl2ErrorHandler& errorHandler,
                                                  Xabsl2Array<Xabsl2State*>& states,
                                                  Xabsl2Array<double>& parameters,
                                                  Xabsl2Symbols& symbols,    
                                                  unsigned long& timeOfOptionExecution,
                                                  unsigned long& timeOfStateExecution)
{
  char c[100]; 
  
  input.readString(c,1);
  
  switch (*c)
  {
  case 't':
    return new Xabsl2TransitionToState(input, errorHandler, states);
  case 'i':
    return new Xabsl2IfElseBlock(input, subsequentOption,errorHandler, states, 
      parameters, symbols, timeOfOptionExecution, timeOfStateExecution);
  default:
    errorHandler.error("Xabsl2Statement::create(): unknown type \"%c\"",*c);
    return 0;
  }
}

Xabsl2Statement::~Xabsl2Statement()
{
}

Xabsl2IfElseBlock::Xabsl2IfElseBlock(Xabsl2InputSource& input,    
                                                         Xabsl2Option* subsequentOption,
                                     Xabsl2ErrorHandler& errorHandler,
                                     Xabsl2Array<Xabsl2State*>& states,
                                     Xabsl2Array<double>& parameters,
                                     Xabsl2Symbols& symbols,    
                                     unsigned long& timeOfOptionExecution,
                                     unsigned long& timeOfStateExecution) :
ifCondition(0), ifStatement(0), elseStatement(0)
{
  
  // if case
  XABSL2_DEBUG_INIT(errorHandler.message("creating if statement"));
  
  ifCondition = Xabsl2BooleanExpression::create(input, subsequentOption,errorHandler, 
    parameters, symbols, timeOfOptionExecution, timeOfStateExecution);
  if (errorHandler.errorsOccurred)
  {
    errorHandler.error("Xabsl2IfElseBlock::Xabsl2IfElseBlock(): could not create if condition");
    return;
  }
  
  ifStatement = Xabsl2Statement::createStatement(input,subsequentOption,errorHandler,
    states,parameters, symbols, timeOfOptionExecution, timeOfStateExecution);
  if (errorHandler.errorsOccurred)
  {
    errorHandler.error("Xabsl2IfElseBlock::Xabsl2IfElseBlock(): could not create if statement");
    return;
  }
  
  // else if cases
  int numberOfElseIfStatements = (int)input.readValue();
  
  for (int i=0;i<numberOfElseIfStatements; i++)
  {
    XABSL2_DEBUG_INIT(errorHandler.message("creating else-if statement"));
    
    elseIfConditions.append("else-if",Xabsl2BooleanExpression::create(input, subsequentOption,
      errorHandler, parameters, symbols,timeOfOptionExecution, timeOfStateExecution));
    if (errorHandler.errorsOccurred)
    {
      errorHandler.error("Xabsl2IfElseBlock::Xabsl2IfElseBlock(): could not create else-if condition");
      return;
    }
    
    elseIfStatements.append("else-if",Xabsl2Statement::createStatement(input,subsequentOption,
      errorHandler,states, parameters, symbols, timeOfOptionExecution, timeOfStateExecution));
    if (errorHandler.errorsOccurred)
    {
      errorHandler.error("Xabsl2IfElseBlock::Xabsl2IfElseBlock(): could not create else-if statement");
      return;
    }
  }
  
  // else case
  XABSL2_DEBUG_INIT(errorHandler.message("creating else statement"));
  
  elseStatement = Xabsl2Statement::createStatement(input, subsequentOption,
    errorHandler, states, parameters, symbols, timeOfOptionExecution, timeOfStateExecution);
  if (errorHandler.errorsOccurred)
    errorHandler.error("Xabsl2IfElseBlock::Xabsl2IfElseBlock(): could not create else statement");
  
}

Xabsl2IfElseBlock::~Xabsl2IfElseBlock()
{
  if (ifCondition != 0) delete ifCondition;
  if (ifStatement != 0) delete ifStatement;
  
  int i;
  for (i=0; i<elseIfStatements.getSize(); i++)
  {
    if (elseIfConditions[i] != 0)
      delete elseIfConditions[i];
    if (elseIfStatements[i] != 0) 
      delete elseIfStatements[i];
  }
  
  if (elseStatement != 0) delete elseStatement;
}

Xabsl2State* Xabsl2IfElseBlock::getNextState()
{
  if (ifCondition->getValue()) return ifStatement->getNextState();
  
  for (int i=0; i<elseIfConditions.getSize(); i++)
  {
    if (elseIfConditions[i]->getValue()) return elseIfStatements[i]->getNextState();
  }
  
  return elseStatement->getNextState();
}

Xabsl2TransitionToState::Xabsl2TransitionToState(Xabsl2InputSource& input,    
                                                 Xabsl2ErrorHandler& errorHandler,
                                                 Xabsl2Array<Xabsl2State*>& states)
{
  char buf[100];
  input.readString(buf,99);
  
  nextState = states.getElement(buf);
  
  XABSL2_DEBUG_INIT(errorHandler.message("creating a transition to state \"%s\"",nextState->n));
}

Xabsl2State* Xabsl2TransitionToState::getNextState()
{
  return nextState;
}

Xabsl2State::Xabsl2State(const char* name, Xabsl2ErrorHandler& errorHandler,
                         unsigned long (*pTimeFunction)())
: Xabsl2NamedItem(name), errorHandler(errorHandler), decisionTree(0),
targetState(false), pTimeFunction(pTimeFunction)
{
}

Xabsl2State::~Xabsl2State()
{
  if (decisionTree != 0) delete decisionTree;
  for (int i=0; i<parameterValues.getSize(); i++)
    if (parameterValues[i]!=0)
      delete parameterValues[i];
}

void Xabsl2State::create(Xabsl2InputSource& input,
                         Xabsl2Array<Xabsl2Option*>& options,
                         Xabsl2Array<Xabsl2BasicBehavior&>& basicBehaviors,
                         Xabsl2Array<Xabsl2State*>& states,
                         Xabsl2Array<double>& parameters,
                         Xabsl2Symbols& symbols,
                         unsigned long& timeOfOptionExecution)
{ 
  XABSL2_DEBUG_INIT(errorHandler.message("creating state \"%s\"",n));

  char c[100]; char buf[100];
  
  // target state or not
  input.readString(c,1);
  if (*c == '1') 
  {
    targetState = true;
  }
  
  // subsequent option or basic behavior
  input.readString(c,1);
  input.readString(buf,99);
  switch (*c)
  {
  case 'o':
    subsequentOption = options.getElement(buf);
    subsequentBasicBehavior = 0;
    break;
  case 'b':
    if (!basicBehaviors.exists(buf))
    {
      errorHandler.error("Xabsl2State::create(): the subsequent basic behavior \"%s\" was not registered",buf);
      return;
    }
    subsequentBasicBehavior = &basicBehaviors.getElement(buf);
    subsequentOption = 0;
    break;
  }
  
  // read the parameters 
  int numberOfParameters = (int)input.readValue();
  
  int i;
  for (i = 0; i<numberOfParameters; i++)
  {
    input.readString(buf,99);
    XABSL2_DEBUG_INIT(errorHandler.message("creating expession for set parameter \"%s\"",buf));
    
    if (subsequentOption != 0)
    {
      if (!subsequentOption->parameters.exists(buf))
      {
        errorHandler.error("Xabsl2State::create(): the subsequent option does not have a parameter \"%s\"",buf);
        return;
      }
      parametersOfSubsequentBehavior.append(buf,&subsequentOption->parameters.getPElement(buf)->e);
    }
    else
    {
      if (!subsequentBasicBehavior->parameters.exists(buf))
      {
        errorHandler.error("Xabsl2State::create(): the subsequent basic behavior does not have a parameter \"%s\"",buf);
        return;
      }
      parametersOfSubsequentBehavior.append(buf,&subsequentBasicBehavior->parameters.getPElement(buf)->e);
    }

    parameterValues.append(buf,Xabsl2DecimalExpression::create(input,subsequentOption,errorHandler,parameters,symbols,timeOfOptionExecution,timeOfStateExecution));
    if (errorHandler.errorsOccurred)
    {
      errorHandler.error("Xabsl2State::create(): could not create decimal expression for parameter \"%s\"",buf);
      return;
    }
  }
  
  // set output symbols
  int numberOfSetOutputSymbol = (int)input.readValue();
  
  for(i = 0; i< numberOfSetOutputSymbol; i++)
  {
    input.readString(buf,99);
    XABSL2_DEBUG_INIT(errorHandler.message("creating reference to enumerated output symbol \"%s\"",buf));
    
    if (!symbols.existsEnumeratedOutputSymbol(buf))
    {
      errorHandler.error("Xabsl2State::create(): enumerated output symbol \"%s\" does not exist",buf);
      return;
    }
    outputSymbols.append(buf,symbols.getEnumeratedOutputSymbol(buf));
    input.readString(buf,99);
    
    if (!outputSymbols[i]->enumElements.exists(buf))
    {
      errorHandler.error("Xabsl2State::create(): enum element \"%s\" of enumerated output symbol \"%s\" does not exist",buf, outputSymbols[i]->n);
      return;
    }
    outputSymbolValues.append(buf,outputSymbols[i]->enumElements.getElement(buf)->v);
  }
  
  // transition to state or if / else block
  decisionTree = Xabsl2Statement::createStatement(input,subsequentOption,errorHandler,
    states,parameters, symbols, timeOfOptionExecution,timeOfStateExecution);
  if (errorHandler.errorsOccurred)
    errorHandler.error("Xabsl2State::create(): could not create decision tree for state \"%s\"",n);
}

Xabsl2State* Xabsl2State::getNextState()
{
  timeOfStateExecution = pTimeFunction() - timeWhenStateWasActivated;
  
  Xabsl2State* nextState = decisionTree->getNextState();
  
  return nextState;
}

void Xabsl2State::setOutputSymbols()
{
  int i;

  // set all parameters of the next subsequent behavior to 0
  if (subsequentBasicBehavior!=0)
  {
    for (i=0;i<subsequentBasicBehavior->parameters.getSize();i++)
    {
      subsequentBasicBehavior->parameters[i] = 0;
    }
  }
  else
  {
    for (i=0;i<subsequentOption->parameters.getSize();i++)
    {
      subsequentOption->parameters.setElement(i,0);
    }
  }
  
  // Sets the subsequent behavior parameters.
  for (i=0; i<parametersOfSubsequentBehavior.getSize(); i++)
  {
    *parametersOfSubsequentBehavior[i] = parameterValues[i]->getValue();
  }

  for (i = 0; i < outputSymbols.getSize(); i++)
  {
    outputSymbols[i]->activeValue = outputSymbolValues[i];
    outputSymbols[i]->activeValueWasSet = true;
  }
}

void Xabsl2State::reset()
{
  timeWhenStateWasActivated = pTimeFunction();
  timeOfStateExecution = 0;
}

bool Xabsl2State::isTargetState() const
{
  return targetState;
}

Xabsl2Option::Xabsl2Option(const char* name,
                           Xabsl2InputSource& input,
                           Xabsl2ErrorHandler& errorHandler,
                           unsigned long (*pTimeFunction)())
                           : Xabsl2NamedItem(name), activeState(0), optionIsActive(false), optionWasActive(false),
                           initialState(0), errorHandler(errorHandler),
                           pTimeFunction(pTimeFunction)
{
  int numberOfOptionParameters = (int)input.readValue();
  char buf[100];
  for (int i=0; i< numberOfOptionParameters; i++)
  {
    input.readString(buf,99);
    XABSL2_DEBUG_INIT(errorHandler.message("registering option parameter \"%s\" for option \"%s\"",buf,n));
    
    parameters.append(buf,0);
  }
}

Xabsl2Option::~Xabsl2Option()
{
  for (int i=0;i<states.getSize();i++)
    if (states[i]!=0) 
      delete states[i];
}

void Xabsl2Option::create(Xabsl2InputSource& input,    
                          Xabsl2Array<Xabsl2Option*>& options,
                          Xabsl2Array<Xabsl2BasicBehavior&>& basicBehaviors,
                          Xabsl2Symbols& symbols
                          )
{
  int i;
  
  // the number of states of the option
  int numberOfStates = (int)input.readValue();
  
  char stateName[100];
  
  // register all states to the array
  for (i=0; i< numberOfStates; i++)
  {
    input.readString(stateName, 99);
    states.append(stateName, new Xabsl2State(stateName,errorHandler,pTimeFunction));
  }
  XABSL2_DEBUG_INIT(errorHandler.message("Xabsl2Option::create(): registered %i states",numberOfStates));
  
  // set the initial state
  input.readString(stateName,99);
  initialState = states.getElement(stateName);
  activeState = initialState;
  
  XABSL2_DEBUG_INIT(errorHandler.message("Xabsl2Option::create(): initial state: \"%s\"",initialState->n));
  
  // create the states and their subelements
  for (i=0; i< numberOfStates; i++)
  {
    states[i]->create(input, options, basicBehaviors, states, parameters, symbols, timeOfOptionExecution);
    if (errorHandler.errorsOccurred)
    {
      errorHandler.error("Xabsl2Option::create(): could not create state \"%s\"",states[i]->n);
      return;
    }
  }
  
}  

void Xabsl2Option::execute()
{
  if (!optionWasActive)
  {
    timeWhenOptionWasActivated = pTimeFunction();
    activeState = initialState;
    activeState->reset();
  }
  optionIsActive = true;
  timeOfOptionExecution = pTimeFunction() - timeWhenOptionWasActivated;
  
  Xabsl2State* newState = activeState->getNextState();
  
  if (newState != activeState)
  {
    newState->reset();
  }
  activeState = newState;
  activeState->setOutputSymbols();
}

bool Xabsl2Option::getOptionReachedATargetState() const
{
  if (optionWasActive && activeState->isTargetState())
  {
    return true;
  }
  else
  {
    return false;
  }
}

/*
* Change Log:
*
* $Log: Xabsl2Option.cpp,v $
* Revision 1.1.1.1  2004/05/22 17:37:56  cvsadm
* created new repository GT2004_WM
*
* Revision 1.4  2004/05/12 13:58:26  loetzsch
* bug fix
* (creation of option graph crashed if a basic behavior was not registered)
*
* Revision 1.3  2004/03/15 09:58:15  dueffert
* time-of-state-execution bug fixed
*
* Revision 1.2  2003/10/08 11:50:09  loetzsch
* made the Xabsl2Engine really platform independent
* (removed inclusion of Platform/SystemCall.h)
* A time function is given to the engine by parameter.
*
* Revision 1.1  2003/10/07 10:13:25  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.5  2003/09/30 10:51:11  dueffert
* typos fixed
*
* Revision 1.4  2003/09/20 16:34:16  loetzsch
* renamed "following-option-..." to "subsequent-option-.." and
* "following-basic-behavior-.." to "subsequent-basic-behavior-.."
* for consistency with publications
*
* Revision 1.3  2003/08/04 16:00:28  loetzsch
* bug fix: the engine doesn't crash anymore when a parameter of a subsequent option or basic behavior is wrong
*
* Revision 1.2  2003/07/23 22:25:52  loetzsch
* implemented question mark operator
*
* Revision 1.1.1.1  2003/07/02 09:40:29  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.14  2003/05/25 22:36:32  loetzsch
* only when an output symbol was set during the execution of the graph,
* coresponding function is invoked or the coresponding variable is changed.
*
* Revision 1.13  2003/05/05 17:47:55  loetzsch
* added in Xabsl2:
* - marking of states as target-states with attribute is-target-state="true"
* - boolean expression "subsequent-option-reached-target-state"
* - common-decision-tree
*
* Revision 1.12  2003/03/28 14:18:49  loetzsch
* Parameters that are not set become set to 0.
*
* Revision 1.11  2003/03/06 11:44:47  dueffert
* re-order warnings removed
*
* Revision 1.10  2003/01/27 14:38:13  loetzsch
* in Xabsl2State::reset() the timeOfStateExecution is set to 0
*
* Revision 1.9  2003/01/15 08:18:47  juengel
* GT2003 compiles now.
*
* Revision 1.8  2003/01/14 16:28:32  loetzsch
* creation of references to output symbols added
* setting of output symbols added
*
* Revision 1.7  2003/01/13 22:39:38  loetzsch
* implemented the execution of the engine.
*
* Revision 1.6  2003/01/13 13:18:18  loetzsch
* Creation of boolean and decimal expressions finished.
*
* Revision 1.5  2003/01/12 14:54:04  loetzsch
* continued creation of option tree: Xabsl2Statement and derivates added
*
* Revision 1.4  2003/01/11 14:41:42  loetzsch
* continued creation of the option tree
*
* Revision 1.3  2003/01/09 17:28:33  loetzsch
* introduced Xabsl2Agent, continued Xabsl2Option
*
* Revision 1.2  2003/01/08 18:47:17  loetzsch
* first version of state implementation
*
* Revision 1.1  2003/01/08 15:22:33  loetzsch
* - started implementation of the option tree
* - started the reading of the intermediate code
*
*/
