/** 
* @file Xabsl2Engine.h
*
* Declaration class Xabsl2Engine
*
* @author Matthias Jngel
* @author Martin Ltzsch
*/

#ifndef __Xabsl2Engine_h_
#define __Xabsl2Engine_h_

#include "Xabsl2Agent.h"

/**
* Executes a behavior that was specified in the Xabsl language.
*
* @author Matthias Jngel
* @author Martin Ltzsch
*
*/
class Xabsl2Engine : public Xabsl2Symbols
{
public:
  /** 
  * Constructor 
  * @param e Is invoked when there are problems during initialization
  * @param pTimeFunction a pointer to a function that returns the system time in ms.
  */
  Xabsl2Engine(Xabsl2ErrorHandler& e, unsigned long (*pTimeFunction)());

  /** Destructor */
  ~Xabsl2Engine();

  /** 
  * Executes the engine for the selected agent starting from the root option.
  * (Including the selected basic behavior)
  */
  void execute();

  /** Executes only the selected basic behavior */
  void executeSelectedBasicBehavior();

  /**
  * Reads the intermediate code from an input source and creates the option graph.
  * Note that the basic behaviors and symbols have to be registered before that function is 
  * called.
  */
  void createOptionGraph(Xabsl2InputSource& input);

  /** 
  * Registers a basic behavior at the engine. 
  * This must be done before the intermediate code is read.
  * @param basicBehavior A reference to the basic behavior
  */
  void registerBasicBehavior(Xabsl2BasicBehavior& basicBehavior);

  /** 
  * Sets the selected Agent.
  * If the last selected agent was different from the new one, 
  * the root option is changed depending on the new agent.
  * @param name The name of the agent
  * @return if the requested agent exists
  */
  bool setSelectedAgent(const char* name);

private:
  /** The agents of the engine */
  Xabsl2Array<Xabsl2Agent*> agents;

  /** The selected agent */
  Xabsl2Agent* selectedAgent;

  /** The options of the engine */
  Xabsl2Array<Xabsl2Option*> options;

  /** The option where the execution of the option graph starts */
  Xabsl2Option* rootOption;

  /** The registered basic behaviors of the engine */
  Xabsl2Array<Xabsl2BasicBehavior&> basicBehaviors;

  /** The selected basic behavior */
  Xabsl2BasicBehavior* selectedBasicBehavior;

  /** Is invoked when there are problems during initialization */
  Xabsl2ErrorHandler& errorHandler;

  /** 
  * A recursive function that is used to check for loops in the option graph.
  * @param currenOptionPath An array of the currently traced option path
  * @param currentDepth The depth of the current option path
  * @return If true, then a loop was detected.
  */
  bool checkForLoops(Xabsl2Option* currenOptionPath[], int currentDepth);

  /** If true, the engine was successfully initialized */
  bool initialized;

  /** A pointer to a function that returns the system time in ms. */
  unsigned long (*pTimeFunction)();
public:
  //!@name Debugging Interface 
  //!@{

  /** 
  * Executes the option graph starting from a given option.
  * Can be called to test a single option.
  * @param name The name of the option
  * @return When false, the option is not known to the engine
  */
  bool setRootOption(const char* name);

  /** 
  * Sets the root option of the selected agent
  */
  void setRootOption();

  /** Returns the selected root option */
  const Xabsl2Option* getRootOption() const;

  /** 
  * Sets a parameter of a option.
  * @param name The name of the option
  * @param param The name of of the parameter
  * @param value The value for the parameter
  * @return When false, the option or the parameter does not exist
  */
  bool setOptionParameter(const char* name, const char* param, double value);


  /** 
  * Sets the selected basic behavior. 
  * Can be called to test a single basic behavior.
  * @param name The name of the basic behavior.
  * @return When false, the basic behavior was not registered
  */
  bool setSelectedBasicBehavior(const char* name);

  /** 
  * Sets a parameter of a basic behavior.
  * @param name The name of the basic behavior
  * @param param The name of of the parameter
  * @param value The value for the parameter
  * @return When false, the basic behavior or the parameter were not registered
  */
  bool setBasicBehaviorParameter(const char* name, const char* param, double value);

  /** Returns the name of the selected agent */
  const char* getSelectedAgentName();

  /** Returns the selected basic behavior */
  const Xabsl2BasicBehavior* getSelectedBasicBehavior();

  //!@}
};




#endif // __Xabsl2Engine_h_

/*
* Change Log
* 
* $Log: Xabsl2Engine.h,v $
* Revision 1.1.1.1  2004/05/22 17:37:56  cvsadm
* created new repository GT2004_WM
*
* Revision 1.3  2003/12/16 18:53:22  loetzsch
* The XabslEngine now checks for loops
*
* Revision 1.2  2003/10/08 11:50:09  loetzsch
* made the Xabsl2Engine really platform independent
* (removed inclusion of Platform/SystemCall.h)
* A time function is given to the engine by parameter.
*
* Revision 1.1  2003/10/07 10:13:25  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.4  2003/09/30 10:51:11  dueffert
* typos fixed
*
* Revision 1.3  2003/09/16 13:27:21  loetzsch
* changed all occurrences of "option tree" to "option graph"
*
* Revision 1.2  2003/08/04 16:02:50  loetzsch
* ::setSelectedAgent doesn't throw error messages when anymore
*
* Revision 1.1.1.1  2003/07/02 09:40:29  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.12  2003/01/28 18:07:47  loetzsch
* no message
*
* Revision 1.11  2003/01/28 17:51:35  loetzsch
* added function setOptionParameter()
*
* Revision 1.10  2003/01/19 13:04:52  loetzsch
* xabsl2 agents now can be changed by using the Module and SolutionRequest
* mechanism
*
* Revision 1.9  2003/01/12 14:54:04  loetzsch
* continued creation of option tree: Xabsl2Statement and derivates added
*
* Revision 1.8  2003/01/11 14:41:42  loetzsch
* continued creation of the option tree
*
* Revision 1.7  2003/01/09 17:28:33  loetzsch
* introduced Xabsl2Agent, continued Xabsl2Option
*
* Revision 1.6  2003/01/08 15:22:33  loetzsch
* - started implementation of the option tree
* - started the reading of the intermediate code
*
* Revision 1.5  2002/12/11 12:23:31  loetzsch
* basic behaviors register their parameters in their constructor
* the parameters array contains only references to doubles in the basic behavior
*
* Revision 1.4  2002/12/06 21:13:37  loetzsch
* Decimal input symbols can now be registered at the engine
*
* Revision 1.3  2002/12/02 19:56:32  loetzsch
* - Xabsl2Array now seems to work for more than 1 element
* - Basic behaviors now can be registered at the engine
* - Basic behaviors have access to their parameters
*
* Revision 1.2  2002/12/01 17:54:30  loetzsch
* continued Xabsl2Engine: Xabsl2Array seems to work now
*
* Revision 1.1  2002/12/01 13:45:58  loetzsch
* first version of Xabsl2Engine
*
*/

