/** 
* @file Xabsl2DecimalExpression.cpp
*
* Implementation of Xabsl2DecimalExpression and derivates
* 
* @author Martin Ltzsch
*/

#include "Xabsl2DecimalExpression.h"
#include "Xabsl2BooleanExpression.h"

Xabsl2DecimalExpression::~Xabsl2DecimalExpression()
{
}

Xabsl2DecimalExpression* Xabsl2DecimalExpression::create(Xabsl2InputSource& input, 
                                                         Xabsl2Option* subsequentOption,
                                                         Xabsl2ErrorHandler& errorHandler,
                                                         Xabsl2Array<double>& parameters,
                                                         Xabsl2Symbols& symbols,
                                                         unsigned long& timeOfOptionExecution,
                                                         unsigned long& timeOfStateExecution)
{
  char c[100];
  input.readString(c,1);
  Xabsl2ArithmeticOperator* arithmeticOperator;
  Xabsl2DecimalExpression* operand1;
  Xabsl2DecimalExpression* operand2;
  
  switch (*c)
  {
  case 'r':
    return new Xabsl2DecimalInputSymbolRef(input,errorHandler,symbols);
  case 'f':
    return new Xabsl2DecimalInputFunctionCall(input,subsequentOption,errorHandler,parameters,symbols,timeOfOptionExecution,timeOfStateExecution);
  case 'c':
    // constants are treates as decimal values (there is no difference from the engines point of view.)
  case 'v':
    return new Xabsl2DecimalValue(input,errorHandler, symbols);
  case 'p':
    return new Xabsl2OptionParameterRef(input,errorHandler,parameters);
  case '+':
    if (!Xabsl2DecimalExpression::createOperand(operand1,input,subsequentOption,errorHandler,parameters,symbols,timeOfOptionExecution,timeOfStateExecution))
      return 0;
    if (!Xabsl2DecimalExpression::createOperand(operand2,input,subsequentOption,errorHandler,parameters,symbols,timeOfOptionExecution,timeOfStateExecution))
      return 0;
    
    XABSL2_DEBUG_INIT(errorHandler.message("creating + operator"));
    arithmeticOperator = new Xabsl2PlusOperator();
    arithmeticOperator->create(operand1,operand2);
    return arithmeticOperator;
  case '-':
    if (!Xabsl2DecimalExpression::createOperand(operand1,input,subsequentOption,errorHandler,parameters,symbols,timeOfOptionExecution,timeOfStateExecution))
      return 0;
    if (!Xabsl2DecimalExpression::createOperand(operand2,input,subsequentOption,errorHandler,parameters,symbols,timeOfOptionExecution,timeOfStateExecution))
      return 0;
    
    XABSL2_DEBUG_INIT(errorHandler.message("creating - operator"));
    arithmeticOperator = new Xabsl2MinusOperator();
    arithmeticOperator->create(operand1,operand2);
    return arithmeticOperator;
  case '*':
    if (!Xabsl2DecimalExpression::createOperand(operand1,input,subsequentOption,errorHandler,parameters,symbols,timeOfOptionExecution,timeOfStateExecution))
      return 0;
    if (!Xabsl2DecimalExpression::createOperand(operand2,input,subsequentOption,errorHandler,parameters,symbols,timeOfOptionExecution,timeOfStateExecution))
      return 0;
    
    XABSL2_DEBUG_INIT(errorHandler.message("creating * operator"));
    arithmeticOperator = new Xabsl2MultiplyOperator();
    arithmeticOperator->create(operand1,operand2);
    return arithmeticOperator;
  case 'd':
    if (!Xabsl2DecimalExpression::createOperand(operand1,input,subsequentOption,errorHandler,parameters,symbols,timeOfOptionExecution,timeOfStateExecution))
      return 0;
    if (!Xabsl2DecimalExpression::createOperand(operand2,input,subsequentOption,errorHandler,parameters,symbols,timeOfOptionExecution,timeOfStateExecution))
      return 0;
    
    XABSL2_DEBUG_INIT(errorHandler.message("creating / operator"));
    arithmeticOperator = new Xabsl2DivideOperator();
    arithmeticOperator->create(operand1,operand2);
    return arithmeticOperator;
    
  case '%':
    if (!Xabsl2DecimalExpression::createOperand(operand1,input,subsequentOption,errorHandler,parameters,symbols,timeOfOptionExecution,timeOfStateExecution))
      return 0;
    if (!Xabsl2DecimalExpression::createOperand(operand2,input,subsequentOption,errorHandler,parameters,symbols,timeOfOptionExecution,timeOfStateExecution))
      return 0;
    
    XABSL2_DEBUG_INIT(errorHandler.message("creating % operator"));
    arithmeticOperator = new Xabsl2ModOperator();
    arithmeticOperator->create(operand1,operand2);
    return arithmeticOperator;
  case 's':
    return new Xabsl2TimeRef(errorHandler,timeOfStateExecution);
  case 'o':
    return new Xabsl2TimeRef(errorHandler,timeOfOptionExecution);
  case 'q':
    return new Xabsl2ConditionalExpression(input,subsequentOption,errorHandler,parameters,symbols,timeOfOptionExecution,timeOfStateExecution);
  default:
    errorHandler.error("Xabsl2DecimalExpression::create(): unknown expression type: \"%c\"",*c);
    return 0;
  }
}

bool Xabsl2DecimalExpression::createOperand(Xabsl2DecimalExpression*& operand,
                                            Xabsl2InputSource& input, 
                                            Xabsl2Option* subsequentOption,
                                            Xabsl2ErrorHandler& errorHandler,
                                            Xabsl2Array<double>& parameters,
                                            Xabsl2Symbols& symbols,
                                            unsigned long& timeOfOptionExecution,
                                            unsigned long& timeOfStateExecution)
{
  operand = Xabsl2DecimalExpression::create(input,subsequentOption,errorHandler,parameters, symbols,
    timeOfOptionExecution,timeOfStateExecution);
  
  if (operand == 0) 
  {
    errorHandler.error("Xabsl2DecimalExpression::createOperand(): created operand is 0");
    return false;
  }
  
  if (errorHandler.errorsOccurred)
  {
    errorHandler.error("Xabsl2DecimalExpression::createOperand(): could not create operand");
    if (operand != 0) delete operand;
    return false;
  }
  
  return true;
}

Xabsl2DecimalInputSymbolRef::Xabsl2DecimalInputSymbolRef(Xabsl2InputSource& input, 
                                                         Xabsl2ErrorHandler& errorHandler,
                                                         Xabsl2Symbols& symbols)
{
  char buf[100];
  input.readString(buf,99);
  XABSL2_DEBUG_INIT(errorHandler.message("creating a reference to decimal input symbol \"%s\"",buf));
  
  if (!symbols.existsDecimalInputSymbol(buf))
  {
    errorHandler.error("Xabsl2DecimalInputSymbolRef::Xabsl2DecimalInputSymbolRef(): decimal input symbol \"%s\" was not registered",buf);
    return;
  }
  
  symbol = symbols.getDecimalInputSymbol(buf);
}

double Xabsl2DecimalInputSymbolRef::getValue()
{
  return symbol->getValue();
}

Xabsl2DecimalValue::Xabsl2DecimalValue(Xabsl2InputSource& input, 
                                       Xabsl2ErrorHandler& errorHandler,
                                       Xabsl2Symbols& symbols)
{
  value = input.readValue();
  
  XABSL2_DEBUG_INIT(errorHandler.message("created decimal value: \"%.2f\"",value));
}

double Xabsl2DecimalValue::getValue()
{
  return value;
}

Xabsl2OptionParameterRef::Xabsl2OptionParameterRef(Xabsl2InputSource& input, 
                                                   Xabsl2ErrorHandler& errorHandler,
                                                   Xabsl2Array<double>& parameters)
{
  char buf[100];
  input.readString(buf,99);
  
  XABSL2_DEBUG_INIT(errorHandler.message("creating a reference to option parameter \"%s\"",buf));
  
  if (!parameters.exists(buf))
  {
    errorHandler.error("Xabsl2OptionParameterRef::Xabsl2OptionParameterRef(): option parameter \"%s\" does not exist",buf);
    return;
  }
  
  parameter = &parameters.getPElement(buf)->e;
}

double Xabsl2OptionParameterRef::getValue()
{
  return *parameter;
}

void Xabsl2ArithmeticOperator::create(Xabsl2DecimalExpression* operand1, Xabsl2DecimalExpression* operand2)
{
  this->operand1 = operand1;
  this->operand2 = operand2;
}

Xabsl2ArithmeticOperator::~Xabsl2ArithmeticOperator()
{
  if (operand1 != 0) delete operand1;
  if (operand2 != 0) delete operand2;
}

double Xabsl2PlusOperator::getValue()
{
  return operand1->getValue() + operand2->getValue();
}

double Xabsl2MinusOperator::getValue()
{
  return operand1->getValue() - operand2->getValue();
}

double Xabsl2MultiplyOperator::getValue()
{
  return operand1->getValue() * operand2->getValue();
}

double Xabsl2DivideOperator::getValue()
{
  double o2 = operand2->getValue();
  if (o2==0) 
    return operand1->getValue() / 0.0000001;
  else
    return operand1->getValue() / o2;
}

double Xabsl2ModOperator::getValue()
{
  return (int)operand1->getValue() % (int)operand2->getValue();
}

Xabsl2TimeRef::Xabsl2TimeRef(Xabsl2ErrorHandler& errorHandler,
                             unsigned long& time) :
time(time)
{
  XABSL2_DEBUG_INIT(errorHandler.message("creating a reference to state or option execution time"));
}

double Xabsl2TimeRef::getValue()
{
  return time;
}

Xabsl2DecimalInputFunctionCall::Xabsl2DecimalInputFunctionCall(Xabsl2InputSource& input, 
                                                               Xabsl2Option* subsequentOption,
                                                               Xabsl2ErrorHandler& errorHandler,
                                                               Xabsl2Array<double>& parameters,
                                                               Xabsl2Symbols& symbols,
                                                               unsigned long& timeOfOptionExecution,
                                                               unsigned long& timeOfStateExecution)
{
  char buf[100];
  input.readString(buf,99);
  
  XABSL2_DEBUG_INIT(errorHandler.message("creating a call to decimal input function \"%s\"",buf));
  
  if (!symbols.existsDecimalInputFunction(buf))
  {
    errorHandler.error("Xabsl2DecimalInputFunctionCall::Xabsl2DecimalInputFunctionCall(): decimal input function \"%s\" was not registered",buf);
    return;
  }
  
  function = symbols.getDecimalInputFunction(buf);
  
  int numberOfParameterValues = (int)input.readValue();
  
  for (int i=0; i< numberOfParameterValues; i++)
  {
    input.readString(buf,99);
    if (!function->parameters.exists(buf))
    {
      errorHandler.error("Xabsl2DecimalInputFunctionCall::Xabsl2DecimalInputFunctionCall(): parameter \"%s\" of function \"%s\" does not exist.", buf,function->n);
      return;
    }
    functionParameters.append(buf,&function->parameters.getPElement(buf)->e);
    
    XABSL2_DEBUG_INIT(errorHandler.message("creating expression for parameter \"%s\"",buf));
    
    Xabsl2DecimalExpression* parameterValue;
    if (!Xabsl2DecimalExpression::createOperand(parameterValue,input,subsequentOption,errorHandler,parameters,symbols,timeOfOptionExecution,timeOfStateExecution))
    {
      errorHandler.error("Xabsl2DecimalInputFunctionCall::Xabsl2DecimalInputFunctionCall(): could not create decimal expression",buf);
      return;
    }
    parameterValues.append(buf,parameterValue);
  }
}

Xabsl2DecimalInputFunctionCall::~Xabsl2DecimalInputFunctionCall()
{
  for (int i=0; i<parameterValues.getSize(); i++)
    if (parameterValues[i] != 0)
      delete parameterValues[i];
}

double Xabsl2DecimalInputFunctionCall::getValue()
{
  // set the function parameters
  for (int i=0; i<functionParameters.getSize(); i++)
    *functionParameters[i] = parameterValues[i]->getValue();
  
  // call the function
  return function->getValue();
}

Xabsl2ConditionalExpression::Xabsl2ConditionalExpression(Xabsl2InputSource& input, 
    Xabsl2Option* subsequentOption,
    Xabsl2ErrorHandler& errorHandler,
    Xabsl2Array<double>& parameters,
    Xabsl2Symbols& symbols,
    unsigned long& timeOfOptionExecution,
    unsigned long& timeOfStateExecution)
{
  XABSL2_DEBUG_INIT(errorHandler.message("creating a question mark operator",buf));

  condition = Xabsl2BooleanExpression::create(input,subsequentOption,errorHandler,parameters, symbols,
    timeOfOptionExecution,timeOfStateExecution);

  if (condition == 0) 
  {
    errorHandler.error("Xabsl2QuestionMarkOperator::Xabsl2QuestionMarkOperator(): created condition is 0");
    return;
  }
  else if (errorHandler.errorsOccurred)
  {
    errorHandler.error("Xabsl2QuestionMarkOperator::Xabsl2QuestionMarkOperator(): could not create condition");
    delete condition;
    return;
  }

  if (!Xabsl2DecimalExpression::createOperand(expression1,input,subsequentOption,errorHandler,parameters,symbols,timeOfOptionExecution,timeOfStateExecution))
  {
    errorHandler.error("Xabsl2QuestionMarkOperator::Xabsl2QuestionMarkOperator(): could not create decimal expression1");
    return;
  }
  
  if (!Xabsl2DecimalExpression::createOperand(expression2,input,subsequentOption,errorHandler,parameters,symbols,timeOfOptionExecution,timeOfStateExecution))
  {
    errorHandler.error("Xabsl2QuestionMarkOperator::Xabsl2QuestionMarkOperator(): could not create decimal expression2");
    return;
  }
  
}

Xabsl2ConditionalExpression::~Xabsl2ConditionalExpression()
{
  if (condition!=0) delete condition;
  if (expression1!=0) delete expression1;
  if (expression2!=0) delete expression2;
}

double Xabsl2ConditionalExpression::getValue()
{
  if (condition->getValue())
  {
    return expression1->getValue();
  }
  else
  {
    return expression2->getValue();
  }
}

/*
* Change Log:
*
* $Log: Xabsl2DecimalExpression.cpp,v $
* Revision 1.1.1.1  2004/05/22 17:37:54  cvsadm
* created new repository GT2004_WM
*
* Revision 1.2  2004/03/27 21:06:14  loetzsch
* now division by 0 possible
*
* Revision 1.1  2003/10/07 10:13:25  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.5  2003/09/30 10:51:10  dueffert
* typos fixed
*
* Revision 1.4  2003/09/20 16:34:15  loetzsch
* renamed "following-option-..." to "subsequent-option-.." and
* "following-basic-behavior-.." to "subsequent-basic-behavior-.."
* for consistency with publications
*
* Revision 1.3  2003/08/09 12:19:05  loetzsch
* renamed question-mark-operator to conditional-expression
*
* Revision 1.2  2003/07/23 22:25:52  loetzsch
* implemented question mark operator
*
* Revision 1.1.1.1  2003/07/02 09:40:29  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.3  2003/04/01 13:08:11  dueffert
* warning removed
*
* Revision 1.2  2003/01/14 21:07:42  loetzsch
* change intermediate code for xabsl:divide from '/' to 'd', because '/'
* caused the input file stream to crash
*
* Revision 1.1  2003/01/13 13:18:18  loetzsch
* Creation of boolean and decimal expressions finished.
*
*/
