/** 
* @file Xabsl2BasicBehavior.h
*
* Declaration class Xabsl2BasicBehavior
*
* @author Matthias Jngel
* @author Martin Ltzsch
*/

#ifndef __Xabsl2BasicBehavior_h_
#define __Xabsl2BasicBehavior_h_

#include "Xabsl2Tools.h"

/**
* The base class for basic behaviors that are used by the XabslEngine
*
* @author Matthias Jngel
* @author Martin Ltzsch
*
*/
class Xabsl2BasicBehavior : public Xabsl2NamedItem
{
public:
  /** 
  * Constructor 
  * @param name The name of the basic behavior
  * @param errorHandler A reference 
  */
  Xabsl2BasicBehavior(const char* name, Xabsl2ErrorHandler& errorHandler)
    : Xabsl2NamedItem(name), errorHandler(errorHandler),
    basicBehaviorWasActiveDuringLastExecutionOfEngine(false)
  {};
  
  /** 
  * executes the basic behavior
  */
  virtual void execute() = 0;
  
  /** sets all parameter values to 0 */
  void reset() 
  { for (int i=0; i<parameters.getSize(); i++) parameters[i]=0; }
  
  /** The parameters of the basic behavior */
  Xabsl2Array<double&> parameters;

  /** Determines if the basic behavior was active during the last execution of the engine */
  bool basicBehaviorWasActiveDuringLastExecutionOfEngine;

/** 
* Registeres a reference to a parameter at the parameters array.
* @param name The name of the parameter
* @param parameter The reference to a parameter of the basic behavior
  */
  void registerParameter(const char* name, double& parameter)
  {
    XABSL2_DEBUG_INIT(errorHandler.message("registering parameter \"%s\" of basic behavior \"%s\"",name,n));
    
    if (parameters.exists(name))
    {
      errorHandler.error("registerParameter(): parameter \"%s\" was already registered",name);
      return;
    }
    parameters.append(name,parameter);
    parameter = 0;
  };
  
protected:
  /** Is invoked when a errors occur */
  Xabsl2ErrorHandler& errorHandler;
};


#endif // __Xabsl2BasicBehavior_h_

/*
* Change Log
* 
* $Log: Xabsl2BasicBehavior.h,v $
* Revision 1.1.1.1  2004/05/22 17:37:53  cvsadm
* created new repository GT2004_WM
*
* Revision 1.1  2003/10/07 10:13:25  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.3  2003/09/30 10:51:10  dueffert
* typos fixed
*
* Revision 1.2  2003/09/20 16:34:15  loetzsch
* renamed "following-option-..." to "subsequent-option-.." and
* "following-basic-behavior-.." to "subsequent-basic-behavior-.."
* for consistency with publications
*
* Revision 1.1.1.1  2003/07/02 09:40:29  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.14  2003/05/06 07:38:08  risler
* registerParameter public
*
* Revision 1.13  2003/05/05 17:47:55  loetzsch
* added in Xabsl2:
* - marking of states as target-states with attribute is-target-state="true"
* - common-decision-tree
*
* Revision 1.12  2003/04/14 16:18:51  loetzsch
* ATH after GermanOpen CVS merge
* added basicBehaviorWasActiveDuringLastExecutionOfEngine
*
* Revision 1.2  2003/04/09 17:01:06  loetzsch
* added the boolean basicBehaviorWasActiveDuringLastExecutionOfEngine
*
* Revision 1.1.1.1  2003/04/09 14:23:22  loetzsch
* started Aibo Team Humboldt's GermanOpen CVS
*
* Revision 1.11  2003/03/06 11:45:42  dueffert
* re-order warning removed
*
* Revision 1.10  2003/02/20 15:55:02  dueffert
* bug fixed
*
* Revision 1.9  2003/01/13 22:39:38  loetzsch
* implemented the execution of the engine.
*
* Revision 1.8  2003/01/13 21:59:08  timrie
* corrected doxygen comments
*
* Revision 1.7  2003/01/11 14:41:06  loetzsch
* The name of the basic behavior must be specified in the constructor
*
* Revision 1.6  2003/01/03 14:43:55  dueffert
* doxygen docu corrected
*
* Revision 1.5  2002/12/11 12:23:31  loetzsch
* basic behaviors register their parameters in their constructor
* the parameters array contains only references to doubles in the basic behavior
*
* Revision 1.4  2002/12/06 21:13:37  loetzsch
* Decimal input symbols can now be registered at the engine
*
* Revision 1.3  2002/12/02 19:56:32  loetzsch
* - Xabsl2Array now seems to work for more than 1 element
* - Basic behaviors now can be registered at the engine
* - Basic behaviors have access to their parameters
*
* Revision 1.2  2002/12/01 17:54:30  loetzsch
* continued Xabsl2Engine: Xabsl2Array seems to work now
*
* Revision 1.1  2002/12/01 13:45:58  loetzsch
* first version of Xabsl2Engine
*
*/

