/**
* @file GTXabsl2Profiler.h
* 
* Definition and Implementation of classes GTXabsl2Profiler, GTXabsl2ProfilerNameTable
*
* @author Michael Spranger
*/
#ifndef __GTXabsl2Profiler_h_
#define __GTXabsl2Profiler_h_

#ifdef _WIN32
#pragma warning(disable:4786) 
// the constructor with all it's long parameter type names results in a too
// long debug identifier
#endif

#include <string>
#include <vector>
#include <deque>

#include "Platform/GTAssert.h"

#include "Tools/Streams/OutStreams.h"
#include "Platform/SystemCall.h"

#include "Tools/Xabsl2/Xabsl2Engine/Xabsl2Array.h"
#include "Tools/Xabsl2/Xabsl2Engine/Xabsl2Engine.h"
#include "Tools/Debugging/Debugging.h"

#include "Tools/Streams/InStreams.h"
#include "Tools/Module/SolutionRequest.h"

/* Nr of Logelements written out per call to writeLog */
#define NROFELEMW 2

/*
* @class GTXabsl2ProfilerNameTableEntry
* Class that encapsulates string representations for options, state in options and optionparameters
* @author Michael Spranger
*/
class GTXabsl2ProfilerNameTableEntry{
  public:
    /** The name of the Option represented by this entry */
    std::string optionName;
    /**The vector of state names */
    std::vector<std::string> states;
    /** A vector of parameter names */
    std::vector<std::string> parameters;
    /** The maximal depth of this option in the option-tree (XABSL)*/
    int maxdepth;
    
    /**
    * Constructor
    * Initializes the strings as zero length strings.
    */
    GTXabsl2ProfilerNameTableEntry(){};

    
    /**
    * Constructor
    * @param option The option name
    * @param state The names of all state within the corresponding option
    * @param params The names of all parameters for this Option
    * @param depth The maximal depth of this option in the option tree.
    */
    GTXabsl2ProfilerNameTableEntry(const std::string option,std::vector<std::string> state, std::vector<std::string> params, const int depth)
      : optionName(option), states(state), parameters(params), maxdepth(depth){}
};

/*
* @class GTXabsl2ProfilerNameTable
* Dynamic Array for keeping a correlation of Xabsl2Options and States to Numbers for profiling purposes (remembering activation paths).
* @author Michael Spranger
*/
class GTXabsl2ProfilerNameTable
{
  /** The array */
  GTXabsl2ProfilerNameTableEntry* data;
  
  
  /** The number of elements in the array */
  int usedSize, length;
  public:
    /** Constructor 
    * Initializes all members with 0.
    */
    GTXabsl2ProfilerNameTable():usedSize(0), length(0), data(NULL)
    {}
    
    /** Destructor 
    * Deletes the data
    */
    ~GTXabsl2ProfilerNameTable() 
    { 
      if(data)
        delete[] data;
    }
    /**
    * Initializes the Array to a given length
    * @param length_ The length of the initialised array
    */
    void init(int length_){
      if(data){
        delete[] data; data = NULL;
      }
      usedSize = 0;
      length = length_;
      data = new GTXabsl2ProfilerNameTableEntry[length];
    }
    
    /** Clears the array */
    void clear()
    { 
      if(data){
        delete[] data;
        usedSize = 0;
        data = new GTXabsl2ProfilerNameTableEntry[length];
      }
      
    }
    
    /** 
    * Returns the Position for a given optionname
    * Note that the function crashes if the element does not exist.
    */
    int getOptionPosition(const std::string optionName) const
    {
      return findOption(optionName);
    }

    /** 
    * Returns the Position for a given state within the option
    * Note that the function crashes if the element does not exist.
    * @param optionName Name of the option the state is in
    * @param state Name of the state the position is returned for
    * @return The Position of the state within NameTableEntry
    */
    int getStatePosition(const std::string optionName, const std::string state){
      return findState(optionName, state);
    }
    
    /**
    * The function appends a new GTXabsl2ProfilerNameTableEntry to the array.
    * @param optionName A string label for the option..
    * @param stateName The names of the state in the corresponding option.
    * @param params The names of the parameters for this option.
    * @param depth The maximal depth of this Entry/Option in the option tree.
    */  
    void append(const std::string optionName, const std::vector<std::string> states, const std::vector<std::string> params, int depth)
    {
      if(usedSize == length)
      {
        length += length > 1 ? length / 2 : 2; 
        
        GTXabsl2ProfilerNameTableEntry* temp = new GTXabsl2ProfilerNameTableEntry[length];
        if(data){
          for(int i = 0; i < getSize(); ++i)
            temp[i] = data[i];
          delete[] data;
        }
        data = temp;
      }
      data[usedSize++]=GTXabsl2ProfilerNameTableEntry(optionName, states, params, depth);
    }
    
    
    /**
    * The function sets the value of an element in the array.
    * Note that the function crashes if the element does not exist.
    * @param pos The position of the element in the array.
    * @param value The new element.
    */
    void setElement(int pos, GTXabsl2ProfilerNameTableEntry value)
    {
      data[pos] = value;
    }
    
    /**
    * The function returns the number of elements in the array.
    * @return The length of the list.
    */
    int getSize() const {return usedSize;}
    
    /** 
    * Returns the value for a given array position.
    * Note that the function crashes if the required position is bigger than the 
    * size of the array.
    */
    GTXabsl2ProfilerNameTableEntry& operator[](int pos) const
    {
      return data[pos];
    }
    
    /** Returns whether an element for the given name exists 
    * @param optionName Name of the option searched for
    * @param stateName Name of the state that in conjunction with optionName makes the request unique
    */
    bool exists(const std::string optionName, const std::string stateName) const
    {
      return find(optionName, stateName) >= 0;
    }

    /** Returns whether an element for the given namey exists 
    * @param optionName Name of the option searched for
    */
    bool existsOption(const std::string option){
      return findOption(option) >=0;
    }
    
   protected:
   /** 
   * Returns the index of an element with the given names.
   * @return The index of the element of -1 if the name does not exist.
   */
     int find(const std::string optionName, const std::string state) const
     {
       for(int i = 0; i < getSize(); ++i){
         
         if(data[i].optionName  == optionName){
           for(std::vector<std::string>::iterator j = data[i].states.begin(); j != data[i].states.end(); ++j){
             if((*j) == state)
               return i;
           }
           
         }
       }
       return -1;
     }
     
     /** 
     * Returns the index of an element with the given names.
     * @return The index of the element of -1 if the name does not exist.
     */
     int findState(const std::string optionName, const std::string state) const
     {
       for(int i = 0; i < getSize(); ++i){
        
         if(data[i].optionName  == optionName){
           for(int j = 0; j < (int) data[i].states.size(); ++j){
             if(state == data[i].states[j])
               return j;
           }
           
         }
       }
       return -1;
     }

     /** 
     * Returns the index of an element with the given names.
     * @return The index of the element of -1 if the name does not exist.
     */
     int findOption(const std::string optionName) const
     {
       for(int i = 0; i < getSize(); ++i)
         if(data[i].optionName  == optionName) 
           return i;
         return -1;
     }
};

/*
* @class GTXabsl2ProfilerNameTable
* Encapsulation of a part of a LogEntry. Wraps the knwoledge about a active option. That is an option which is active at a given point in time, while the profiler is running.
* @author Michael Spranger
*/
class GTXabsl2ActiveOption{
public:
  /** Number of the option. Index within the NameTable used for lookup */
  int optionNumber;
  
  /** Number of the state that is active within the option. Index within a NameTableEntry used for lookup */
  int stateNumber;
  
  /** Values for parameters of the option, where the position is the same as the names within a NametableEntry */
  std::vector<double> parameters;

  /*
   *	Constructor
   * Initialises to non-possible values
   */
  GTXabsl2ActiveOption(): optionNumber(-1), stateNumber(-1){}
  
  /*
   * Constructor
   * Initialises to corresponding values
   */
  GTXabsl2ActiveOption(int o, int st, std::vector<double> p): optionNumber(o), stateNumber(st), parameters(p){}
  
  /*
   *	Equality operator. Based on memberwise comparison. 
   */
  bool operator ==(const GTXabsl2ActiveOption other)const
  {
    if(this->optionNumber == other.optionNumber && this->stateNumber == other.stateNumber
      && this->parameters == other.parameters)
      return true;
    return false;
  }
};

/**
* @class GTXabsl2LogEntry
* Class for keeping numerical activation paths and framenumbers
* @author Michael Spranger
*/
class GTXabsl2LogEntry{
public:
  /** framenumber of the entry */
  unsigned long framenumber;

  /** activation path within XABSL-tree */
  std::vector<GTXabsl2ActiveOption> activeOptions;

  /*
   *	Constructor
   */
  GTXabsl2LogEntry(unsigned long fn, std::vector<GTXabsl2ActiveOption> aO):framenumber(fn), activeOptions(aO){}
  GTXabsl2LogEntry(unsigned long fn):framenumber(fn){}
};

/**
* @class GTXabsl2Profiler
* Class for profiling and analysis of Xabsl-Activation-Paths
* @author Michael Spranger
*/
class GTXabsl2Profiler
{
private:

  /** NameTable for keeping corresponding names to numerical logs */
  GTXabsl2ProfilerNameTable nameTable;

  /** Numerical Log where indices are usually corresponding to indices in the NameTable*/
  std::deque<GTXabsl2LogEntry> log;

  /** Name of the file the Log is written to */
  std::string outFileName;

  /** Maximal depth of the XABSL-Option-Tree */
  int maxDepth;

  
  /*
   * Registers Option, calls itself recursively for subsequent options, registering all subsequent options as well
   * @param  option Option to be registered
   * @param depth depth of the option to be registered
   */
  void registerOptions(const Xabsl2Option* option, int depth);

  /*
   *	Does a depth count for the option, where depth is initial depth
   */
  void doDepthCount(const Xabsl2Option* option, int depth);

  /*
   *	Writes the number of NROFELEMW to the stream, erasing the logentries from the log.
   */
  void writeLogToStream(Out&);

  /*
   *	Writes complete log to stream. Erases complete log.
   */
  void writeCompleteLogToStream(Out&);

  /*
   *	Write NameTable to Stream
   */
  void writeNameTableToStream(Out&);

  /** A reference to a variable containing the current frame number */
  const unsigned long* frameNumber;

public:
  /** The Xabsl-Output-Symbols for the Profiler */
  enum { dontCollectProfiles, collectProfiles} profilerCollectMode;
  enum { dontWriteProfiles, writeProfiles, writeCompleteProfiles} profilerWriteMode;
  
  /*
   * Called by Xabls-Engine to make profiler symbols visible to behavior
   */
  void registerSymbols(Xabsl2Engine&);

  GTXabsl2Profiler();
  /** Constructor
  * @param id The id of the observed xabsl engine 
  * @param frameNumber A reference to the frame number
  */ 
  GTXabsl2Profiler(SolutionRequest::xabsl2EngineID id,
      const unsigned long* frameNumber);

  /*
   *	Initialises Engine, creates nametable and writes it to outfilename
   * @param pEnginge Reference to the engine to be profiled
   * @param length if known length refers to the number of options, so the nametable can be iniialised
   */
  void init(Xabsl2Engine& pEngine,int length =1);

  /*
   * Runs through the Engine determine active options, states and their parameters, writes the option-activation-path to log
   * @param pEngine XabslEngine to be profiled
   */
  void doProfiling(Xabsl2Engine& pEngine);

  /*
   * Write a certain number (NROFELEMW) of logs to outfilename, erasing them from the log.
   */
  void recordCollectedLogs();

  /*
   * Writes the complete log to outfilename, erasing all up to now collected logs.
   */
  void recordCompleteLog();
  
  /*
   * Writes an XML-representation of the current held log to the stream.
   */
  void writeXMLtoStream(Out& out);

  /*
   * Write an XML-representation of the current log to the file outf
   */
  void exportXMLFile(const char* outf);

  /*
   * Imports a log from a logfile
   * @param filename name of the logfile to be imported
   */
  bool importLogFile(const char* filename);

  /*
   * Gets all the options that were active at specified point in time
   * @param time framenumber of the active-option-path to be returned
   * @return active-option-path
   */
  std::vector<GTXabsl2ActiveOption> getActiveOptionsAtFrame(int time);

  /*
   * Returns corresponding name to optionnumber from nametable
   * @param optionnumber number of the option (index to nametable)
   */
  std::string getOptionName(int optionNumber);
  /*
   * Returns corresponding name to statenumber from nametable
   * @param optionnumber number of the option (index to nametable)
   */
  std::string getStateName(int optionNumber, int stateNumber);

  /*
   * @return The maximal depth of the option in the option tree
   */
  int getDepth(char* optionName);

  /*
   * @return The maximal depth of the option in the option tree
   */
  int getDepth(int optionNumber);

  /*
   * @return the maximal depth of the whole option-tree
   */
  int getMaxDepth(){ return maxDepth; }

  /*
   * @return the beginning framenumber of the current log
   */
  int getBeginningFramenumber(){ return log.size()?log.front().framenumber:0; }

  
  /*
   * @return the last framenumber of the current log
   */
  int getLastFramenumber(){ return log.size()?log.back().framenumber:0;}

  /*
   * @return the current number of log entries
   */
  int getNumberofLogEntries();

  /*
   * @return a representation of the active options and their parameters and the active states at framenumber
   */
  std::vector<GTXabsl2ActiveOption> getActiveOptionsAtNumber(int frameNumber);

  /*
   * Gets the active options at the position index in log
   * @return numerical representation of active options at logentry index
   */
  GTXabsl2LogEntry getLogEntryAtIndex(int index);

  /*
   * Gets the following framenumber of frame
   */
  int getFollowingFramenumber(int frame); 
  
  /*
   * Gets the index to the frameNumber
   */
  int getIndex(int frameNumber);


  /*
   * Access to the log
   */
  GTXabsl2LogEntry& operator[](int i);

  /*
   * Gets size of log.
   */
  int size();

  /*
   * Gets complete nametableentryfor index(which is equal to an optionnumber)
   */
  GTXabsl2ProfilerNameTableEntry& getNameTableEntry(int index);

  /*
   * Gets an activeOption at index with a maximum depth
   * @param index index of the logentry
   * @param maxdepth maximum depth of the returned option
   * @param Reference to option, holding an option of maximumdeppth at index
   * @return wether there was an option found at specified maxdepth and index
   */
  bool getActiveOption(int index, int maxdepth, GTXabsl2ActiveOption& retour);
};

#endif// __GTXabsl2Profiler_h_


/*
* Change Log:
*
* $Log: GTXabsl2Profiler.h,v $
* Revision 1.5  2004/06/14 17:15:51  spranger
* added documentation
*
* Revision 1.4  2004/05/27 18:42:24  loetzsch
* removed warnings
*
* Revision 1.3  2004/05/24 16:34:43  spranger
* removed DebugLogToStream, bugfix
*
* Revision 1.2  2004/05/24 13:17:51  spranger
* comment out xml-export
*
* Revision 1.1  2004/05/23 18:59:42  spranger
* changed interface, added framenumber to Constructor
*
* Revision 1.10  2004/05/22 16:58:04  spranger
* changed dialog appearance, added features (including extended profiler-interface)
*
* Revision 1.9  2004/05/04 15:35:02  spranger
* extended interface
*
* Revision 1.8  2004/05/03 13:21:57  spranger
* extended interface
*
* Revision 1.7  2004/04/30 09:26:23  spranger
* redone interface, added xml-export
*
* Revision 1.6  2004/04/28 17:31:40  spranger
* redesign of nametable, added import facilities
*
* Revision 1.5  2004/04/22 12:21:33  spranger
* removed compiler warnings
*
* Revision 1.4  2004/04/22 11:33:19  spranger
* different log handling (queue), added writingoglogs which does not write out whole log
*
* Revision 1.3  2004/04/20 18:17:01  spranger
* added nametable-structure, added log functionality
*
* Revision 1.2  2004/04/14 13:33:54  loetzsch
* added change log
*
*/
