/**
* @file GTXabsl2EngineExecutor.cpp
* 
* Implementation of class GTXabsl2EngineExecutor.
*
* @author Martin Ltzsch
*/

#include "GTXabsl2EngineExecutor.h"
#include "Tools/Debugging/Debugging.h"
#include "Platform/GTAssert.h"
#include "Platform/SystemCall.h"

GTXabsl2EngineExecutor::GTXabsl2EngineExecutor (SolutionRequest::xabsl2EngineID id,
                                                SolutionRequest::ModuleID module,
                                                const unsigned long& frameNumber) 
: pEngine(0), errorHandler(id), id(id), debugMode(executeOption), 
module(module), profiler(id, &frameNumber)
{
}

GTXabsl2EngineExecutor::~GTXabsl2EngineExecutor()
{
  if (pEngine!=0) delete pEngine;
}

void GTXabsl2EngineExecutor::init(Xabsl2InputSource& input)
{
  if (pEngine != 0) delete pEngine;
  watchedBooleanInputSymbols.clear();
  watchedDecimalInputSymbols.clear();
  watchedEnumeratedInputSymbols.clear();
  watchedEnumeratedOutputSymbols.clear();
  setEnumeratedOutputSymbols.clear();
  setEnumeratedOutputSymbolValues.clear();
  
  this->debugMode = executeRootOption;
  
  errorHandler.errorsOccurred = false; // reset the error handler
  
  unsigned long startTime = SystemCall::getCurrentSystemTime();
  unsigned long freeMemoryBefore = SystemCall::getFreeMem();
  
  pEngine = new Xabsl2Engine(errorHandler,&SystemCall::getCurrentSystemTime);
  
  registerSymbolsAndBasicBehaviors();
  profiler.registerSymbols(*pEngine);
  
  pEngine->createOptionGraph(input);

  if (!errorHandler.errorsOccurred)
  {
    errorHandler.message("created a new Engine (%li ms, %li bytes)", 
      SystemCall::getCurrentSystemTime() - startTime, 
      freeMemoryBefore - SystemCall::getFreeMem());
    profiler.init(*pEngine);
    
  }
}

void GTXabsl2EngineExecutor::executeEngine()
{
  if (errorHandler.errorsOccurred)
  {
    executeIfEngineCouldNotBeCreated();
    return;
  }
  
  switch (debugMode)
  {
  case executeRootOption:
    pEngine->setRootOption();
    pEngine->execute();
    break;
  case executeOption:
    // execute the complete option graph beginning from the root option
    // (the root option can be changed from the Xabsl2 dialog 
    pEngine->execute();
    break;
  case executeBasicBehavior:
    // don't execute the option graph but only the basic behavior
    pEngine->executeSelectedBasicBehavior();
    break;
  }
  
  // do profiling (only when executing an engine)
  if(debugMode == executeRootOption || debugMode == executeOption){
    
    if(profiler.profilerCollectMode == GTXabsl2Profiler::collectProfiles)
      profiler.doProfiling(*pEngine);
    if(profiler.profilerWriteMode == GTXabsl2Profiler::writeProfiles && profiler.size()){
      profiler.recordCollectedLogs();
    }
    else if(profiler.profilerWriteMode == GTXabsl2Profiler::writeCompleteProfiles && profiler.size()){
      profiler.recordCompleteLog();
    }
    profiler.profilerWriteMode = GTXabsl2Profiler::dontWriteProfiles;
  }
  // Set the output symbols that were requested by the Xabsl2 Dialog
  for (int i=0; i<setEnumeratedOutputSymbols.getSize(); i++)
  {
    setEnumeratedOutputSymbols[i]->activeValue = setEnumeratedOutputSymbolValues[i];
    setEnumeratedOutputSymbols[i]->activeValueWasSet = true;
  }
  pEngine->setOutputSymbols();
  
  // send debug messages to the Xabsl2 Dialog if requested
  switch(module)
  {
  case SolutionRequest::behaviorControl:
    if (getDebugKeyTable().isActive(DebugKeyTable::sendXabsl2DebugMessagesForBehaviorControl)) 
      sendDebugMessage();
    break;
  case SolutionRequest::headControl:
    if (getDebugKeyTable().isActive(DebugKeyTable::sendXabsl2DebugMessagesForHeadControl)) 
      sendDebugMessage();
    break;
  default:
    ASSERT(false);

  }
}

void GTXabsl2EngineExecutor::sendDebugMessage()
{
  int i, j;// char buf[200];
  
  Out& out = getDebugOut().bin;
  
  // send the id of the behavior contol solution
  out << (char)id;
  
  // send the name of the selected agent 
  out << pEngine->getSelectedAgentName();
  
  // watched decimal input symbols;
  out << watchedDecimalInputSymbols.getSize();
  for (i = 0; i < watchedDecimalInputSymbols.getSize(); i++)
    out << watchedDecimalInputSymbols[i]->n << watchedDecimalInputSymbols[i]->getValue();
  
  // watched boolean input symbols;
  out << watchedBooleanInputSymbols.getSize();
  for (i = 0; i < watchedBooleanInputSymbols.getSize(); i++)
    out << watchedBooleanInputSymbols[i]->n << (char)watchedBooleanInputSymbols[i]->getValue();
  
  // watched enumerated input symbols;
  out << watchedEnumeratedInputSymbols.getSize();
  for (i = 0; i < watchedEnumeratedInputSymbols.getSize(); i++)
  {
    Xabsl2EnumeratedInputSymbol* s = watchedEnumeratedInputSymbols[i];
    out << s->n;
    int v = s->getValue();
    for (j = 0; j < s->enumElements.getSize();j++)
    {
      if (s->enumElements[j]->v == v) break;
    }
    if (j==s->enumElements.getSize())
    {
      out << " ";
    }
    else
    {
      out << s->enumElements[j]->n; 
    }
  }
  
  // watched enumerated output symbols;
  out << watchedEnumeratedOutputSymbols.getSize();
  for (i=0; i<watchedEnumeratedOutputSymbols.getSize(); i++)
  {
    Xabsl2EnumeratedOutputSymbol* s = watchedEnumeratedOutputSymbols[i];
    out << s->n;
    for (j = 0; j < s->enumElements.getSize();j++)
    {
      if (s->enumElements[j]->v == s->activeValue) break;
    }
    if (j==s->enumElements.getSize())
    {
      out << " ";
    }
    else
    {
      out << s->enumElements[j]->n; 
    }
  }
  
  // the number of active options
  int numberOfActiveOptions = 0;
  const Xabsl2Option* option;  
  if (debugMode == executeOption || debugMode == executeRootOption)  
    // number of active options is 0 when basic behaviors are tested separately
  {
    option = pEngine->getRootOption();
    
    while (option!=0)
    {
      option = option->activeState->subsequentOption;
      numberOfActiveOptions++;
    }
  }
  out << numberOfActiveOptions;
  
  // For each active option: 
  // name, duration of activation, active state, duration of state activation
  if (debugMode == executeOption || debugMode == executeRootOption)  
  {
    option = pEngine->getRootOption();
    
    while (option!=0)
    {
      out << option->n << option->timeOfOptionExecution 
        << option->activeState->n << option->activeState->timeOfStateExecution; 
      
      // the number of parameters of the executed option
      int numberOfOptionParameters = option->parameters.getSize();
      out << (char)numberOfOptionParameters;
      
      for (i = 0; i < numberOfOptionParameters; i++)
      {
        out << option->parameters.getName(i);
        out << option->parameters[i];
      }
      
      option = option->activeState->subsequentOption;
    }
  }
  
  // the executed basic behavior
  const Xabsl2BasicBehavior* bb = pEngine->getSelectedBasicBehavior();
  out << bb->n;
  
  // the number of parameters of the executed basic behavior
  char numberOfBasicBehaviorParameters = bb->parameters.getSize();
  out << numberOfBasicBehaviorParameters;
  
  // sends for each parameter of the executed basic behavior the name and the value
  for (i=0;i<numberOfBasicBehaviorParameters;i++)
  {
    out << bb->parameters.getName(i);
    out << bb->parameters[i];
  }
  
  // send the generated main action as a string
  char buf[200];
  printGeneratedMainActionToString(buf);
  out << buf;
  
  getDebugOut().finishMessage(idXabsl2DebugMessage);
}

bool GTXabsl2EngineExecutor::handleMessage(InMessage& message)
{
  switch (message.getMessageID())
  {
  case idXabsl2DebugRequest:
    {
      // temporary variables
      char buf[100];
      char buf2[100];
      double valueDouble;
      char c;
      int i, numberOfWatchedInputSymbols, numberOfWatchedEnumeratedOutputSymbols;
      
      message.bin >> c;
      
      if ((SolutionRequest::xabsl2EngineID)c != this->id)
      {
        // the request was sent for another engine
        return true;
      }

      // clear the old pointer arrays
      watchedDecimalInputSymbols.clear();
      watchedBooleanInputSymbols.clear();
      watchedEnumeratedInputSymbols.clear();
      watchedEnumeratedOutputSymbols.clear();
      setEnumeratedOutputSymbols.clear();
      setEnumeratedOutputSymbolValues.clear();
      
      // input symbols
      message.bin >> numberOfWatchedInputSymbols;
      
      for (i=0; i< numberOfWatchedInputSymbols; i++)
      {
        message.bin >> c;
        message.bin >> buf;
        switch(c)
        {
        case 'd':
          if (pEngine->existsDecimalInputSymbol(buf))
            watchedDecimalInputSymbols.append(buf,pEngine->getDecimalInputSymbol(buf));
          break;
        case 'b':
          if (pEngine->existsBooleanInputSymbol(buf))
            watchedBooleanInputSymbols.append(buf,pEngine->getBooleanInputSymbol(buf));
          break;
        case 'e':
          if (pEngine->existsEnumeratedInputSymbol(buf))
            watchedEnumeratedInputSymbols.append(buf,pEngine->getEnumeratedInputSymbol(buf));
          break;
        }
      }
      
      // enumerated output symbols
      message.bin >> numberOfWatchedEnumeratedOutputSymbols;
      for (i=0; i< numberOfWatchedEnumeratedOutputSymbols; i++)
      {
        message.bin >> buf;
        if (pEngine->existsEnumeratedOutputSymbol(buf))
          watchedEnumeratedOutputSymbols.append(buf,pEngine->getEnumeratedOutputSymbol(buf));
      }
      // 
      char mode;
      char numberOfParameters;
      message.bin >> mode;
      switch (mode)
      {
      case 'b':
        debugMode = executeBasicBehavior;
        message.bin >> buf;
        if(!pEngine->setSelectedBasicBehavior(buf))
        {
          OUTPUT(idText, text, "BasicBehavior " << buf << " does not exist");
        }
        
        message.bin >> numberOfParameters;
        for(i = 0; i < numberOfParameters; i++)
        {
          message.bin >> buf2;
          message.bin  >> valueDouble;
          pEngine->setBasicBehaviorParameter(buf, buf2, valueDouble);
        }
        break;
      case 'o':
        debugMode = executeOption;
        message.bin >> buf;
        if(!pEngine->setRootOption(buf))
        {
          OUTPUT(idText, text, "Option " << buf << " does not exist");
        }
        message.bin >> numberOfParameters;
        for(i = 0; i < numberOfParameters; i++)
        {
          message.bin >> buf2;
          message.bin  >> valueDouble;
          pEngine->setOptionParameter(buf, buf2, valueDouble);
        }
        break;
      case 'd':
        debugMode = executeRootOption;
        break;
      }
      
      // changed output symbols
      char numberOfChangedOutputSymbols;
      message.bin >> numberOfChangedOutputSymbols;
      for(i = 0; i < numberOfChangedOutputSymbols; i++)
      {
        message.bin >> buf;  //name
        message.bin >> buf2; // value
        if (!pEngine->existsEnumeratedOutputSymbol(buf))
        {
          OUTPUT(idText,text,"Enumerated output symbol " << buf << " does not exist");
        }
        else
        {
          Xabsl2EnumeratedOutputSymbol* pSymbol = pEngine->getEnumeratedOutputSymbol(buf);
          int value = 0;
          
          setEnumeratedOutputSymbols.append(buf,pSymbol);
          
          if (!pSymbol->enumElements.exists(buf2))
          {
            OUTPUT(idText,text,"Enum element " << buf2 << " for symbol " << buf 
              << " does not exist");
          }
          else
          {
            value = pSymbol->enumElements.getElement(buf2)->v;
          }
          setEnumeratedOutputSymbolValues.append(buf,value);
        }
      }
      
      return true;
    }
  case idXabsl2IntermediateCode:
    {
      char c;
      message.bin >> c;
      
      if ((SolutionRequest::xabsl2EngineID)c != this->id)
      {
        // the request was sent for another engine
        return true;
      }

      Xabsl2MessageInputSource input(message.config);
      init(input);
      return true;
    }
  default:
    return false;
  }
}

void GTXabsl2EngineExecutor::setSelectedAgent(const char* name)
{
  if (pEngine != 0) 
    pEngine->setSelectedAgent(name);
}

GTXabsl2ErrorHandler::GTXabsl2ErrorHandler(SolutionRequest::xabsl2EngineID id)
: id (id)
{
}

void GTXabsl2ErrorHandler::printError(const char* text)
{
  OUTPUT(idText, text, "(" 
    << SolutionRequest::getXabsl2EngineIDName(id)
    << " Xabsl2Engine) error: " << text);
}

void GTXabsl2ErrorHandler::printMessage(const char* text)
{
  OUTPUT(idText, text, "(" 
    << SolutionRequest::getXabsl2EngineIDName(id)
    << " Xabsl2Engine): " << text);
}

Xabsl2FileInputSource::Xabsl2FileInputSource(const char* fileName)
: Xabsl2NamedItem(fileName), file(0)
{
}

Xabsl2FileInputSource::~Xabsl2FileInputSource()
{
  if (file!=0) delete file;
}

bool Xabsl2FileInputSource::open()
{
  if (file==0) file=new InConfigFile(n);
  return (file!=0 && file->exists());
}


void Xabsl2FileInputSource::close()
{
  if (file!=0) delete file;
  file = 0;
}

double Xabsl2FileInputSource::readValue()
{
  double d=0;
  if (file!=0) *file >> d;
  return d;
}

bool Xabsl2FileInputSource::readString(char* destination, int maxLength)
{
  if (file!=0) *file >> destination;
  return true;
}

Xabsl2MessageInputSource::Xabsl2MessageInputSource(InConfigMessage& message)
: message(message)
{
}

double Xabsl2MessageInputSource::readValue()
{
  double d;
  message >> d;
  return d;
}

bool Xabsl2MessageInputSource::readString(char* destination, int maxLength)
{
  message >> destination;
  return true;
}


/*
* Change log :
* 
* $Log: GTXabsl2EngineExecutor.cpp,v $
* Revision 1.5  2004/06/24 17:01:39  spranger
* profiler: no log-write-out-attempts when log is empty
*
* Revision 1.4  2004/06/14 17:16:06  spranger
* changed profiler interface
*
* Revision 1.3  2004/05/24 16:33:08  spranger
* bugfix
*
* Revision 1.2  2004/05/23 18:59:06  spranger
* added framenumber to GTXabsl2EngineExecutor for profiler,
* added profiler
*
* Revision 1.1.1.1  2004/05/22 17:37:40  cvsadm
* created new repository GT2004_WM
*
* Revision 1.3  2004/05/18 14:28:21  loetzsch
* now Xabsl2 intermediate code can be sent to different modules
*
* Revision 1.2  2004/05/17 18:35:23  loetzsch
* continued support for multiple Xabsl engines in different modules
*
* Revision 1.1  2004/05/14 11:37:08  loetzsch
* support for multiple xabsl2engines in different modules
* preliminary GT2004HeadControl (does not work at all)
*
*/

