/**
* @file InStreams.cpp
*
* Implementation of in stream classes.
*
* @author Thomas Rfer
* @author Martin Ltzsch
*/

#include "InStreams.h"
#include <string.h>
#include "Platform/GTAssert.h"

void StreamReader::skipData(int size, PhysicalInStream& stream)
{
  // default implementation
  char* dummy = new char[size];
  readData(dummy, size, stream);
  delete [] dummy;
}

void PhysicalInStream::skipInStream(int size)
{
  // default implementation
  char* dummy = new char[size];
  readFromStream(dummy, size);
  delete [] dummy;
}

void InText::readString(char* value, PhysicalInStream& stream)
{
  skipWhitespace(stream);
  while(!isEof(stream) && !isWhitespace())
  {
    if(theChar == '\\')
      nextChar(stream);
    *value++ = theChar;
    if(!isEof(stream))
      nextChar(stream);
  }
  *value = 0;
  skipWhitespace(stream);
}

void InText::readData(void *p,int size, PhysicalInStream& stream)
{
  for(int i = 0; i < size; ++i)
    readChar(*((char*&) p)++,stream);
}

bool InText::isWhitespace()
{
  return theChar == ' ' || theChar == '\n' || theChar == '\r' || theChar == '\t';
}

void InText::skipWhitespace(PhysicalInStream& stream)
{
  while(!isEof(stream) && isWhitespace())
    nextChar(stream);
}

void InConfig::create(const char* sectionName, PhysicalInStream& stream)
{
  if(stream.exists() && sectionName) 
  {
    char fileEntry[200];
    char section[200];
    
    ASSERT(strlen(sectionName) + 2 < sizeof(section));
    strcpy(section,"[");
    strcat(section,sectionName);
    strcat(section,"]");
    
    while(!isEof(stream)) 
    {
      readString(fileEntry,stream);
      if(!strcmp(fileEntry,section)) 
        break;
    }
    readSection = true;
  }
}

bool InConfig::isWhitespace()
{
  return theChar == '/' || theChar == '#' || InText::isWhitespace();
}

void InConfig::skipWhitespace(PhysicalInStream& stream)
{
  while(!isEof(stream) && isWhitespace())
  {
    while(!isEof(stream) && InText::isWhitespace())
      nextChar(stream);
    if(!isEof(stream))
      if(theChar == '/')
      {
        nextChar(stream);
        if(theChar == '/')
          skipLine(stream);
        else if(theChar == '*')
          skipComment(stream);
        else
          ASSERT(false); // "/" used in stream
      }
      else if(theChar == '#')
        skipLine(stream);
  }
}

void InConfig::nextChar(PhysicalInStream& stream) 
{
  InText::nextChar(stream);
  if (readSection && theChar == '[')
    while (!isEof(stream)) 
      InText::nextChar(stream);
}

void InConfig::skipLine(PhysicalInStream& stream)
{
  while(!isEof(stream) && theChar != '\n')
    nextChar(stream);
  if(!isEof(stream))
    nextChar(stream);
}

void InConfig::skipComment(PhysicalInStream& stream)
{
  while(!isEof(stream) && theChar != '/')
  {
    while(!isEof(stream) && theChar != '*')
      nextChar(stream);
    if(!isEof(stream))
      nextChar(stream);
  }
  if(!isEof(stream))
    nextChar(stream);
}

void InMemory::readFromStream(void* p, int size)
{
  if (memory!=0) 
  { 
    memcpy(p,memory,size); 
    memory += size; 
  }
}


/*
* Change Log:
*
* $Log: InStreams.cpp,v $
* Revision 1.1.1.1  2004/05/22 17:37:38  cvsadm
* created new repository GT2004_WM
*
* Revision 1.2  2003/12/30 20:12:04  roefer
* Image size is now 208 x 160. Smaller images are placed in the upper left corner
*
* Revision 1.1  2003/10/07 10:13:24  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:29  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.3  2002/10/11 21:21:16  roefer
* eof() handling in text input corrected
*
* Revision 1.2  2002/09/24 18:41:58  risler
* human readable solutionrequest streaming operator
* default module solutions read from modules.cfg
*
* Revision 1.1  2002/09/10 15:53:59  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.6  2002/08/29 13:48:21  dueffert
* includes in correct case, system includes in <>
*
* Revision 1.5  2002/08/22 14:41:04  risler
* added some doxygen comments
*
* Revision 1.4  2002/08/04 17:53:18  roefer
* SimGT2002 connection to physical robots added
*
* Revision 1.3  2002/07/28 21:24:20  roefer
* EOF-handling improved
*
* Revision 1.2  2002/07/23 16:40:13  roefer
* Router and SimGT2002 adapted to new message queue and streams
*
* Revision 1.1  2002/07/23 13:46:43  loetzsch
* - new streaming classes
*
*/
