/**
* @file RobotConfiguration.h
*
* Definition of class RobotConfiguration
* 
* @author Thomas Rfer
*/
#ifndef __RobotConfiguration_h_
#define __RobotConfiguration_h_

#include "Tools/MessageQueue/InMessage.h"
#include "Platform/SystemCall.h"
#include "Tools/Actorics/RobotDimensions.h"

class RobotConfiguration;

/**
* Returns a reference to a process wide RobotConfiguration configuration.
*/
RobotConfiguration& getRobotConfiguration();

/** 
* @class RobotCalibration
*
* A class that represents the calibration of the robot. 
*
* @author Thomas Rfer
*/
class RobotCalibration
{
public:
  /** the offset between the bodyTilt as delivered by the walking engine and the
  * real bodyTilt of a particular robot. */
  double bodyTiltOffset;

  /** the offset between the bodyRoll as delivered by the walking engine and the
  * real bodyRoll of a particular robot. */
  double bodyRollOffset;

  /** the offset between the headTilt2 as delivered by the walking engine and the
  * real headRoll of a particular robot. */
  double headTiltOffset;

  /** the offset between the headRoll as delivered by the walking engine and the
  * real headRoll of a particular robot. */
  double headRollOffset;

  /** a factor the measurements of the tilt joint are scaled with. */
  double tiltFactor;

  /** a factor the measurements of the pan joint are scaled with. */
  double panFactor;

  /** a factor the measurements of the tilt2 joint are scaled with. */
  double tilt2Factor;

  RobotCalibration()
  {
    bodyTiltOffset = bodyRollOffset = headTiltOffset = headRollOffset = 0;
    tiltFactor = panFactor = tilt2Factor = 1;
  }
};

/** 
* @class RobotConfiguration
*
* A class that represents the configuration of the robot. 
*
* @author Thomas Rfer
*/
class RobotConfiguration : public RobotDesign
{
public:
  /**
  * Constructor.
  */
  RobotConfiguration();

  /**
  * Loads the data from robot.cfg 
  */
  void load();

  /** returns the calibration values of the robot */
  const RobotCalibration& getRobotCalibration() const {return robotCalibration;}

  /** sets the calibration values of the robot */
  void setRobotCalibration(const RobotCalibration& r) {robotCalibration = r;}

  /** returns the mac address of the robot*/
  const char* getMacAddressString() const {return macAddressString;}

  /** returns the design of the robot*/
  Design getRobotDesign() const {return robotDesign;}

  /** returns the dimensions of the robot*/
  const RobotDimensions& getRobotDimensions() const {return robotDimensions;}

  /** updates the robot configuration based on message queue data. */
  bool handleMessage(InMessage& message);

private:
  RobotCalibration robotCalibration; /**< The calibration values of the robot. */
  char macAddressString[13]; /**< The MAC address of the robot. */
  Design robotDesign; /**< The design of the robot. */
  RobotDimensions robotDimensions; /**< The dimensions of the robot. */
};

/**
 * Streaming operator that reads a RobotConfiguration from a stream.
 * @param stream The stream from which is read.
 * @param robotConfiguration The RobotConfiguration object.
 * @return The stream.
 */ 
In& operator>>(In& stream,RobotConfiguration& robotConfiguration);
 
/**
 * Streaming operator that writes a RobotConfiguration to a stream.
 * @param stream The stream to write on.
 * @param robotConfiguration The RobotConfiguration object.
 * @return The stream.
 */ 
Out& operator<<(Out& stream, const RobotConfiguration& robotConfiguration);


#endif //__RobotConfiguration_h_

/*
* Change Log:
*
* $Log: RobotConfiguration.h,v $
* Revision 1.1.1.1  2004/05/22 17:35:55  cvsadm
* created new repository GT2004_WM
*
* Revision 1.7  2004/03/08 02:32:21  roefer
* Calibration parameters changed
*
* Revision 1.6  2004/03/03 18:35:17  roefer
* Acceleration sensors magically work again
*
* Revision 1.5  2004/02/29 14:56:03  roefer
* Additional calibration parameters
*
* Revision 1.4  2004/02/24 19:01:06  roefer
* Additional calibration parameters added
*
* Revision 1.3  2004/01/01 10:58:52  roefer
* RobotDimensions are in a class now
*
* Revision 1.2  2003/12/31 12:21:25  roefer
* getRobotDesign added
*
* Revision 1.1  2003/10/07 10:13:21  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.3  2003/09/25 11:21:59  juengel
* Removed BlobCollection.
*
* Revision 1.2  2003/07/05 09:49:05  roefer
* Generic debug message for bodyOffsets improved
*
* Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.3  2003/05/11 23:56:15  dueffert
* doxygen bugs fixed
*
* Revision 1.2  2003/05/08 14:55:02  juengel
* Added getMacAddressString().
*
* Revision 1.1  2003/05/03 16:20:01  roefer
* robot.cfg added
*
*
*/
