/**
 * @file RingBufferWithSum.h
 *
 * Declaration of class RingBufferWithSum
 *
 * @author Matthias Jngel
 */

#ifndef __RingBufferWithSum_h_
#define __RingBufferWithSum_h_

#include <limits.h>

/**
 * @class RingBufferWithSum
 *
 * template class for cyclic buffering of the last n values of Type int
 * and with a function that returns the sum of all entries
 */
template <int n> class RingBufferWithSum
{
  public:
    /** Constructor */
    RingBufferWithSum() {init();}

    /**
     * initializes the RingBufferWithSum
     */
    void init () {current = n - 1; numberOfEntries = 0; sum = 0;}

    /**
     * adds an entry to the buffer
     * \param value value to be added
     */
    void add (int value) 
    {
      if(numberOfEntries == n) sum -= getEntry(numberOfEntries - 1);
      sum += value;
      current++;
      if (current==n) current=0;
      if (++numberOfEntries >= n) numberOfEntries = n;
      buffer[current] = value;
    }

    /**
     * returns an entry
     * \param i index of entry counting from last added (last=0,...)
     * \return a reference to the buffer entry
     */
    int getEntry (int i)
    {
      int j = current - i;
      j %= n;
      if (j < 0) j += n;
      return buffer[j];
    }

    int getSum()
    {
      return sum;
    }

    int getMinimum()
    {
      int min = INT_MAX;
      for(int i = 0; i < numberOfEntries;i++)
      {
        if(buffer[i] < min) min = buffer[i];
      }
      return min;
    }

    /**
     * returns an entry
     * \param i index of entry counting from last added (last=0,...)
     * \return a reference to the buffer entry
     */
    int operator[] (int i)
    {
      return getEntry(i);
    }

    /**
     * returns a constant entry.
     * \param i index of entry counting from last added (last=0,...)
     * \return a reference to the buffer entry
     */
    int operator[] (int i) const
    {
      return buffer[i > current ? n + current - i : current - i];
    }

    int getNumberOfEntries() const
    {
      return numberOfEntries;
    }

  private:
    int current;
    int numberOfEntries;
    int buffer[n];

    int sum;
};

#endif // __RingBufferWithSum_h_

/*
 * Change log :
 * 
 * $Log: RingBufferWithSum.h,v $
 * Revision 1.1.1.1  2004/05/22 17:35:55  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.2  2003/10/29 13:17:16  juengel
 * bug fixed
 *
 * Revision 1.1  2003/10/07 10:13:21  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.2  2003/08/08 11:37:30  dueffert
 * doxygen docu corrected
 *
 * Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.3  2003/06/27 15:09:14  juengel
 * Added getMinimum().
 *
 * Revision 1.2  2003/06/21 09:48:36  dueffert
 * useless const warning removed
 *
 * Revision 1.1  2003/06/20 17:56:21  juengel
 * Added RingBufferWithSum.
 *
 */
