/**
* @file RandomMotionGenerator.h
* 
* Definition of class RandomMotionGenerator
*
* @author <a href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</a>
*/

#ifndef RANDOM_MOTION_GENERATOR_H_
#define RANDOM_MOTION_GENERATOR_H_


#include "PfieldDatatypes.h"
#include "Pfield.h"


/**
* @class RandomMotionGenerator
*
* An class for generating random motion vectors
*/
class RandomMotionGenerator
{
public:
  /** Constructor 
  * @param minValue See class members
  * @param maxValue See class members
  * @param valueDx See class members
  * @param directionDx See class members
  * @param changeType See class members
  * @param n See class members
  */
  RandomMotionGenerator(double minValue, double maxValue, double valueDx,
                        double directionDx, ChangeType changeType, unsigned long n);

  /** Returns a random motion vector
  * @return The vector
  */
  PfVec getMotionVector();

private:
  /** The minimum length of the random vector*/
  double minValue;
  /** The maximum length of the random vector*/
  double maxValue;
  /** The maximum difference of the length of two successively generated vector*/
  double valueDx;
  /** The maximum difference of the direction of two successively generated vector*/
  double directionDx;
  /** The unit of n*/
  ChangeType changeType;
  /** The number of steps to wait until a new vector is generated*/
  unsigned long n;
  /** The last generated vector*/
  PfVec lastVec;
  /** The last generated direction*/
  double lastDirection;
  /** The number of calls since the last vector was generated*/
  unsigned long calls;
  /** The point of time the last vector was generated*/
  unsigned long pointOfGenerationTime;

  /** Returns a direction
  * @param previousDirection The previous direction in radian
  * @return A direction in radian
  */
  double computeDirection(double previousDirection) const;

  /** Returns a length
  * @param previousLength The previous length
  * @return A length
  */
  double computeVecLength(double previousLength) const;

  /** Returns a random number between two numbers
  * @param min The minimum value
  * @param max The maximum value
  * @return A random number between min and max
  */
  double getRandomNumberBetween(double min, double max) const;
};


#endif	//RANDOM_MOTION_GENERATOR_H_



/*
* $Log: RandomMotionGenerator.h,v $
* Revision 1.1.1.1  2004/05/22 17:37:36  cvsadm
* created new repository GT2004_WM
*
* Revision 1.1  2004/01/20 15:42:19  tim
* Added potential fields implementation
*
* Revision 1.1  2003/06/13 14:27:58  tim
* added random generator and tangential fields
*
*/
