/**
* @file RandomMotionGenerator.cpp
* 
* Implementation of class RandomMotionGenerator
*
* @author <a href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</a>
*/

#include <cstdlib>
#include "RandomMotionGenerator.h"
#include "PfieldConfig.h"



RandomMotionGenerator::RandomMotionGenerator
                       (double minValue, double maxValue, double valueDx,
                        double directionDx, ChangeType changeType, unsigned long n)
{
  this->minValue = minValue;
  this->maxValue = maxValue;
  this->valueDx = valueDx;
  this->directionDx = directionDx;
  this->changeType = changeType;
  this->n = n; 
  // Initialize random generator 
  srand (getSystemTime());
  // Set initial vector
  lastVec.x = computeVecLength(minValue + ((maxValue-minValue) / 2.0));
  lastVec.y = 0.0;
  lastDirection = getRandomNumberBetween(-pi,pi);
  lastVec.rotate(lastDirection);
  pointOfGenerationTime = getSystemTime();
}


PfVec RandomMotionGenerator::getMotionVector()
{
  calls++;
  if((changeType == CALLS) && (calls < n))
  {
    return lastVec;
  }
  else if((changeType == MILLISECONDS) && 
          ((getSystemTime() - pointOfGenerationTime) < n))
  {
    return lastVec;
  }
  else
  {
    PfVec result(computeVecLength(lastVec.length()),0.0);
    lastDirection = computeDirection(lastDirection);
    result.rotate(lastDirection);
    lastVec = result;
    pointOfGenerationTime = getSystemTime();
    calls = 0;
    return result;
  }
}


inline double RandomMotionGenerator::computeDirection(double previousDirection) const
{
  double maxVal(previousDirection + directionDx);
  double minVal(previousDirection - directionDx);
  double newDirection(getRandomNumberBetween(minVal,maxVal));
  while(newDirection > pi) newDirection-=pi2;
  while(newDirection < -pi) newDirection+=pi2;
  return newDirection;
}


inline double RandomMotionGenerator::computeVecLength(double previousLength) const
{
  if(minValue == maxValue)
  {
    return previousLength;
  }
  else
  {
    double minVal(previousLength - valueDx);
    double maxVal(previousLength + valueDx);
    if(minVal < minValue) minVal = minValue;
    if(maxVal > maxValue) maxVal = maxValue;
    return getRandomNumberBetween(minVal, maxVal);
  }
}


inline double RandomMotionGenerator::getRandomNumberBetween(double min, double max) const
{
  double range(max - min);
  double scalar(((double)rand()) / ((double)RAND_MAX));
  return (min + scalar*range);
}



/*
* $Log: RandomMotionGenerator.cpp,v $
* Revision 1.1.1.1  2004/05/22 17:37:36  cvsadm
* created new repository GT2004_WM
*
* Revision 1.1  2004/01/20 15:42:19  tim
* Added potential fields implementation
*
* Revision 1.1  2003/06/13 14:27:58  tim
* added random generator and tangential fields
*
*/
