/**
* @file PotentialfieldComposition.h
* 
* Definition of class PotentialfieldComposition
*
* @author <a href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</a>
*/

#ifndef POTENTIALFIELDCOMPOSITION_H_
#define POTENTIALFIELDCOMPOSITION_H_


#include "PfieldConfig.h"
#include "PfieldDatatypes.h"
#include <vector>
#include <string>
#include <list>
#include <map>

class Object;
class Parser;
class Potentialfield;


/** A type for describing the selection procedure of the next result*/
enum SelectionProcedure {BEST_OF_N, SUCCESSIVE_N_TIMES};

/** A type for mapping names to indizes in vectors*/
typedef std::map<std::string, unsigned int> NameToIndexMap;


/**
* @class PotentialfieldResult
*
* Describes the result of a potential field
*/
class PotentialfieldResult
{
public:
  /** The validity of the result*/
  double value;
  /** The name of the action*/
  std::string action;
  /** Additional information about the action*/
  std::string subAction;
  /** A motion, not used in all results*/
  PfPose motion;
  /** Flag: true, if the result describes a possible action*/
  bool actionPossible;
  /** The number of the selected field (for internal purposes)*/
  unsigned int fieldNumber;
  /** The point of time, this result was generated*/
  unsigned long timeStamp;

  /** Copy operator
  * @param other Another PotentialfieldResult
  */
  void operator = (const PotentialfieldResult& other)
  {
    value = other.value;
    action = other.action;
    subAction = other.subAction;
    motion = other.motion;
    actionPossible = other.actionPossible;
    fieldNumber = other.fieldNumber;
    timeStamp = other.timeStamp;
  }
};


/**
* @class ObjectStateDescription
*
* Describes the state of dynamic objects in a potential field
*/
class ObjectStateDescription
{
public:
  /** The name of the object state symbol*/
  std::string objectName;
  /** The internal identifier*/
  int objectId;
  /** The pose of the object*/
  PfPose pose;
  /** Flag: true, if the object is active*/
  bool isActive;

  /** Constructor */
  ObjectStateDescription()
  {}

  /** Copy operator
  * @param other Another ObjectStateDescription
  */
  void operator = (const ObjectStateDescription& other)
  {
    objectName = other.objectName;
    objectId = other.objectId;
    pose = other.pose;
    isActive = other.isActive;
  }

  /** Copy-Constructor
  * @param other Another ObjectStateDescription
  */
  ObjectStateDescription(const ObjectStateDescription& other)
  {
    objectName = other.objectName;
    objectId = other.objectId;
    pose = other.pose;
    isActive = other.isActive;
  }
};


/**
* @class PotentialfieldComposition
*
* The main class for the potential field implementation.
* It contains all objects and fields, executes the fields and selects
* the best action.
*/
class PotentialfieldComposition
{
public:
  /** Constructor */
  PotentialfieldComposition();

  /** Destructor */
  ~PotentialfieldComposition();

  /** Loads a configuration file
  * @param filename The name of the file
  */
  void load(const std::string& filename);

  /** Destroys all objects and fields*/
  void close();

  /** Executes computation
  * @param result Returns the result
  */
  void execute(PotentialfieldResult& result);

  /** Sets the name
  * @param name The name
  */
  void setName(const std::string& name)
  { this->name = name;}

  /** Returns the name
  * @return The name
  */
  std::string getName() const
  { return name;}

  /** Sets the state of one object (may set desc.objectId)
  * @param desc The description of the object and its new state
  */
  void setObjectState(ObjectStateDescription& desc);

  /** Sets the own pose
  * @param pose The pose
  */
  void setOwnPose(const PfPose& pose);

  /** Adds a new field to the composition
  * @param field The field
  */
  void addField(Potentialfield* field);

  /** Adds a new object to the composition
  * @param object The object
  */
  void addObject(Object* object);

  /** Adds a new symbol for external object states
  * @param objectName The name of the symbol
  */
  void addDynamicObjectState(const std::string& objectName);

  /** Returns the internal id of a symbol
  * @param objectName The name of the symbol
  * @return The identifier
  */
  unsigned int getIdFromObjectStateSymbol(const std::string& objectName);

  /** Returns a object description
  * @param objectId The object identifier
  * @return The description
  */
  ObjectStateDescription getDescriptionFromId(unsigned int objectId);

  /** Returns a list of the names of all fields
  * @return A vector containing the names
  */
  std::vector<std::string> getFieldNames();

  /** Sets parameters for field selection
  * @param selectionProcedure The selection mechanism
  * @param n A parameter for selectionProcedure
  */
  void setFieldSelectionParameters(SelectionProcedure selectionProcedure, 
                                   unsigned int n)
  { this->selectionProcedure = selectionProcedure; this->n = n;}

  /** Computes an array of field values, used by visualization
  * @param fieldname The name of the potential field
  * @param x1 Minimum x-coordinate to compute
  * @param y1 Minimum y-coordinate to compute
  * @param x2 Maximum x-coordinate to compute
  * @param y2 Maximum y-coordinate to compute
  * @param xSteps The computation steps in x-direction
  * @param ySteps The computation steps in y-direction
  * @param value An array containing all computed values, memory has to be allocated BEFORE calling the function
  * @param max Returns the maximum value
  */
  void getValueArray(const std::string& fieldname, double x1, double y1,
                     double x2, double y2, int xSteps, int ySteps, double value[], double& max);

  /** Computes an array of gradient directions, used by visualization
  * @param fieldname The name of the potential field
  * @param x1 Minimum x-coordinate to compute
  * @param y1 Minimum y-coordinate to compute
  * @param x2 Maximum x-coordinate to compute
  * @param y2 Maximum y-coordinate to compute
  * @param xSteps The computation steps in x-direction
  * @param ySteps The computation steps in y-direction
  * @param directions An array containing all computed gradients, memory has to be allocated BEFORE calling the function
  */
  void getDirectionArray(const std::string& fieldname, double x1, double y1,
                         double x2, double y2, int xSteps, int ySteps, PfVec directions[]);

  /** Activates or deactivates a field
  * @param fieldname The name of the field
  * @param activation true, if the field is to be activated; false, otherwise
  */
  void setFieldActivation(const std::string& fieldname, bool activation);

  /** The PotentialfieldComposition is a friend of its parser*/
  friend class Parser;

private:
  /** The name of the composition*/
  std::string name;
  /** A list of all objects */
  std::vector<Object*> objects;
  /** Fast mapping from object names to indizes*/
  NameToIndexMap objectMap;
  /** The states of the dynamic objects*/
  std::vector<ObjectStateDescription> dynamicObjectStates;
  /** Fast mapping from object state names to indizes*/
  NameToIndexMap objectStateMap;
  /** A list of all fields */ 
  std::vector<Potentialfield*> fields;
  /** Fast mapping from field names to indizes*/
  NameToIndexMap fieldMap;
  /** A list for the results of all fields*/
  std::vector<PotentialfieldResult> results;
  /** A list for the selected results of the last runs*/
  std::list<PotentialfieldResult> resultList;
  /** The result of the last run*/
  PotentialfieldResult lastResult;
  /** The own pose*/
  PfPose ownPose;
  /** The parser*/
  Parser* parser;
  /** A flag*/
  bool fileLoaded;
  /** The procedure to choose the next active field*/
  SelectionProcedure selectionProcedure;
  /** A parameter for the selectionProcedure*/
  unsigned int n;

  /** Selects the next result from the list
  * @return The result
  */
  PotentialfieldResult selectNextResult();

  /** Adds a result to the list and keeps the size of the list <= n
  * @param result The result
  */
  void addResultToList(const PotentialfieldResult& result);

  /** Updates the state of all dynamic objects*/
  void updateDynamicObjects();

  /** Returns the index of a field
  * @param fieldname The name of the field
  * @return The index
  */
  unsigned int getFieldIndexFromName(const std::string& fieldname);
};


#endif	//POTENTIALFIELDCOMPOSITION_H_



/*
* $Log: PotentialfieldComposition.h,v $
* Revision 1.1.1.1  2004/05/22 17:37:33  cvsadm
* created new repository GT2004_WM
*
* Revision 1.1  2004/01/20 15:42:19  tim
* Added potential fields implementation
*
* Revision 1.5  2003/06/09 20:00:04  tim
* Changed potentialfield architecture
*
* Revision 1.4  2003/03/30 15:32:09  tim
* several minor changes
*
* Revision 1.3  2003/03/25 15:37:59  timrie
* Doxygen-comments corrected
*
* Revision 1.2  2003/03/23 20:32:37  loetzsch
* removed green compiler warning: no newline at end of file
*
* Revision 1.1  2003/03/23 17:51:27  tim
* Added potentialfields
*
*/
