/**
* @file Pfield.h
* 
* Definition of class Potentialfield
* This file was originally named Potentialfield.h but had to be renamed
* because of another Potentialfield.h in this project.
*
* @author <a href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</a>
*/

#ifndef PFIELD_H_
#define PFIELD_H_


#include "PfieldConfig.h"
#include <vector>
#include <string>
#include "PotentialfieldComposition.h"

class Object;
class Parser;
class PfPose;
class PfVec;


/** The criterions for computing the value of a field result*/
enum Criterion {CRITERION_GAIN, CRITERION_ABSOLUTE, CRITERION_GRADIENT,
                CRITERION_CONST};

/** A datatype for different time constraints*/
enum ChangeType {MILLISECONDS, CALLS};

/** A datatype describing to what to keep: a result or a field*/
enum KeepType {KEEP_RESULT, KEEP_FIELD};

/** A type to distinguish between the two different field types*/
enum BehaviorFieldType {ACTION_FIELD, MOTION_FIELD};

/**
* @class Potentialfield
*
* An abstract base class for all fields
*/
class Potentialfield
{
public:
  /** Constructor */
  Potentialfield();

  /** Destructor */
  virtual ~Potentialfield();

  /** Initializes values and / or allocates additional memory for subsequent computations.
      This function should be called after all values are set and all objects have 
      been assigned */
  virtual void init() {}

  /** Computes the result of a field
  * @param pose The pose on which the field affects
  * @param result The returned result
  */
  void getResult(const PfPose& pose, PotentialfieldResult& result);

  /** Computes an array of field values, used by visualization
  * @param x1 Minimum x-coordinate to compute
  * @param y1 Minimum y-coordinate to compute
  * @param x2 Maximum x-coordinate to compute
  * @param y2 Maximum y-coordinate to compute
  * @param xSteps The computation steps in x-direction
  * @param ySteps The computation steps in y-direction
  * @param value An array containing all computed values, memory has to be allocated BEFORE calling the function
  * @param max Returns the maximum value
  */
  void getValueArray(double x1, double y1, double x2, double y2,
                     int xSteps, int ySteps, double value[], double& max);

  /** Computes an array of gradient directions, used by visualization
  * @param x1 Minimum x-coordinate to compute
  * @param y1 Minimum y-coordinate to compute
  * @param x2 Maximum x-coordinate to compute
  * @param y2 Maximum y-coordinate to compute
  * @param xSteps The computation steps in x-direction
  * @param ySteps The computation steps in y-direction
  * @param directions An array containing all computed gradients, memory has to be allocated BEFORE calling the function
  */
  void getDirectionArray(double x1, double y1, double x2, double y2,
                         int xSteps, int ySteps, PfVec directions[]);

  /** Computes the gradient at a given position
  * @param pose The pose
  * @return The gradient
  */
  PfVec getFieldVecAt(const PfPose& pose);

  /** Computes the field value at a given position
  * @param pose The pose
  * @return The field value
  */
  double getFieldValueAt(const PfPose& pose);

  /** Adds an object to the field
  * @param object The object
  */
  virtual void addObject(Object* object)
  { objects.push_back(object);}

  /** Returns the name of the field
  * @return The name
  */
  std::string getName() const
  { return name;}

  /** Returns the type of the field
  * @return The type
  */
  virtual BehaviorFieldType getBehaviorFieldType() const = 0;

  /** Sets the time constraints
  * @param whatToKeep Describes what to keep: result or field
  * @param keepForN Keep for n CALLS or MILLISECONDS
  * @param n The value corresponding to keepForN
  * @param blockForM Block for m CALLS or MILLISECONDS
  * @param m The value corresponding to blockForM
  * @param keepMaxForO Keep this field at most o CALLS or MILLISECONDS
  * @param o The value corresponding to blockForO
  */
  void setTimeConstraints(KeepType whatToKeep, ChangeType keepForN, int n,
                          ChangeType blockForM, int m, ChangeType keepMaxForO, int o)
  {
    this->whatToKeep = whatToKeep; 
    this->keepForN = keepForN; 
    this->n = n;
    this->blockForM = blockForM; 
    this->m = m; 
    this->keepMaxForO = keepMaxForO; 
    this->o = o;
  }

  /** Sets the validity criterion
  * @param criterion The criterion
  * @param value An additional parameter (for criterion == CRITERION_CONST)
  */
  void setCriterion(Criterion criterion, double value=0)
  { this->criterion = criterion; criterionParameter = value;}

  /** Returns the names of the fields, this field is combined with
  * @return A list of field names
  */
  std::vector<std::string> getCombinedFields() const
  { return combinedFields;}
  
  /** Adds the name of a field, this field is combined with
  * @param fieldName The name of the field
  */
  void addCombinedField(const std::string& fieldName)
  { combinedFields.push_back(fieldName);}

  /** Returns whether the field is combined with other fields or not
  * @return true, if the field is combined with at least one other field
  */
  bool isCombined()
  { return (combinedFields.size()>0);}

  /** Gives the field information about being selected or not
  * @param selected True, if the field has been selected in the current run
  */
  void setSelectionFeedback(bool selected);

  /** Checks, if the field is still blocking other fields
  * @return true, if the field has to remain active
  */
  bool hasToRemainActive();

  /** Activates and / or deactivates the field
  * @param isActive Deactivates the field, if false.
  */
  void setActivation(bool isActive)
  {this->isActive = isActive;}

protected:
  /** The objects assigned to this field*/
  std::vector<Object*> objects;
  /** The name of the field*/
  std::string name;
  /** The criterion for result computation*/
  Criterion criterion;
  /** An additional parameter for criterion*/
  double criterionParameter;
  /** The names of the combined fields*/
  std::vector<std::string> combinedFields;
  /** The last result computed by this field*/
  PotentialfieldResult lastResult;
  /** Describes what to keep: result or field*/
  KeepType whatToKeep;
  /** Keep for n CALLS or MILLISECONDS */
  ChangeType keepForN;
  /** The value corresponding to keepForN */
  unsigned int n;
  /** Block for m CALLS or MILLISECONDS */
  ChangeType blockForM;
  /** The value corresponding to blockForM */
  unsigned int m;
  /** Keep this field at most for o CALLS or MILLISECONDS*/
  ChangeType keepMaxForO;
  /** The value corresponding to keepMaxForO */
  int o;
  /** Describes, if the field has been selected last turn*/
  bool currentlySelected;
  /** Describes how long this field has been selected*/
  unsigned long selectedSince;
  /** Describes how often this field has been selected*/
  unsigned long selectedCalls;
  /** Describes whether calls to be blocked or the point of time until the block is active*/
  unsigned long block;
  /** Flag: True, if the field is active and executable*/
  bool isActive;

  /** Computes the result of a field
  * @param pose The pose on which the field affects
  * @param result The returned result
  */
  virtual void execute(const PfPose& pose, PotentialfieldResult& result) = 0;

  /** Checks if the field is blocked and cannot compute a result
  * @return true, if the field is blocked
  */
  bool isBlocked();

  /** Checks if the field has to be blocked in future and sets 
  * a block if necessary
  */
  void checkForBlockAppliance();
};


#endif	//PFIELD_H_



/*
* $Log: Pfield.h,v $
* Revision 1.2  2004/06/07 18:19:39  tim
* removed dynamic_cast and RTTI
*
* Revision 1.1.1.1  2004/05/22 17:37:31  cvsadm
* created new repository GT2004_WM
*
* Revision 1.1  2004/01/20 15:42:19  tim
* Added potential fields implementation
*
* Revision 1.6  2003/06/09 20:00:04  tim
* Changed potentialfield architecture
*
* Revision 1.5  2003/05/08 15:26:06  tim
* no message
*
* Revision 1.4  2003/04/02 16:32:28  dueffert
* warning removed
*
* Revision 1.3  2003/04/02 14:39:12  tim
* Changed Bremen Byters Behavior
*
* Revision 1.2  2003/03/23 20:32:37  loetzsch
* removed green compiler warning: no newline at end of file
*
* Revision 1.1  2003/03/23 17:51:27  tim
* Added potentialfields
*
*/
