/**
* @file PfcParser.h
* 
* Definition of class Parser
* This parser is used to parse pfc-files for potential fields
*
* @author <a href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</a>
*/

#ifndef PFC_PARSER_H_
#define PFC_PARSER_H_

#include "PfieldConfig.h"
#include <string>
#include <vector>
#ifdef POTENTIALFIELDS_FOR_GT2004_
#include "Tools/Streams/InStreams.h"  
#else
#include <fstream>
#endif //POTENTIALFIELDS_FOR_GT2004_
#include "PotentialfieldComposition.h"

class Object;
class FormationObject;
class SingleFormation;
class Action;
class Potentialfield;
class PotentialfieldTransformation;
class PotentialfieldFunction;
class PfPose;
class PfVec;
class PfieldGeometricObject;


/**
* @class InstanceGroup
*
* A container for temporary storage of 
* a group of object instances
*/
class InstanceGroup
{
public:
  /** The name of the group*/
  std::string name;
  /** The objects belonging to the group*/
  std::vector<Object*> objects;
};


/**
* @class Parser
*
* The parser for PotentialfieldComposition
*/
class Parser
{
public:
  /** Destructor */
  ~Parser();

  /** Executes the parser
  * @param composition The calling PotentialfieldComposition
  * @param filename The file to open and parse
  */
  void parse(PotentialfieldComposition* composition, std::string filename);

private:
#ifdef POTENTIALFIELDS_FOR_GT2004_
  /** A file object */
  InBinaryFile* file;
#else
  /** A file object */
  std::ifstream file;
#endif //POTENTIALFIELDS_FOR_GT2004_
  /** The last parsed token*/
  std::string currentToken;
  /** Parsed objects*/
  std::vector<Object*> objects;
  /** Fast mapping from name to index*/
  NameToIndexMap objectMap;
  /** The cached instance groups*/
  std::vector<InstanceGroup> instanceGroups;
  /** The calling PotentialfieldComposition*/
  PotentialfieldComposition* composition;

  /** Returns the next token from file
  * @return A Token
  */
  std::string nextToken();

  /** Returns one character from file
  * @return A character
  */
  char nextChar();

  /** Tests for the end of the file
  * @return true, if the end of the file has been reached
  */
  bool endOfFile();

  /** Parses an angle
  * @return The angle in radian
  */
  double parseAngle();

  /** Parses parameters for PotentialfieldComposition*/
  void parseComposition();
  
  /** Parses an object
  * @return A pointer to the parsed object
  */
  Object* parseObject();

  /** Parses an object and inserts it in the object list */
  void parseAndInsertObject();

  /** Parses an ObjectStateSymbol*/
  void parseObjectStateSymbol();

  /** Parses a group of instances*/
  void parseInstanceGroup();

  /** Parses the instance of an object*/
  void parseInstance();

  /** Parses a formation object*/
  void parseFormationObject();

  /** Parses and creates single formation rule for a formation
  * @param formation The single formation
  * @param formationToken Tehe first token to parse
  */
  void parseSingleFormation(SingleFormation*& formation, const std::string& formationToken);

  /** Parses a Motionfield*/
  void parseMotionfield();

  /** Parses and creates a transformation
  * @param transformation The transformation
  * @param typeToken The first token (indicates the type)
  */
  void parseTransformation(PotentialfieldTransformation*& transformation,
                           const std::string& typeToken);

  /** Parses an Action for an Actionfield
  * @param action The action
  */
  void parseAction(Action& action);

  /** Parses an Actionfield*/
  void parseActionfield();

  /** Parses and creates a function
  * @param function The function to parse
  */
  void parseFunction(PotentialfieldFunction*& function);

  /** Parses and creates a geometric object
  * @param geometricObject A pointer to a geometric object
  * @param preparsedToken The first token to parse
  */
  void parseGeometricObject(PfieldGeometricObject*& geometricObject,
                            const std::string& preparsedToken = "");

  /** Parses a list of object references
  * @param nameOfFirst The name of the first object
  * @param field The field to which the objects will be assigned
  */
  void parseObjectsForField(const std::string& nameOfFirst, Potentialfield* field);

  /** Parses some timing variables for a field
  * @param field The field
  */
  void parseTimeConstraintsForField(Potentialfield* field);

  /** Adds the elements of an object group to a field
  * @param groupName The name of the group
  * @param field The field to which the group is assigned
  */
  void addGroupToField(const std::string& groupName, Potentialfield* field);

  /** Finds an object given its name
  * @param name The name
  * @return A pointer to the object
  */
  Object* getObject(const std::string& name);

  /** Finds an object given its name and creates an instance
  * @param name The name
  * @return A pointer to the object
  */
  Object* getInstance(const std::string& name);
};


#endif	//PFC_PARSER_H_



/*
* $Log: PfcParser.h,v $
* Revision 1.1.1.1  2004/05/22 17:37:29  cvsadm
* created new repository GT2004_WM
*
* Revision 1.1  2004/01/20 15:42:19  tim
* Added potential fields implementation
*
* Revision 1.7  2003/06/13 14:27:58  tim
* added random generator and tangential fields
*
* Revision 1.6  2003/05/22 14:23:47  tim
* Changed representation of transformations
*
* Revision 1.5  2003/04/22 14:35:17  tim
* Merged changes from GO
*
* Revision 1.5  2003/04/09 19:03:06  tim
* Last commit before GermanOpen
*
* Revision 1.4  2003/04/03 15:47:32  tim
* Added modelling for teammates
*
* Revision 1.3  2003/03/25 15:37:59  timrie
* Doxygen-comments corrected
*
* Revision 1.2  2003/03/23 20:32:37  loetzsch
* removed green compiler warning: no newline at end of file
*
* Revision 1.1  2003/03/23 17:51:27  tim
* Added potentialfields
*
*/
