/**
* @file GTStandardConverter.h
* 
* This file contains a class which wraps the GT2004-WorldModel
* to the representation of the PotentialfieldComposition.
*
* @author <a href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</a>
*/

#ifndef GT_STANDARD_CONVERTER_H_
#define GT_STANDARD_CONVERTER_H_


#include "PotentialfieldComposition.h"

class BallModel;
class RobotPose;
class PlayerPoseCollection;
class ObstaclesModel;
class TeamMessageCollection;
class Pose2D;
class string;


/**
* @class GTStandardConverter
*
* A class to use the PotentialfieldComposition inside the GT2004 architecture
*/
class GTStandardConverter
{
public:
  /** 
  * Constructor.
  * @param fileName The name of a file containing a configuration for a PotentialFieldComposition.
  */
  GTStandardConverter(const char* fileName);
  
  /** Constructor */
  GTStandardConverter();

  /** Loads a configuration file
  * @param filename The name of the file
  */
  void load(const char* filename);

  /** Calls the potential fields with data from the world model and returns an action 
  * @param robotPose The robot's pose
  * @param ballPosition The position of the ball
  * @param playerPoseCollection The positions of all other robots
  * @param obstaclesModel Other obstacles
  * @param result Returns the result
  */
  void execute(const RobotPose& robotPose,
               const BallModel& ballPosition,
               const PlayerPoseCollection& playerPoseCollection,
               const ObstaclesModel& obstaclesModel,
               const TeamMessageCollection& teamMessageCollection,
               PotentialfieldResult& result);

  /** Calls the potential fields with data from the world model and returns an action 
  * @param robotPose The robot's pose
  * @param ballPosition The position of the ball
  * @param playerPoseCollection The positions of all other robots
  * @param obstaclesModel Other obstacles
  * @param specialDestination1 A destination, e.g. for go-to-point skills
  * @param specialDestination2 A second destination
  * @param numberOfSpecialDestinations The number of used special destinations 0,1 or 2
  * @param result Returns the result
  */
  void execute(const RobotPose& robotPose,
               const BallModel& ballPosition,
               const PlayerPoseCollection& playerPoseCollection,
               const ObstaclesModel& obstaclesModel,
               const TeamMessageCollection& teamMessageCollection,
               const Pose2D& specialDestination1,
               const Pose2D& specialDestination2,
               const int& numberOfSpecialDestinations,
               PotentialfieldResult& result);

  /** Computes an array of field values, used by visualization
  * @param robotPose The pose of the robot
  * @param ballPosition The position of the ball
  * @param playerPoseCollection The positions of all other robots
  * @param obstaclesModel The obstacles
  * @param fieldname The name of the potential field
  * @param x1 Minimum x-coordinate to compute
  * @param y1 Minimum y-coordinate to compute
  * @param x2 Maximum x-coordinate to compute
  * @param y2 Maximum y-coordinate to compute
  * @param xSteps The computation steps in x-direction
  * @param ySteps The computation steps in y-direction
  * @param value An array containing all computed values, memory has to be allocated BEFORE calling the function
  * @param max Returns the maximum value
  */
  void getValueArray(const RobotPose& robotPose,
                     const BallModel& ballPosition,
                     const PlayerPoseCollection& playerPoseCollection, 
                     const ObstaclesModel& obstaclesModel,
                     const TeamMessageCollection& teamMessageCollection,
                     const std::string& fieldname, double x1, double y1,
                     double x2, double y2, int xSteps, int ySteps, double value[], double& max);

  /** Computes an array of gradient directions, used by visualization
  * @param robotPose The pose of the robot
  * @param ballPosition The position of the ball
  * @param playerPoseCollection The positions of all other robots
  * @param obstaclesModel The obstacles
  * @param fieldname The name of the potential field
  * @param x1 Minimum x-coordinate to compute
  * @param y1 Minimum y-coordinate to compute
  * @param x2 Maximum x-coordinate to compute
  * @param y2 Maximum y-coordinate to compute
  * @param xSteps The computation steps in x-direction
  * @param ySteps The computation steps in y-direction
  * @param directions An array containing all computed gradients, memory has to be allocated BEFORE calling the function
  */
  void getDirectionArray(const RobotPose& robotPose,
                         const BallModel& ballPosition,
                         const PlayerPoseCollection& playerPoseCollection,
                         const ObstaclesModel& obstaclesModel,
                         const TeamMessageCollection& teamMessageCollection,
                         const std::string& fieldname, double x1, double y1,
                         double x2, double y2, int xSteps, int ySteps, PfVec directions[]);


  /** Returns the names of all fields within the potential field composition
  * @param fieldNames The names
  */
  void getFieldNames(std::vector<std::string>& fieldNames);

private:
  /** An instance of the potential field implementation*/
  PotentialfieldComposition potentialfields;
  /** Information about the ball*/
  ObjectStateDescription seenBallDesc, ballDesc;
  /** Information about the robot, needed by some behaviors*/
  ObjectStateDescription selfDesc;
  /** Information about the own team*/
  ObjectStateDescription goalieDesc, player2Desc, player3Desc, player4Desc;
  ObjectStateDescription unknownTeammate1Desc, unknownTeammate2Desc, 
                         unknownTeammate3Desc, unknownTeammate4Desc;
  /** Information about the other team*/
  ObjectStateDescription opponent1Desc, opponent2Desc, opponent3Desc, 
                         opponent4Desc, opponent5Desc, opponent6Desc;
  /** Information about special destinations*/
  ObjectStateDescription destination1Desc, destination2Desc;
  /** Information about obstacles*/
  ObjectStateDescription obstacleFrontDesc, obstacleLeftDesc, obstacleRightDesc, 
                         obstacleFrontLeftDesc, obstacleFrontRightDesc;
  /** Information about the free part of the goal*/
  ObjectStateDescription freeGoalDesc;
  /** Position to look be looked at by goalie*/
  ObjectStateDescription goalieLookingPositionDesc;
  /** The goalie position*/
  ObjectStateDescription goaliePositionDesc;
  /** Angles to obstacles*/
  double angleToFrontObstacle, angleToLeftObstacle, angleToRightObstacle,
         angleToFrontLeftObstacle, angleToFrontRightObstacle;
  /** Opening angle of an obstacle sector*/
  double openingAngle;

  /** Updates the potentialfields with data from the world model
  * @param robotPose The robot's pose
  * @param ballPosition The position of the ball
  * @param playerPoseCollection The positions of all other robots
  * @param obstaclesModel The obstacles
  */
  void setData(const RobotPose& robotPose,
               const BallModel& ballPosition,
               const PlayerPoseCollection& playerPoseCollection,
               const ObstaclesModel& obstaclesModel,
               const TeamMessageCollection& teamMessageCollection);
};


#endif  //GT_STANDARD_CONVERTER_H_


/*
* $Log: GTStandardConverter.h,v $
* Revision 1.2  2004/06/17 16:20:30  tim
* improved ;-)
*
* Revision 1.1  2004/05/24 13:45:10  tim
* added potential field viewer again
*
*/
