/**
* @file GTStandardConverter.cpp
* 
* This file contains a class which wraps the GT2003-WorldModel
* to the represantation of the PotentialfieldComposition.
*
* @author <a href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</a>
*/

#include "GTStandardConverter.h"
#include "Representations/Cognition/RobotPose.h"
#include "Representations/Cognition/BallModel.h"
#include "Representations/Cognition/PlayerPoseCollection.h"
#include "Representations/Cognition/ObstaclesModel.h"
#include "Representations/WLan/TeamMessageCollection.h"
#include "Tools/FieldDimensions.h"
#include <string>



GTStandardConverter::GTStandardConverter(const char* filename)
{
  load(filename);
}


GTStandardConverter::GTStandardConverter()
{
}


void GTStandardConverter::load(const char* filename)
{
  std::string file(filename);
  potentialfields.load(file);
  freeGoalDesc.objectId = potentialfields.getIdFromObjectStateSymbol("free-goal-symbol");
  freeGoalDesc.objectName = "free-goal-symbol";
  goalieLookingPositionDesc.objectId = potentialfields.getIdFromObjectStateSymbol("goalie-lookat-symbol");
  goalieLookingPositionDesc.objectName = "goalie-lookat-symbol";
  goalieLookingPositionDesc.isActive = true;
  goaliePositionDesc.objectId = potentialfields.getIdFromObjectStateSymbol("goalie-position-symbol");
  goaliePositionDesc.objectName = "goalie-position-symbol";
  goaliePositionDesc.isActive = true;
  selfDesc.objectId = potentialfields.getIdFromObjectStateSymbol("self-symbol");
  selfDesc.objectName = "self-symbol";
  ballDesc.objectId = potentialfields.getIdFromObjectStateSymbol("ball-symbol");
  ballDesc.objectName = "ball-symbol";
  seenBallDesc.objectId = potentialfields.getIdFromObjectStateSymbol("seen-ball-symbol");
  seenBallDesc.objectName = "seen-ball-symbol";
  goalieDesc.objectId = potentialfields.getIdFromObjectStateSymbol("goalie-symbol");
  goalieDesc.objectName = "goalie-symbol";
  player2Desc.objectId = potentialfields.getIdFromObjectStateSymbol("player-2-symbol");
  player2Desc.objectName = "player-2-symbol";
  player3Desc.objectId = potentialfields.getIdFromObjectStateSymbol("player-3-symbol");
  player3Desc.objectName = "player-3-symbol";
  player4Desc.objectId = potentialfields.getIdFromObjectStateSymbol("player-4-symbol");
  player4Desc.objectName = "player-4-symbol";
  unknownTeammate1Desc.objectId = potentialfields.getIdFromObjectStateSymbol("unknown-teammate-1-symbol");
  unknownTeammate1Desc.objectName = "unknown-teammate-1-symbol";
  unknownTeammate2Desc.objectId = potentialfields.getIdFromObjectStateSymbol("unknown-teammate-2-symbol");
  unknownTeammate2Desc.objectName = "unknown-teammate-2-symbol";
  unknownTeammate3Desc.objectId = potentialfields.getIdFromObjectStateSymbol("unknown-teammate-3-symbol");
  unknownTeammate3Desc.objectName = "unknown-teammate-3-symbol";
  unknownTeammate4Desc.objectId = potentialfields.getIdFromObjectStateSymbol("unknown-teammate-4-symbol");
  unknownTeammate4Desc.objectName = "unknown-teammate-4-symbol";
  opponent1Desc.objectId = potentialfields.getIdFromObjectStateSymbol("opponent-1-symbol");
  opponent1Desc.objectName = "opponent-1-symbol";
  opponent2Desc.objectId = potentialfields.getIdFromObjectStateSymbol("opponent-2-symbol");
  opponent2Desc.objectName = "opponent-2-symbol";
  opponent3Desc.objectId = potentialfields.getIdFromObjectStateSymbol("opponent-3-symbol");
  opponent3Desc.objectName = "opponent-3-symbol";
  opponent4Desc.objectId = potentialfields.getIdFromObjectStateSymbol("opponent-4-symbol");
  opponent4Desc.objectName = "opponent-4-symbol";
  opponent5Desc.objectId = potentialfields.getIdFromObjectStateSymbol("opponent-5-symbol");
  opponent5Desc.objectName = "opponent-5-symbol";
  opponent6Desc.objectId = potentialfields.getIdFromObjectStateSymbol("opponent-6-symbol");
  opponent6Desc.objectName = "opponent-6-symbol";
  destination1Desc.objectId = potentialfields.getIdFromObjectStateSymbol("destination1-symbol");
  destination1Desc.objectName = "destination1-symbol";
  destination2Desc.objectId = potentialfields.getIdFromObjectStateSymbol("destination2-symbol");
  destination2Desc.objectName = "destination2-symbol";
  obstacleFrontDesc.objectId = potentialfields.getIdFromObjectStateSymbol("obstacle-front-symbol");
  obstacleFrontDesc.objectName = "obstacle-front-symbol";
  obstacleLeftDesc.objectId = potentialfields.getIdFromObjectStateSymbol("obstacle-left-symbol");
  obstacleLeftDesc.objectName = "obstacle-left-symbol";
  obstacleRightDesc.objectId = potentialfields.getIdFromObjectStateSymbol("obstacle-right-symbol");
  obstacleRightDesc.objectName = "obstacle-right-symbol";
  obstacleFrontLeftDesc.objectId = potentialfields.getIdFromObjectStateSymbol("obstacle-front-left-symbol");
  obstacleFrontLeftDesc.objectName = "obstacle-front-left-symbol";
  obstacleFrontRightDesc.objectId = potentialfields.getIdFromObjectStateSymbol("obstacle-front-right-symbol");
  obstacleFrontRightDesc.objectName = "obstacle-front-right-symbol";
  // Set the angles
  angleToFrontObstacle = 0.0;
  angleToLeftObstacle = fromDegrees(72.0);
  angleToRightObstacle = fromDegrees(-72.0);
  angleToFrontLeftObstacle = fromDegrees(36.0);
  angleToFrontRightObstacle = fromDegrees(-36.0);
  openingAngle = fromDegrees(36.0);
}


void GTStandardConverter::execute(const RobotPose& robotPose,
                                const BallModel& ballPosition,
                                const PlayerPoseCollection& playerPoseCollection,
                                const ObstaclesModel& obstaclesModel,
                                const TeamMessageCollection& teamMessageCollection,
                                PotentialfieldResult& result)
{
  Pose2D dummy1, dummy2;
  execute(robotPose, ballPosition, playerPoseCollection, 
    obstaclesModel, teamMessageCollection, dummy1, dummy2, 0, result);
}


void GTStandardConverter::execute(const RobotPose& robotPose,
                                const BallModel& ballPosition,
                                const PlayerPoseCollection& playerPoseCollection,
                                const ObstaclesModel& obstaclesModel,
                                const TeamMessageCollection& teamMessageCollection,
                                const Pose2D& specialDestination1,
                                const Pose2D& specialDestination2,
                                const int& numberOfSpecialDestinations,
                                PotentialfieldResult& result)
{
  setData(robotPose, ballPosition, playerPoseCollection, obstaclesModel, teamMessageCollection);
  if(numberOfSpecialDestinations > 1)
  {
    destination1Desc.isActive = true;
    destination1Desc.pose.pos.x = specialDestination1.translation.x;
    destination1Desc.pose.pos.y = specialDestination1.translation.y;
    destination1Desc.pose.rotation = specialDestination1.rotation;
    destination1Desc.pose.probability = 1.0;
    destination2Desc.isActive = true;
    destination2Desc.pose.pos.x = specialDestination2.translation.x;
    destination2Desc.pose.pos.y = specialDestination2.translation.y;
    destination2Desc.pose.rotation = specialDestination2.rotation;
    destination2Desc.pose.probability = 1.0;
  }
  else if(numberOfSpecialDestinations > 0)
  {
    destination1Desc.isActive = true;
    destination1Desc.pose.pos.x = specialDestination1.translation.x;
    destination1Desc.pose.pos.y = specialDestination1.translation.y;
    destination1Desc.pose.rotation = specialDestination1.rotation;
    destination1Desc.pose.probability = 1.0;
    destination2Desc.isActive = false;
  }
  else
  {
    destination1Desc.isActive = false;
    destination2Desc.isActive = false;
  }
  potentialfields.setObjectState(destination1Desc);
  potentialfields.setObjectState(destination2Desc);
  potentialfields.execute(result);
}


void GTStandardConverter::setData(const RobotPose& robotPose,
                                const BallModel& ballPosition,
                                const PlayerPoseCollection& playerPoseCollection,
                                const ObstaclesModel& obstaclesModel,
                                const TeamMessageCollection& teamMessageCollection)
{
  //Set own position
  PfPose ownPose;
  ownPose.rotation = robotPose.rotation;
  ownPose.pos.x = robotPose.translation.x;
  ownPose.pos.y = robotPose.translation.y;
  potentialfields.setOwnPose(ownPose);
  selfDesc.pose = ownPose;
  potentialfields.setObjectState(selfDesc);

  //Set the goal position
  freeGoalDesc.pose.pos.x = (double)xPosOpponentGroundline;
  freeGoalDesc.pose.pos.y = 0.0;
  freeGoalDesc.pose.rotation = 0.0;
  freeGoalDesc.isActive = true;
  potentialfields.setObjectState(freeGoalDesc);

  //Set the position of the own goalie
  bool ballKnown(ballPosition.getTimeSinceLastKnown(3000) < 5000);
  if(ballPosition.getKnownPosition(3000).x < -1000 && ballKnown && 
     ballPosition.getKnownPosition(3000).y > 500)
  {
    goaliePositionDesc.pose.pos.x = -1930;
    goaliePositionDesc.pose.pos.y = 350;
  }
  else if(ballPosition.getKnownPosition(3000).x < -1000 && ballKnown && 
          ballPosition.getKnownPosition(3000).y < -500)
  {
    goaliePositionDesc.pose.pos.x = -1930;
    goaliePositionDesc.pose.pos.y = -350;
  }
  else
  {
    goaliePositionDesc.pose.pos.x = -2000;
    goaliePositionDesc.pose.pos.y = 0;
  }
  potentialfields.setObjectState(goaliePositionDesc);

  //Set the ball positions
  if(SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen) < 2000)
  {
    ballDesc.pose.pos.x = ballPosition.seen.x;
    ballDesc.pose.pos.y = ballPosition.seen.y;
    seenBallDesc.pose.pos.x = ballPosition.seen.x;
    seenBallDesc.pose.pos.y = ballPosition.seen.y;
    seenBallDesc.isActive = true;
  }
  else
  {
    ballDesc.pose.pos.x = ballPosition.getKnownPosition(2000).x;
    ballDesc.pose.pos.y = ballPosition.getKnownPosition(2000).y;
    seenBallDesc.pose.pos.x = ballPosition.getKnownPosition(2000).x;
    seenBallDesc.pose.pos.y = ballPosition.getKnownPosition(2000).y;
    seenBallDesc.isActive = false;
  }
  goalieLookingPositionDesc.isActive = true;
  if(ballPosition.getTimeSinceLastKnown(2000)>3000)
  {
    goalieLookingPositionDesc.pose.pos.x = 0;
    goalieLookingPositionDesc.pose.pos.y = 0;
    ballDesc.isActive = false;
  }
  else
  {
    goalieLookingPositionDesc.pose = ballDesc.pose;
    ballDesc.isActive = true;
  }
  potentialfields.setObjectState(ballDesc);
  potentialfields.setObjectState(seenBallDesc);
  potentialfields.setObjectState(goalieLookingPositionDesc);
  
  //Set the positions of the opponent robots
  int numOfOpponents = playerPoseCollection.numberOfOpponentPlayers;
  PlayerPose playerPose;
  if(numOfOpponents > 0)
  {
    playerPose = playerPoseCollection.getOpponentPlayerPose(0);
    opponent1Desc.pose.pos.x = playerPose.getPose().translation.x;
    opponent1Desc.pose.pos.y = playerPose.getPose().translation.y;
    opponent1Desc.pose.rotation = playerPose.getPose().rotation;
    opponent1Desc.pose.probability = playerPose.getValidity();
    opponent1Desc.isActive = true;
  }
  else
  {
    opponent1Desc.isActive = false;
  }
  if(numOfOpponents > 1)
  {
    playerPose = playerPoseCollection.getOpponentPlayerPose(1);
    opponent2Desc.pose.pos.x = playerPose.getPose().translation.x;
    opponent2Desc.pose.pos.y = playerPose.getPose().translation.y;
    opponent2Desc.pose.rotation = playerPose.getPose().rotation;
    opponent2Desc.pose.probability = playerPose.getValidity();
    opponent2Desc.isActive = true;
  }
  else
  {
    opponent2Desc.isActive = false;
  }
  if(numOfOpponents > 2)
  {
    playerPose = playerPoseCollection.getOpponentPlayerPose(2);
    opponent3Desc.pose.pos.x = playerPose.getPose().translation.x;
    opponent3Desc.pose.pos.y = playerPose.getPose().translation.y;
    opponent3Desc.pose.rotation = playerPose.getPose().rotation;
    opponent3Desc.pose.probability = playerPose.getValidity();
    opponent3Desc.isActive = true;
  }
  else
  {
    opponent3Desc.isActive = false;
  }
  if(numOfOpponents > 3)
  {
    playerPose = playerPoseCollection.getOpponentPlayerPose(3);
    opponent4Desc.pose.pos.x = playerPose.getPose().translation.x;
    opponent4Desc.pose.pos.y = playerPose.getPose().translation.y;
    opponent4Desc.pose.rotation = playerPose.getPose().rotation;
    opponent4Desc.pose.probability = playerPose.getValidity();
    opponent4Desc.isActive = true;
  }
  else
  {
    opponent4Desc.isActive = false;
  }
  if(numOfOpponents > 4)
  {
    playerPose = playerPoseCollection.getOpponentPlayerPose(4);
    opponent5Desc.pose.pos.x = playerPose.getPose().translation.x;
    opponent5Desc.pose.pos.y = playerPose.getPose().translation.y;
    opponent5Desc.pose.rotation = playerPose.getPose().rotation;
    opponent5Desc.pose.probability = playerPose.getValidity();
    opponent5Desc.isActive = true;
  }
  else
  {
    opponent5Desc.isActive = false;
  }
  if(numOfOpponents > 5)
  {
    playerPose = playerPoseCollection.getOpponentPlayerPose(5);
    opponent6Desc.pose.pos.x = playerPose.getPose().translation.x;
    opponent6Desc.pose.pos.y = playerPose.getPose().translation.y;
    opponent6Desc.pose.rotation = playerPose.getPose().rotation;
    opponent6Desc.pose.probability = playerPose.getValidity();
    opponent6Desc.isActive = true;
  }
  else
  {
    opponent6Desc.isActive = false;
  }
  potentialfields.setObjectState(opponent1Desc);
  potentialfields.setObjectState(opponent2Desc);
  potentialfields.setObjectState(opponent3Desc);
  potentialfields.setObjectState(opponent4Desc);
  potentialfields.setObjectState(opponent5Desc);
  potentialfields.setObjectState(opponent6Desc);
  
  //Set the positions of the teammates...
  RobotPose teammatePose;
  int numOfTeammates = teamMessageCollection.numberOfTeamMessages;
  if(numOfTeammates > 0)
  {
    teammatePose = teamMessageCollection[0].robotPose;
    unknownTeammate1Desc.pose.pos.x = teammatePose.getPose().translation.x;
    unknownTeammate1Desc.pose.pos.y = teammatePose.getPose().translation.y;
    unknownTeammate1Desc.pose.rotation = teammatePose.getPose().rotation;
    unknownTeammate1Desc.pose.probability = teammatePose.getValidity();
    unknownTeammate1Desc.isActive = true;
  }
  else
  {
    unknownTeammate1Desc.isActive = false;
  }
  if(numOfTeammates > 1)
  {
    teammatePose = teamMessageCollection[0].robotPose;
    unknownTeammate2Desc.pose.pos.x = teammatePose.getPose().translation.x;
    unknownTeammate2Desc.pose.pos.y = teammatePose.getPose().translation.y;
    unknownTeammate2Desc.pose.rotation = teammatePose.getPose().rotation;
    unknownTeammate2Desc.pose.probability = teammatePose.getValidity();
    unknownTeammate2Desc.isActive = true;
  }
  else
  {
    unknownTeammate2Desc.isActive = false;
  }
  if(numOfTeammates > 2)
  {
    teammatePose = teamMessageCollection[0].robotPose;
    unknownTeammate3Desc.pose.pos.x = teammatePose.getPose().translation.x;
    unknownTeammate3Desc.pose.pos.y = teammatePose.getPose().translation.y;
    unknownTeammate3Desc.pose.rotation = teammatePose.getPose().rotation;
    unknownTeammate3Desc.pose.probability = teammatePose.getValidity();
    unknownTeammate3Desc.isActive = true;
  }
  else
  {
    unknownTeammate3Desc.isActive = false;
  }
  unknownTeammate4Desc.isActive = false;
/*  int numOfTeammates = playerPoseCollection.numberOfOwnPlayers; 
  if(numOfTeammates > 0)
  {
    playerPose = playerPoseCollection.getOwnPlayerPose(0);
    unknownTeammate1Desc.pose.pos.x = playerPose.getPose().translation.x;
    unknownTeammate1Desc.pose.pos.y = playerPose.getPose().translation.y;
    unknownTeammate1Desc.pose.rotation = playerPose.getPose().rotation;
    unknownTeammate1Desc.pose.probability = playerPose.getValidity();
    unknownTeammate1Desc.isActive = true;
  }
  else
  {
    unknownTeammate1Desc.isActive = false;
  }
  if(numOfTeammates > 1)
  {
    playerPose = playerPoseCollection.getOwnPlayerPose(1);
    unknownTeammate2Desc.pose.pos.x = playerPose.getPose().translation.x;
    unknownTeammate2Desc.pose.pos.y = playerPose.getPose().translation.y;
    unknownTeammate2Desc.pose.rotation = playerPose.getPose().rotation;
    unknownTeammate2Desc.pose.probability = playerPose.getValidity();
    unknownTeammate2Desc.isActive = true;
  }
  else
  {
    unknownTeammate2Desc.isActive = false;
  }
  if(numOfTeammates > 2)
  {
    playerPose = playerPoseCollection.getOwnPlayerPose(2);
    unknownTeammate3Desc.pose.pos.x = playerPose.getPose().translation.x;
    unknownTeammate3Desc.pose.pos.y = playerPose.getPose().translation.y;
    unknownTeammate3Desc.pose.rotation = playerPose.getPose().rotation;
    unknownTeammate3Desc.pose.probability = playerPose.getValidity();
    unknownTeammate3Desc.isActive = true;
  }
  else
  {
    unknownTeammate3Desc.isActive = false;
  }
  if(numOfTeammates > 3)
  {
    playerPose = playerPoseCollection.getOwnPlayerPose(3);
    unknownTeammate4Desc.pose.pos.x = playerPose.getPose().translation.x;
    unknownTeammate4Desc.pose.pos.y = playerPose.getPose().translation.y;
    unknownTeammate4Desc.pose.rotation = playerPose.getPose().rotation;
    unknownTeammate4Desc.pose.probability = playerPose.getValidity();
    unknownTeammate4Desc.isActive = true;
  }
  else
  {
    unknownTeammate4Desc.isActive = false;
  }*/
  potentialfields.setObjectState(unknownTeammate1Desc);
  potentialfields.setObjectState(unknownTeammate2Desc);
  potentialfields.setObjectState(unknownTeammate3Desc);
  potentialfields.setObjectState(unknownTeammate4Desc);

  /* Specific roles are deactivated until a better player modelling has 
     been implemented*/
  goalieDesc.isActive = false;
  player2Desc.isActive = false;
  player3Desc.isActive = false;
  player4Desc.isActive = false;
  potentialfields.setObjectState(goalieDesc);
  potentialfields.setObjectState(player2Desc);
  potentialfields.setObjectState(player3Desc);
  potentialfields.setObjectState(player4Desc);

  //Set the obstacle positions
  PfVec relPos;
  int distanceToFrontObstacle = obstaclesModel.getDistanceInDirection(angleToFrontObstacle,openingAngle);
  if(distanceToFrontObstacle < 600)
  {
    obstacleFrontDesc.isActive = true;
    relPos.x = distanceToFrontObstacle;
    relPos.y = 0.0;
    relPos.rotate(ownPose.rotation);
    obstacleFrontDesc.pose.pos = (ownPose.pos+relPos);
  }
  else
  {
    obstacleFrontDesc.isActive = false;
  }
  int distanceToLeftObstacle = obstaclesModel.getDistanceInDirection(angleToLeftObstacle,openingAngle);
  if(distanceToLeftObstacle < 600)
  {
    obstacleLeftDesc.isActive = true;
    relPos.x = distanceToLeftObstacle;
    relPos.y = 0.0;
    relPos.rotate(ownPose.rotation);
    relPos.rotate(angleToLeftObstacle);
    obstacleLeftDesc.pose.pos = (ownPose.pos+relPos);
  }
  else
  {
    obstacleLeftDesc.isActive = false;
  }
  int distanceToRightObstacle = obstaclesModel.getDistanceInDirection(angleToRightObstacle,openingAngle);
  if(distanceToRightObstacle < 600)
  {
    obstacleRightDesc.isActive = true;
    relPos.x = distanceToRightObstacle;
    relPos.y = 0.0;
    relPos.rotate(ownPose.rotation);
    relPos.rotate(angleToRightObstacle);
    obstacleRightDesc.pose.pos = (ownPose.pos+relPos);
  }
  else
  {
    obstacleRightDesc.isActive = false;
  }
  int distanceToFrontLeftObstacle = obstaclesModel.getDistanceInDirection(angleToFrontLeftObstacle,openingAngle);
  if(distanceToFrontLeftObstacle < 600)
  {
    obstacleFrontLeftDesc.isActive = true;
    relPos.x = distanceToFrontLeftObstacle;
    relPos.y = 0.0;
    relPos.rotate(ownPose.rotation);
    relPos.rotate(angleToFrontLeftObstacle);
    obstacleFrontLeftDesc.pose.pos = (ownPose.pos+relPos);
  }
  else
  {
    obstacleFrontLeftDesc.isActive = false;
  }
  int distanceToFrontRightObstacle = obstaclesModel.getDistanceInDirection(angleToFrontRightObstacle,openingAngle);
  if(distanceToFrontRightObstacle < 600)
  {
    obstacleFrontRightDesc.isActive = true;
    relPos.x = distanceToFrontRightObstacle;
    relPos.y = 0.0;
    relPos.rotate(ownPose.rotation);
    relPos.rotate(angleToFrontRightObstacle);
    obstacleFrontRightDesc.pose.pos = (ownPose.pos+relPos);
  }
  else
  {
    obstacleFrontRightDesc.isActive = false;
  }
  potentialfields.setObjectState(obstacleFrontDesc);
  potentialfields.setObjectState(obstacleRightDesc);
  potentialfields.setObjectState(obstacleLeftDesc);
  potentialfields.setObjectState(obstacleFrontRightDesc);
  potentialfields.setObjectState(obstacleFrontLeftDesc);
}


void GTStandardConverter::getValueArray(const RobotPose& robotPose,
                     const BallModel& ballPosition,
                     const PlayerPoseCollection& playerPoseCollection, 
                     const ObstaclesModel& obstaclesModel,
                     const TeamMessageCollection& teamMessageCollection,
                     const std::string& fieldname, double x1, double y1,
                     double x2, double y2, int xSteps, int ySteps, double value[], double& max)
{
  setData(robotPose, ballPosition, playerPoseCollection, obstaclesModel, teamMessageCollection);
  potentialfields.getValueArray(fieldname, x1, y1, x2, y2, xSteps, ySteps, value, max);
}

 
void GTStandardConverter::getDirectionArray(const RobotPose& robotPose,
                         const BallModel& ballPosition,
                         const PlayerPoseCollection& playerPoseCollection,
                         const ObstaclesModel& obstaclesModel,
                         const TeamMessageCollection& teamMessageCollection,
                         const std::string& fieldname, double x1, double y1,
                         double x2, double y2, int xSteps, int ySteps, PfVec directions[])
{
  setData(robotPose, ballPosition, playerPoseCollection, obstaclesModel, teamMessageCollection);
  potentialfields.getDirectionArray(fieldname, x1, y1, x2, y2, xSteps, ySteps, directions);
}


void GTStandardConverter::getFieldNames(std::vector<std::string>& fieldNames)
{
  fieldNames = potentialfields.getFieldNames();
}



/*
 * $Log: GTStandardConverter.cpp,v $
 * Revision 1.2  2004/06/17 16:20:30  tim
 * improved ;-)
 *
 * Revision 1.1  2004/05/24 13:45:10  tim
 * added potential field viewer again
 *
 */
