/**
* @file FormationObject.h
* 
* Definition of class FormationObject
*
* @author <a href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</a>
*/

#ifndef FORMATIONOBJECT_H_
#define FORMATIONOBJECT_H_


#include <string>
#include <vector>
#include "FieldObject.h"
#include "PotentialFunctions.h"
#include "PfieldDatatypes.h"

class PotentialfieldComposition;


/** A type used by RelativeFormation to distinguish between two coordinate systems*/
enum FormationCoordinates {RELATIVE_FORMATION, ABSOLUTE_FORMATION}; 

/** A type used by BestFitFormation to describe the selection criterion*/
enum BestFitSelection {SELECT_MAX_GRADIENT, SELECT_MIN_GRADIENT, 
                       SELECT_MIN_DISTANCE, SELECT_MAX_PRIORITY};


/**
* @class SingleFormation
*
* An abstract class representing a simple formation
*/
class SingleFormation
{
public:
  /** Constructor*/
  SingleFormation();

  /** Destructor*/
  virtual ~SingleFormation(){}

  /** Checks if the formation is active
  * @return true, if the formation is active
  */
  virtual bool isActive() const
  { return false;}

  /** Checks if the formation has a static position
  * @return true, if the formation has a static position
  */
  virtual bool isStatic() const;

  /** Returns the position of the formation in reference to the robot pose
  *   (used for A*-search to a formation)
  * @param robotPose The robot pose
  * @return The position of the formation
  */
  virtual PfVec getPosition(const PfPose& robotPose);

  /** Returns the field vector to/from a formation to/from a given pose
  * @param otherPose The pose on which the formation affects
  * @return The vector
  */
  virtual PfVec getVec(const PfPose& otherPose);

  /** Returns the field charge of the formation at a given pose
  * @param otherPose The pose
  * @return The charge of the formation
  */
  virtual double getCharge(const PfPose& otherPose);

  /** Returns the distance between a pose and the formation area
  * @param otherPose The pose
  * @return The distance
  */
  double getDistanceToFormation(const PfPose& otherPose) const;

  /** Adds an object to the formation
  * @param object The object
  */
  void addObject(Object* object);

  /** Sets the function of the field of the formation
  * @param function The function
  */
  void setFunction(PotentialfieldFunction* function)
  { this->geometricFormationObject.setFunction(function);}

  /** Sets the priority of the formation
  * @param priority
  */
  void setPriority(double priority)
  { this->priority = priority;}

  /** Returns the priority of the formation
  * @return The priority
  */
  double getPriority() const
  { return priority;}

  /** Initializes the formation*/
  void init()
  { updateGeometry();}

  /** Checks if a position is inside the formation area
  * @param p The position
  * @return true, if the position is inside
  */
  virtual bool positionInsideFormation(const PfVec& p) const
  { return false;}

  /** Creates a copy of the formation object
  * @return A pointer to a copy
  */
  virtual SingleFormation* copy() const;

protected:
  /** The objects the formation is spanning*/
  std::vector<Object*> objects;
  /** An object representing the shape of the formation*/
  Object geometricFormationObject;
  /** The position of the robot (kept for internal purposes)*/
  PfVec robotPosition;
  /** The priority of the formation (may be used for selection)*/
  double priority;
  /** Updates the positions of the formation*/
  virtual void updateGeometry() {};

};


/**
* @class BetweenFormation
*
* A class representing a formation between two other objects
*/
class BetweenFormation: public SingleFormation
{
public:
  /** Constructor*/
  BetweenFormation();

  /** Checks if the formation is active
  * @return true, if the formation is active
  */
  bool isActive() const;

  /** Checks if a position is inside the formation area
  * @param p The position
  * @return true, if the position is inside
  */
  virtual bool positionInsideFormation(const PfVec& p) const;

protected:
  /** Updates the positions of the formation*/
  virtual void updateGeometry();
};


/**
* @class AmongFormation
*
* A class representing a formation among several objects
*/
class AmongFormation: public SingleFormation
{
public:
  /** Constructor*/
  AmongFormation();

  /** Checks if the formation is active
  * @return true, if the formation is active
  */
  bool isActive() const;

  /** Checks if a position is inside the formation area
  * @param p The position
  * @return true, if the position is inside
  */
  virtual bool positionInsideFormation(const PfVec& p) const;

protected:
  /** Updates the positions of the formation*/
  virtual void updateGeometry();
};


/**
* @class RelativeFormation
*
* A class representing a formation relative to another object
*/
class RelativeFormation: public SingleFormation
{
public:
  /** Constructor*/
  RelativeFormation();

  /** Checks if the formation is active
  * @return true, if the formation is active
  */
  bool isActive() const;

  /** Sets the two specific parameters of a RelativeFormation
  * @param angle The angle to the other object
  * @param coordinates The coordinate system used
  */
  void setParameters(double angle, FormationCoordinates coordinates)
  { this->angle = angle; this->coordinates = coordinates;}

  /** Creates a copy of the formation object
  * @return A pointer to a copy
  */
  SingleFormation* copy() const;

protected:
  /** The angle to the other object*/
  double angle;
  /** The coordinate system used*/
  FormationCoordinates coordinates;

  /** Updates the positions of the formation*/
  virtual void updateGeometry();
};


/**
* @class BestFitFormation
*
* A class representing a formation selection the "best fitting"
* formation out of a set of SingleFormations.
*/
class BestFitFormation: public SingleFormation
{
public:
  /** Constructor*/
  BestFitFormation();

  /** Destructor*/
  ~BestFitFormation();

  /** Checks if the formation is active
  * @return true, if the formation is active
  */
  bool isActive() const;

  /** Checks if the formation has a static position
  * @return true, if the formation has a static position
  */
  bool isStatic() const;

  /** Returns the position of the formation in reference to the robot pose
  *   (used for A*-search to a formation)
  * @param robotPose The robot pose
  * @return The position of the formation
  */
  virtual PfVec getPosition(const PfPose& robotPose);

  /** Returns the field vector to/from a formation to/from a given pose
  * @param otherPose The pose on which the formation affects
  * @return The vector
  */  
  PfVec getVec(const PfPose& otherPose);

  /** Returns the field charge of the formation at a given pose
  * @param otherPose The pose
  * @return The charge of the formation
  */
  double getCharge(const PfPose& otherPose);

  /** Adds a single formation to the set of formations
  * @param formation The formation
  */
  void addFormation(SingleFormation* formation)
  { 
    formations.push_back(formation);
    resultVecs.push_back(PfVec(0.0,0.0));
    resultDistances.push_back(0.0);
  }

  /** Sets the criterion for choosing the best fitting formation
  * @param bestFitSelection The selection criterion
  */
  void setBestFitSelection(BestFitSelection bestFitSelection)
  { this->bestFitSelection = bestFitSelection;}

  /** Creates a copy of the formation object
  * @return A pointer to a copy
  */
  SingleFormation* copy() const;

protected:
  /** The set of formations*/
  std::vector<SingleFormation*> formations;
  /** A list of vectors corresponding to the formations*/
  std::vector<PfVec> resultVecs;
  /** A list of distances corresponding to the formations*/
  std::vector<double> resultDistances;
  /** The selection criterion*/
  BestFitSelection bestFitSelection;

  /** Updates the positions of the formation*/
  virtual void updateGeometry() {};

  /** Determines the best formation
  * @param otherPose The pose to find the bestFormationFor
  * @return The index of the best formation, -1 if none exists
  */
  int findBestFormation(const PfPose& otherPose);
};


/**
* @class FormationObject
*
* A class representing a formation in a potential field
*/
class FormationObject: public Object
{
public:
  /** Constructor 
  * @param name The name of the formation object
  */
  FormationObject(const std::string& name);

  /** Destructor */
  virtual ~FormationObject();

  /** Returns the pose of the object in reference to the robot pose
  *   (used for getting a pose from formation objects)
  * @param robotPose The robot pose
  * @return The object pose
  */
  virtual PfPose getPose(const PfPose& robotPose);

  /** Computes the gradient of the formation at a given pose
  * @param otherPose The pose
  * @return The gradient (= f'(otherPose))
  */
  virtual PfVec computeAbsFieldVecAt(const PfPose& otherPose);

  /** Computes the "charge" of the formation at a given pose
  * @param otherPose The pose
  * @return The charge (= f(otherPose))
  */
  virtual double computeChargeAt(const PfPose& otherPose);
  
  /** Return whether the object is static or not
  * @return true, if the object is static
  */
  virtual bool isStatic() const;

  /** Returns whether the object is active or not
  * @return true, if active
  */
  virtual bool isActive() const;

  /** Returns a pointer to the geometry of the object
  * @return The geometric description
  */
  virtual PfieldGeometricObject* getGeometry() const
  { return NULL;}

  /** Returns a pointer to the absolute geometry of the object
  * @return The geometric description
  */
  virtual PfieldGeometricObject* getAbsGeometry() const
  { return NULL;}
 
  /** Overrides function from Object, does nothing*/
  virtual void updateData() {}

  /** Polymorph copy function
  * @return A copy of the FormationObject
  */
  virtual Object* getCopy();

  /** Adds a new formation to his object
  * @param formation The formation
  */
  void addSingleFormation(SingleFormation* formation)
  { singleFormations.push_back(formation);}

protected:  
  /** The list of single formations belonging to this object*/
  std::vector< SingleFormation* > singleFormations;
};


#endif	//FORMATIONOBJECT_H_


/*
* $Log: FormationObject.h,v $
* Revision 1.1.1.1  2004/05/22 17:37:26  cvsadm
* created new repository GT2004_WM
*
* Revision 1.1  2004/01/20 15:42:19  tim
* Added potential fields implementation
*
*/
