/**
* @file FieldObject.h
* 
* Definition of class Object
*
* @author <a href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</a>
*/

#ifndef OBJECT_H_
#define OBJECT_H_

#include "PotentialFunctions.h"
#include "PfieldDatatypes.h"
#include "PfieldGeometry.h"
#include <string>

class PotentialfieldComposition;


/** The possible types of tangential fields*/
enum TangentialField {NO_TANGENTIALFIELD, CLOCKWISE, COUNTER_CLOCKWISE};


/**
* @class Object
*
* A class representing an object in a potential field
*/
class Object
{
public:
  /** Constructor */
  Object();

  /** Constructor 
  * @param name The name of the object
  * @param objectType The type of the object
  */
  Object(const std::string& name, ObjectType objectType);

  /** Destructor */
  virtual ~Object();

  /** Computes the impact of an object at a given pose
  * @param otherPose The pose
  * @return The charge (= f(otherPose))
  */
  virtual double computeChargeAt(const PfPose& otherPose);

  /** Computes the gradient of an object at a given pose
  * @param otherPose The pose
  * @return The gradient (= f'(otherPose))
  */
  virtual PfVec computeAbsFieldVecAt(const PfPose& otherPose);

  /** Creates a copy of this object and changes the name of the copied object
  * @param instanceName The name of the copy
  * @return The copy
  */
  Object* createInstance(const std::string& instanceName);

  /** Copy operator
  * @param other The object to copy
  */
  void operator = (const Object& other);

  /** Polymorph copy function
  * @return A copy of the object
  */
  virtual Object* getCopy();

  /** Get new data from other object. It is assumed that a full copy had been done
  *   before. Only changing data (e.g. the pose) is copied.
  * @param other The object to get the data from
  */
  void getMinimalCopyFrom(Object* other);

  /** Returns the name of the object
  * @return The name
  */
  std::string getName() const
  { return name;}

  /** Returns the pose of the object
  * @return The pose
  */
  PfPose getPose() const
  { return pose;}

  /** Returns the pose of the object in reference to the robot pose
  *   (used for getting a pose from formation objects)
  * @param robotPose The robot pose
  * @return The object pose
  */
  virtual PfPose getPose(const PfPose& robotPose)
  { return pose;}

  /** Sets the pose of the object and updates the absolute geometry
  * @param pose The new pose
  */
  void setPose(const PfPose& pose);

  /** Sets the state of the object
  * @param active Flag which determines whether the object is active or not
  */
  void setActivation(const bool& active)
  { this->active = active;}

  /** Returns whether the object is active or not
  * @return true, if active
  */
  virtual bool isActive() const
  { return active;}

  /** Returns the type of the object
  * @return The type
  */
  ObjectType getType() const
  { return objectType;}

  /** Sets the static/dynamic state of the object
  * @param isStatic Flag
  * @param dynamicPoseId The id of the static pose
  */
  void setStatic(const bool& isStatic, unsigned int dynamicPoseId=0)
  { 
    this->isStaticVar = isStatic;
    this->dynamicPoseId = dynamicPoseId;
  }

  /** Return whether the object is static or not
  * @return true, if the object is static
  */
  virtual bool isStatic() const
  { return isStaticVar;}

  /** Sets the function of the object
  * @param function The function
  */
  void setFunction(PotentialfieldFunction* function);

  /** Sets the field type of the object
  * @param type The field type
  */
  void setField(FieldType type);

  /** Sets the field type of the object
  * @param type The field type
  * @param sector A sector description (if type == SECTOR_FIELD)
  */
  void setField(FieldType type, const Sector& sector);

  /** Sets the type of the tangential field
  * @param tangentialField The type of the tangential field
  */
  void setTangentialField(TangentialField tangentialField)
  { this->tangentialField = tangentialField;}

  /** Sets the geometry of the object
  * @param geometry The geometric description
  */
  void setGeometry(PfieldGeometricObject* geometry);
  
  /** Returns a pointer to the geometry of the object
  * @return The geometric description
  */
  virtual PfieldGeometricObject* getGeometry() const
  { return geometry;}

  /** Returns a pointer to the absolute geometry of the object
  * @return The geometric description
  */
  virtual PfieldGeometricObject* getAbsGeometry() const
  { return absGeometry;}

  /** Sets the reference to the PotentialfieldComposition
  * @param ref The reference
  */
  void setPfieldCompositionRef(PotentialfieldComposition* ref)
  { this->refToPfieldComposition = ref;}
 
  /** Gets new data (pose/activation), if dynamic*/
  virtual void updateData();

  /** Computes the absolute geometry*/
  void computeAbsGeometry();

protected:
  /** The pose of the object*/
  PfPose pose;
  /** The name of the object*/
  std::string name;
  /** Flag, determines if object is active*/
  bool active;
  /** Flag, determines if object is static/dynamic*/
  bool isStaticVar;
  /** Flag, determines, if the field has a tangential direction*/
  TangentialField tangentialField;
  /** The type of the potential field*/
  FieldType fieldType;
  /** The function of this object*/
  PotentialfieldFunction* function;
  /** The type of the object (attractive/repulsive)*/
  ObjectType objectType;
  /** The description of the object's geometry*/
  PfieldGeometricObject* geometry;
  /** The description of the object's geometry in absolute coordinates*/
  PfieldGeometricObject* absGeometry;
  /** The sector of the potential field (if fieldType==SECTOR_FIELD)*/
  Sector sector;
  /** A pointer to the PotentialfieldComposition*/
  PotentialfieldComposition* refToPfieldComposition;
  /** The id of the dynamic pose (if used)*/
  unsigned int dynamicPoseId;

  /** Computes the charge at a pose
  * @param objectPose The pose of an object
  * @param position A position influenced by the object
  * @return The charge
  */
  double computeChargeForSinglePose(const PfPose& objectPose, const PfPose& position) const;

  /** Computes the gradient at a pose
  * @param objectPose The pose of an object
  * @param position A position influenced by the object
  * @return The gradient vector
  */
  PfVec computeGradientVecForSinglePose(const PfPose& objectPose, const PfPose& position) const;
};


#endif	//OBJECT_H_


/*
* $Log: FieldObject.h,v $
* Revision 1.1.1.1  2004/05/22 17:37:25  cvsadm
* created new repository GT2004_WM
*
* Revision 1.1  2004/01/20 15:42:19  tim
* Added potential fields implementation
*
* Revision 1.6  2003/06/13 14:27:58  tim
* added random generator and tangential fields
*
* Revision 1.5  2003/05/20 12:43:43  tim
* Changed function representation, fixed crash on SuperCore, integrated social functions
*
* Revision 1.4  2003/04/22 14:35:17  tim
* Merged changes from GO
*
* Revision 1.4  2003/04/09 19:03:06  tim
* Last commit before GermanOpen
*
* Revision 1.3  2003/04/04 14:50:53  tim
* Fixed bugs, added minor features
*
* Revision 1.2  2003/03/23 20:32:37  loetzsch
* removed green compiler warning: no newline at end of file
*
* Revision 1.1  2003/03/23 17:51:27  tim
* Added potentialfields
*
*/
