/**
* @file ModuleSelector.h
*
* Definition of class ModuleSelector.
*
* @author Max Risler
* @author Martin Ltzsch
*/

#ifndef __ModuleSelector_h_
#define __ModuleSelector_h_

#include "SolutionRequest.h"
#include "Module.h"

/**
* @class ModuleSelector
* 
* Base class for module selectors. Module selectors allow to switch between different
* solutions for a modules.
*
* @author Max Risler
* @author Martin Ltzsch
*/
class ModuleSelector : public MessageHandler
{
protected:
  /** The id of the selected solution */
  SolutionRequest::ModuleSolutionID selectedSolutionID;
  
public:
  /** 
  * Constructor 
  * @param id The id of the module. 
  */
  ModuleSelector(SolutionRequest::ModuleID id);
  
  /** Destructor */
  ~ModuleSelector();
  
  /** 
  * Is called on start and when the selected solution changes
  * to create a specific solution.
  * @param id The id of the solution to create
  * @return The created solution or 0
  */
  virtual Module* createSolution(SolutionRequest::ModuleSolutionID id) = 0;

  /** 
  * Creates the default solution.
  * This can not be done in the constructor, 
  * as the derived selector class is not created yet.
  */
  void init();

  /**
  * Selects a module.
  * @param id id of module to select
  */
  virtual void selectSolution(SolutionRequest::ModuleSolutionID id);
  
  /** Returns the index of the selected solution. */
  SolutionRequest::ModuleSolutionID getSelectedSolution() const;
  
  /** Executes the selected module */
  virtual void execute();

  /** 
  * Called from a MessageQueue to distribute messages.
  * Use message.getMessageID to decide if the message is relavant for 
  * the MesssageHandler derivate.
  * @param message The message that can be read.
  * @return true if the message was read (handled).
  */
  virtual bool handleMessage(InMessage& message);

protected:

  /** The id of the module, needed only for error messages. */
  SolutionRequest::ModuleID moduleID;

  /** The selected solution */
  Module *selectedSolution;
};


#endif /// __ModuleSelector_h_

/*
* Change log :
* 
* $Log: ModuleSelector.h,v $
* Revision 1.1.1.1  2004/05/22 17:37:19  cvsadm
* created new repository GT2004_WM
*
* Revision 1.2  2004/01/21 14:31:58  loetzsch
* Module Selectors create only the selected solution.
* When the solution changes, the old solution is erased and the new
* one ist created using createSolution(..)
*
* Revision 1.1  2003/10/07 10:13:24  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:29  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.3  2003/03/29 23:06:20  loetzsch
* maded pSolution protected instead of private
*
* Revision 1.2  2002/09/10 17:53:26  loetzsch
* First draft of new Module/Solution Architecture
*
* Revision 1.1  2002/09/10 15:53:59  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.2  2002/07/23 13:48:41  loetzsch
* - new streaming classes
* - removed many #include statements
* - exchanged StaticQueue by MessageQueue
* - new debug message handling
* - general clean up
*
* Revision 1.1.1.1  2002/05/10 12:40:32  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.9  2002/01/20 13:00:18  risler
* changed implementation of selecting default solution
*
* Revision 1.8  2002/01/17 14:35:48  risler
* SolutionRequest added
*
* Revision 1.7  2001/12/21 14:09:38  roefer
* Added several destructors
*
* Revision 1.6  2001/12/10 17:47:10  risler
* change log added
*
*/
