/**
* @file MessageQueue.h
*
* Definition of class MessageQueue.
*
* Include this file if for declaring instances of MessageQueue.
* 
* @author Martin Ltzsch
*/

#ifndef __MessageQueue_h_
#define __MessageQueue_h_

#include "InMessage.h"
#include "OutMessage.h"

/**
* @class MessageQueue
* 
* A queue for size varying, time stamped and type safe messages. 
* It can be used to collect very different types of messages for exchange between different
* threads, processes or systems.
*
* @author Martin Ltzsch
*
* Usage: 
* <pre>
* MessageQueue myQueue;
* myQueue.setSize(100000); // set the size of data that can be stored in the queue
* Image image1;
* myQueue.out.bin << image1; // write the binary message
* myQueue.out.finishMessage(idImage); // finish the message, set the type id of the message
* //
* // ... copy the queue between processes, systems
* //
* if (myQueue.in.getMessageID() == idImage) // check for the type of the next message
* {
*   Image image2;
*   myQueue.in.bin >> image2;  // read the image from the queue
* }
* </pre>   
*/ 
class MessageQueue
{
protected:
  /** the system dependend message queue base which is responsible for the data storage */
  MessageQueueBase queue;

public:
  /** An interface for reading messages from the queue */
  InMessage in;

  /** An interface for writing messages to the queue */
  OutMessage out;

  /** Default constructor */
  MessageQueue();

  /**
  * Sets the size of memory which is allocated for the queue. 
  * Ignored on the Win32 platform (dynamic allocation).
  * @param size The maximum size of the queue in Bytes. 
  */
  void setSize(int size);

  /** Returns the size of memory which is needed to write the queue to a stream. */
  int getStreamedSize();

  /** 
  * Specifies a team color and a player number that is atached to every new 
  * message. If not set, Player::undefinedTeamColor and Player::undefinedPlayerNumber 
  * is used.
  */
  void setPlayerForNewMessages(const Player& player);

  /** 
  * Calls a given MessageHandler for all messages of a kind in the queue. Note that the messages 
  * still remain in the queue and have to be removed manually with clear().
  * @param id handle only messages with this MessageID
  * @param handler a reference to a MessageHandler derivate
  */
  void handleSpecificMessages(MessageID id, MessageHandler& handler);

  /** 
  * Calls a given MessageHandler for all messages in the queue. Note that the messages 
  * still remain in the queue and have to be removed manually with clear().
  * @param handler a reference to a MessageHandler derivate
  */
  void handleAllMessages(MessageHandler& handler);

  /**
  * Copies all messages from this queue to another queue.
  * @param other the destination queue.
  */ 
  void copyAllMessages(MessageQueue& other);

  /**
  * Moves all messages from this queue to another queue.
  * @param other the destination queue.
  */ 
  void moveAllMessages(MessageQueue& other);

  /** Deletes all older messages from the queue if a newer
    * message of same type is already in the queue.
    * This method should not be called during message handling. */
  void removeRepetitions() { queue.removeRepetitions(); }

  /** Removes all messages from the queue */
  void clear();

  /** Returns if the queue contains no messages */
  bool isEmpty() const;

protected:
  /**
  * Copies a single message to another queue
  * @param message The number of the message
  * @param other The other queue.
  */
  void copyMessage(int message, MessageQueue& other);

  /** Gives the stream operator access to protected members */
  friend In& operator>>(In& stream, MessageQueue& messageQueue);

  /** Gives the stream operator access to protected members */
  friend Out& operator<<(Out& stream, const MessageQueue& messageQueue);
};

/**
 * Streaming operator that reads a MessageQueue from a stream.
 * @param stream The stream from which is read.
 * @param messageQueue The MessageQueue object.
 * @return The stream.
 */ 
In& operator>>(In& stream,MessageQueue& messageQueue);
 

/**
 * Streaming operator that writes a MessageQueue to a stream.
 * @param stream The stream to write on.
 * @param messageQueue The MessageQueue object.
 * @return The stream.
 */ 
Out& operator<<(Out& stream, const MessageQueue& messageQueue);

/**
* Streaming operator that writes a InMessage to another MessageQueue.
* @param message The InMessage to write.
* @param queue The MessageQueue object.
*/ 
void operator >> (InMessage& message, MessageQueue& queue);

#endif //__MesssageQueue_h_

/*
* Change Log:
*
* $Log: MessageQueue.h,v $
* Revision 1.1.1.1  2004/05/22 17:37:18  cvsadm
* created new repository GT2004_WM
*
* Revision 1.3  2004/01/10 18:04:58  loetzsch
* added MessageQueue::getStreamedSize()
*
* Revision 1.2  2003/12/06 23:23:55  loetzsch
* messages in a MessageQueue now contain
* - the team color of the robot which sent the message
* - the player number of the robot which sent the message
* - if the message was sent from a physical robot or not
*
* Revision 1.1  2003/10/07 10:13:24  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.5  2003/02/05 12:37:14  dueffert
* handleSpecificMessage added
*
* Revision 1.4  2003/01/13 22:01:27  timrie
* corrected doxygen comments
*
* Revision 1.3  2002/12/05 16:12:48  dueffert
* started implementing realtime sending
*
* Revision 1.2  2002/11/19 17:38:32  dueffert
* doxygen bugs corrected
*
* Revision 1.1  2002/09/10 15:53:59  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.3  2002/08/22 14:41:04  risler
* added some doxygen comments
*
* Revision 1.2  2002/08/08 16:38:47  loetzsch
* moved some members to the derived class LogPlayer because they are not
* needed in the normal MessageQueue
*
* Revision 1.1  2002/07/23 13:47:14  loetzsch
* - new streaming classes
* - new debug message handling
* - exchanged StaticQueue by MessageQueue
*
*/

