/** 
* @file KickSelectionTable.h
* Declaration of class KickSelectionTable.
*
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Jngel</A>
* @author Martin Ltzsch
*/

#ifndef _KickSelectionTable_h_
#define _KickSelectionTable_h_

#include "Tools/Math/Vector2.h"
#include "Platform/GTAssert.h"
#include "Representations/Motion/MotionRequest.h"

/**
* Represents the results and the preconditions of a single kick experiment
* @author Martin Ltzsch
*/
class KickCase
{
public:
  /** The relative x position of the ball when the kick was requested */
  double ballXR;

  /** The relative y position of the ball when the kick was requested */
  double ballYR;

  /** The time since the ball was seen last when the kick was requested */
  unsigned long ballTR;

  /** The relative x position of the ball when the kick was started */
  double ballXS;

  /** The relative y position of the ball when the kick was started */
  double ballYS;

  /** The time since the ball was seen last when the kick was started */
  unsigned long ballTS;

  /** The relative x position of the ball after the kick */
  double ballXA;

  /** The relative y position of the ball after the kick */
  double ballYA;

  /** The time since the kick was finished when the record was written */
  unsigned long tA;

};

/**
* @class KickSelectionTable
*
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Jngel</A>
* @author Martin Ltzsch
*
* Provides for a set of tables which provides a kick selection depending 
* on a desired kick angle and ball position.
*/
class KickSelectionTable
{
public:
  /** Constructor */
  KickSelectionTable();
  
  /** Destructor */
  ~KickSelectionTable();
  
  /** Ids for all possible actions */
  enum ActionID{
    nothing = 0,
      headLeft,
      headRight,
      anyLeft,
      anyRight,
      hookLeft,
      hookRight,
      leftPaw,
	    rightPaw,
      armLeft,
      armRight,
      pushSoft,
      pushStrong,
      chestSoft,
      chestStrong,
      putRight,
      putLeft,
      forwardKickFast,
      forwardKickHard,
      slapRight,
      slapLeft,
      headLeftSoft,
      headRightSoft,
      numberOfActions
  };
  
  /** returns names for action ids */
  static const char* getActionName(ActionID id)
  {
    switch (id)
    {
    case nothing: return "nothing";
    case headLeft: return "headLeft"; 
    case headRight: return "headRight";
    case anyLeft: return "anyLeft";
    case anyRight: return "anyRight";
    case hookLeft: return "hookLeft";
    case hookRight: return "hookRight";
    case leftPaw: return "leftPaw";
    case rightPaw: return "rightPaw";
    case armLeft: return "armLeft";
    case armRight: return "armRight";
    case pushSoft: return "pushSoft";
    case pushStrong: return "pushStrong";
    case chestSoft: return "chestSoft";
    case chestStrong: return "chestStrong";
    case putRight: return "putRight";
    case putLeft: return "putLeft";
    case forwardKickFast: return "forwardKickFast";
    case forwardKickHard: return "forwardKickHard";
    case slapRight: return "slapRight";
    case slapLeft: return "slapLeft";
    case headLeftSoft: return "headLeftSoft";
    case headRightSoft: return "headRightSoft";

    default: return "please edit KickSelectionTable::getActionName";
    }
  }
  
  /** returns short names for action ids (used by streaming operator)*/
  static const char* getShortActionName(ActionID id)
  {
    switch (id)
    {
    case nothing: return "no";
    case headLeft: return "head_l"; 
    case headRight: return "head_r";
    case anyLeft: return "any_l";
    case anyRight: return "any_r";
    case hookLeft: return "hook_l";
    case hookRight: return "hook_r";
    case leftPaw: return "l_paw";
    case rightPaw: return "r_paw";
    case armLeft: return "arm_l";
    case armRight: return "arm_r";
    case pushSoft: return "push_so";
    case pushStrong: return "pu_str";
    case chestSoft: return "che_so";
    case chestStrong: return "ch_str";
    case putRight: return "put_r";
    case putLeft: return "put_l";
    case forwardKickFast: return "forw_f";
    case forwardKickHard: return "forw_h";
    case slapRight: return "slap_r";
    case slapLeft: return "slap_l";
    case headLeftSoft: return "head_sl";
    case headRightSoft: return "head_sr";

    default: 
      ASSERT(false);
      return 0;
    }
  }
  
  /** IDs for different kick selection tables */
  enum KickSelectionTableID
  {
    inCenterOfField, 
      atLeftBorder, 
      atRightBorder, 
      atLeftOpponentBorder, 
      atRightOpponentBorder, 
      nearOwnGoal, 
      goalie,
      whenStuck,
      numberOfKickSelectionTableIDs
  };
  
  /** returns a name for an action table id */
  static const char* getKickSelectionTableIDName(KickSelectionTableID id)
  {
    switch (id)
    {
    case inCenterOfField: return "inCenterOfField";
    case atLeftBorder: return "atLeftBorder";
    case atRightBorder: return "atRightBorder";
    case atLeftOpponentBorder: return "atLeftOpponentBorder";
    case atRightOpponentBorder: return "atRightOpponentBorder";
    case nearOwnGoal: return "nearOwnGoal"; 
    case goalie: return "goalie";
    case whenStuck: return "whenStuck";
    default: 
      ASSERT(false);
      return 0;
    }
  }
  
  /** Returns an action id for a special action id */
  static const ActionID getActionIDFromSpecialActionID(SpecialActionRequest::SpecialActionID id)
  {
    switch (id)
    {
      case SpecialActionRequest::headLeft: return headLeft; 
      case SpecialActionRequest::headRight: return headRight;
      case SpecialActionRequest::anyLeft: return anyLeft;
      case SpecialActionRequest::anyRight: return anyRight;
      case SpecialActionRequest::hookLeft: return hookLeft;
      case SpecialActionRequest::hookRight: return hookRight;
      case SpecialActionRequest::leftPaw: return leftPaw;
      case SpecialActionRequest::rightPaw: return rightPaw;
      case SpecialActionRequest::armLeft: return armLeft;
      case SpecialActionRequest::armRight: return armRight;
      case SpecialActionRequest::pushSoft: return pushSoft;
      case SpecialActionRequest::pushStrong: return pushStrong;
      case SpecialActionRequest::chestSoft: return chestSoft;
      case SpecialActionRequest::putRight: return putRight;
      case SpecialActionRequest::putLeft: return putLeft;
      case SpecialActionRequest::forwardKickFast: return forwardKickFast;
      case SpecialActionRequest::forwardKickHard: return forwardKickHard;
      case SpecialActionRequest::slapRight: return slapRight;
      case SpecialActionRequest::slapLeft: return slapLeft;
      case SpecialActionRequest::headLeftSoft: return headLeftSoft;
      case SpecialActionRequest::headRightSoft: return headRightSoft;
      default: return nothing;
    }
  }
  
  /**  */ 
  enum{xRange = 20}; 
  
  /**  */ 
  enum{yRange = 40};
  
  /**  */ 
  enum{numberOfSectors = 12};
  
  /** 
  * Calculates the best kick based on the position of the ball relative to the 
  * robot and a target angle. 
  */
  ActionID retrieveKick(
    double ballOffsetX, double ballOffsetY, 
    double destinationAngle,
    KickSelectionTableID kickSelectionTableID
    ) const;
  
  /** The action tables */
  ActionID action[xRange][yRange][numberOfSectors][numberOfKickSelectionTableIDs];
  
  /** Returns a table id for a name */
  static KickSelectionTableID getTableIDFromName(const char* name);
  
  /** Returns an action id for a short name */
  static ActionID getActionIDFromShortName(const char* name);
  
  /** Loads a kick selection table from a file */
  void load(const char* fileName);
};

/**
* Streaming operator that reads KickSelectionTable from a stream.
* @param stream The stream from which is read.
* @param kickSelectionTable The KickSelectionTable object.
* @return The stream.
*/ 
In& operator>>(In& stream, KickSelectionTable& kickSelectionTable);

/**
* Streaming operator that writes KickSelectionTable to a stream.
* @param stream The stream to write on.
* @param kickSelectionTable The KickSelectionTable object.
* @return The stream.
*/ 
Out& operator<<(Out& stream, const KickSelectionTable& kickSelectionTable);

/**
* Streaming operator that reads KickCase from a stream.
* @param stream The stream from which is read.
* @param kickCase The KickCase object.
* @return The stream.
*/ 
In& operator>>(In& stream, KickCase& kickCase);

/**
* Streaming operator that writes KickCase to a stream.
* @param stream The stream to write on.
* @param kickCase The KickCase object.
* @return The stream.
*/ 
Out& operator<<(Out& stream, const KickCase& kickCase);


#endif   //  _KickSelectionTable_h_

/*
* Change log :
* 
* $Log: KickSelectionTable.h,v $
* Revision 1.7  2004/07/02 16:04:54  risler
* added headLeft/RightSoft
*
* Revision 1.6  2004/07/01 12:02:12  risler
* slapLeft/Right added
*
* Revision 1.5  2004/06/30 19:56:55  risler
* added when-stuck kick table
* handle-ball-... options use when-stuck kick table when average ball distance is low
*
* Revision 1.4  2004/06/30 08:37:53  lohmann
* Added forwardKickHard and forwardKickFast
*
* Revision 1.3  2004/06/20 15:22:35  risler
* increased kick editor usability:
* KickLogger now sends kick record via debug key instead of writing to file
* KickEditor automatically displays sent kick records
* KickCase moved to KickSelectionTable
*
* Revision 1.2  2004/05/27 14:37:27  loetzsch
* new actions in the kick selection table
*
* Revision 1.1.1.1  2004/05/22 17:35:53  cvsadm
* created new repository GT2004_WM
*
* Revision 1.9  2004/05/04 18:10:01  juengel
* Added short names for msh kicks
*
* Revision 1.8  2004/05/04 14:44:37  loetzsch
* added a few kicks
*
* Revision 1.7  2004/04/05 17:56:50  loetzsch
* merged the local German Open CVS of the aibo team humboldt with the tamara CVS
*
* Revision 1.3  2004/04/03 15:31:10  juengel
* bug fixed
*
* Revision 1.2  2004/04/03 01:29:25  loetzsch
* added arm kicks
*
* Revision 1.1.1.1  2004/03/31 11:17:00  loetzsch
* created ATH repository for german open 2004
*
* Revision 1.6  2004/03/28 16:02:01  juengel
* beautified
*
* Revision 1.5  2004/03/27 20:33:22  loetzsch
* added table id for goalie
*
* Revision 1.4  2004/03/19 21:19:28  altmeyer
* SideHeadKick and SideHeadKick2 changed and renamed to SideHeadKick L and SideHeadKickR
* a few ATH ERS7-kicks removed
*
* Revision 1.3  2004/03/17 19:55:55  juengel
* Comments added.
*
* Revision 1.2  2004/03/16 16:26:16  altmeyer
* added new ATH-ERS7 kicks
*
* Revision 1.1  2004/03/16 14:00:23  juengel
* Integrated Improvments from "Gnne"
* -ATH2004ERS7Behavior
* -ATHHeadControl
* -KickSelectionTable
* -KickEditor
*
* Revision 1.11  2004/03/11 17:29:11  juengel
* Added kicks with head.
*
* Revision 1.10  2004/03/11 11:40:28  loetzsch
* renamed fastHeadKickRight to headKickRight
*
* Revision 1.9  2004/03/10 18:41:00  loetzsch
* first functioning version
*
* Revision 1.8  2004/03/10 14:06:04  juengel
* Implemented retrieveKick.
*
* Revision 1.7  2004/03/09 18:48:58  juengel
* Added ActionID "nothing".
*
* Revision 1.6  2004/03/09 18:43:45  juengel
* Added ActionID "nothing".
*
* Revision 1.5  2004/03/09 18:41:09  loetzsch
* no message
*
* Revision 1.4  2004/03/09 14:56:37  loetzsch
* bug fix
*
* Revision 1.3  2004/03/09 14:54:09  loetzsch
* changed comments
*
* Revision 1.2  2004/03/09 14:45:27  loetzsch
* automated generation of kick-selection-symbols.xml
*
* Revision 1.1  2004/03/09 13:40:30  juengel
* Moved KickSelectionTable to Tools.
*
* Revision 1.1  2004/03/09 01:14:36  juengel
* Added class KickSelectionTable.
*
*/
