/** 
* @file KickSelectionTable.cpp
* Implementation of class KickSelectionTable.
*
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Jngel</A>
*/

#include "KickSelectionTable.h"
#include "Platform/SystemCall.h"
#include "Tools/Math/Common.h"
#include "Tools/Streams/InStreams.h"
#include "Tools/Debugging/Debugging.h"
#include "Tools/Location.h"


KickSelectionTable::KickSelectionTable()
{
  memset(action,0,sizeof(action));
}

KickSelectionTable::~KickSelectionTable()
{
}

KickSelectionTable::KickSelectionTableID KickSelectionTable::getTableIDFromName(const char* name)
{
  for (int i=0;i<numberOfKickSelectionTableIDs;i++)
  {
    if (strcmp(name,getKickSelectionTableIDName((KickSelectionTableID)i))==0)
    {
      return (KickSelectionTableID)i;
    }
  }
  
  return inCenterOfField;
}

KickSelectionTable::ActionID KickSelectionTable::getActionIDFromShortName(const char* name)
{
  for (int i=0;i<numberOfActions;i++)
  {
    if (strcmp(name,getShortActionName((ActionID)i))==0)
    {
      return (ActionID)i;
    }
  }
  
  return nothing;
}

void KickSelectionTable::load(const char* fileName)
{
  InTextFile f(getLocation().getModelFilename(fileName));
  if (f.exists())
    f >> *this;
}

KickSelectionTable::ActionID KickSelectionTable::retrieveKick
(
 double ballOffsetX, double ballOffsetY,
 double destinationAngle,
 KickSelectionTableID kickSelectionTableID
 ) const
{
  int x,y;
  int sector;
  sector = (int)((normalize(fromDegrees(destinationAngle) + pi + pi2 / KickSelectionTable::numberOfSectors / 2) + pi) / pi2 * KickSelectionTable::numberOfSectors);
  x = (int)(ballOffsetX / 10.0);
  y = (int)(ballOffsetY / 10.0) + KickSelectionTable::yRange / 2;
  
  //  OUTPUT(idText,text,"bx: " << ballOffsetX << ", by: " << ballOffsetY
  //    << ", x: " << x << ", y: " << y << ", a: " << destinationAngle 
  //    << ", s: " << sector << ", t: " << kickSelectionTableID);
  
  if(x < 0 || x >= KickSelectionTable::xRange ||
    y < 0 || y >= KickSelectionTable::yRange) return nothing;
  
  
  
  return action[x][y][sector][kickSelectionTableID];
}

In& operator>>(In& stream, KickSelectionTable& kickSelectionTable)
{
  KickSelectionTable::KickSelectionTableID tableID;
  KickSelectionTable::ActionID actionID;
  int x,y,s;
  char buf[200];
  memset(kickSelectionTable.action,0,sizeof(kickSelectionTable.action));
  while (1)
  {
    stream >> buf;
    if (strcmp(buf,"___")==0) break;
    tableID = kickSelectionTable.getTableIDFromName(buf);
    while(1)
    {
      stream >> buf;
      if (strcmp(buf,"___")==0) break;
      
      x = atoi(buf);
      
      stream >> y >> s >> buf;
      actionID = kickSelectionTable.getActionIDFromShortName(buf);
      kickSelectionTable.action[x][y][s][tableID] = actionID;
    }
  }
  return stream;
}

Out& operator<<(Out& stream, const KickSelectionTable& kickSelectionTable)
{
  int t,x,y,s;
  for (t=0; t<KickSelectionTable::numberOfKickSelectionTableIDs; t++)
  {
    stream << kickSelectionTable.getKickSelectionTableIDName((KickSelectionTable::KickSelectionTableID)t);
    
    for (x = 0; x< kickSelectionTable.xRange; x++)
    {
      for (y = 0; y< kickSelectionTable.yRange; y++)
      {
        for (s = 0; s< kickSelectionTable.numberOfSectors; s++)
        {
          if (kickSelectionTable.action[x][y][s][t] != KickSelectionTable::nothing)
          {
            stream << x << y << s << kickSelectionTable.getShortActionName(
              kickSelectionTable.action[x][y][s][t]);
          }
        }
      }
    }
    stream << "___";
  }
  stream << "___";
  return stream;
}

In& operator>>(In& stream,KickCase& kickCase)
{
  stream.read(&kickCase,sizeof(KickCase));
  return stream;
}
 
Out& operator<<(Out& stream, const KickCase& kickCase)
{
  stream.write(&kickCase,sizeof(KickCase));
  return stream;
}

/*
* Change log :
* 
* $Log: KickSelectionTable.cpp,v $
* Revision 1.5  2004/06/30 11:07:27  cesarz
* kick selection table is location based now for Lisboa penalty shooter
*
* Revision 1.4  2004/06/24 13:57:10  risler
* clear table when reading from stream
*
* Revision 1.3  2004/06/20 15:22:35  risler
* increased kick editor usability:
* KickLogger now sends kick record via debug key instead of writing to file
* KickEditor automatically displays sent kick records
* KickCase moved to KickSelectionTable
*
* Revision 1.2  2004/06/17 14:27:21  risler
* load table only if file exists
*
* Revision 1.1.1.1  2004/05/22 17:35:53  cvsadm
* created new repository GT2004_WM
*
* Revision 1.1  2004/03/16 14:00:23  juengel
* Integrated Improvments from "Gnne"
* -ATH2004ERS7Behavior
* -ATHHeadControl
* -KickSelectionTable
* -KickEditor
*
* Revision 1.4  2004/03/10 18:41:00  loetzsch
* first functioning version
*
* Revision 1.3  2004/03/10 14:06:04  juengel
* Implemented retrieveKick.
*
* Revision 1.2  2004/03/09 18:43:54  juengel
* Added ActionID "nothing".
*
* Revision 1.1  2004/03/09 13:40:30  juengel
* Moved KickSelectionTable to Tools.
*
* Revision 1.1  2004/03/09 01:14:36  juengel
* Added class KickSelectionTable.
*
*/
