/**
 * @file Tools/Field.h
 * 
 * This file contains a class representing the field boundary.
 *
 * @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
 */

#ifndef __Field_h_
#define __Field_h_

#include "Representations/Perception/LinesPercept.h"
#include "Representations/Perception/ObstaclesPercept.h"
#include "Representations/Cognition/PlayerPoseCollection.h"
#include "Tools/Boundary.h"
#include "Tools/Debugging/DebugDrawings.h"
#include "Tools/Math/Pose2D.h"

/**
 * The class represents the field area.
 */
class Field : public Boundary<double>
{
  private:
		/**
		 * This is a collection of line- or boundary segments with start-Pose2D and length.
		 */
    class Table
    {
      private:
        void free()
        {
          if(numberOfEntries) 
          {
            delete [] corner; 
            delete [] length;
            numberOfEntries = 0;
          }
        }

      public:
        int numberOfEntries; /**< The number of corners in the table. */
        Pose2D* corner; /**< The field corners. */
        double* length; /**< The lengths of the border segments starting at a corresponding corner. */
        int index;

        Table() {numberOfEntries = 0;}

        ~Table() {free();}

        void setSize(int size)
        {
          if(size != numberOfEntries)
          {
            free();
            if(size)
            {          
              numberOfEntries = size;
              corner = new Pose2D[size];
              length = new double[size];
            }
          }
          index = 0;
        }

        void push(const Pose2D& p, double l)
        {
          corner[index] = p;
          length[index++] = l;
        }
    };

    Table boundary,
          lines[LinesPercept::numberOfLineTypes + 4];

    void initBoundary(Table& table);
    void initLines(Table& table, Table& xTable, Table& yTable);
    void initBorder(Table& table);
    void initOpponentGoal(Table& table);
    void initOwnGoal(Table& table);
    void initSimpleLines(Table& table);

    void addCoords(Table& table,int number,double* x,double* y);
    void addCoords(Table& table,Table& xTable,Table& yTable,int number,double* x,double* y);
    void addPlayer(const Pose2D& pose);

  public:
    /**
     * Constructor.
     */
    Field();

    /**
     * The function checks whether a point is inside the field.
     * @param v The point.
     * @return Is the point inside?
     */
    bool isReallyInside(const Vector2<double>& v) const;

    /**
     * The function clips a point to the field.
     * @param v The point.
     * @return How far was the point moved?
     */
    double clip(Vector2<double>& v) const;

    /**
     * The function returns the point on a line of a certain type closest to given a point.
     * @param v The reference point.
     * @param type The type of the lines.
     * @return The point on a line.
     */
    Vector2<double> getClosestPoint(const Vector2<double>& v,
                                    LinesPercept::LineType type) const;

    /**
     * The function returns the point on a line of a certain type closest to given a point.
     * @param p The reference point and the rotation of the line.
     * @param numberOfRotations The number of discretizations of line rotations.
     * @param type The type of the lines.
     * @return The point on a line.
     */
    Vector2<double> getClosestPoint(const Pose2D& p, int numberOfRotations,
                                    LinesPercept::LineType type) const;
    /**
     * The function returns the distance between a point and the closest point on a line of a certain type.
     * @param v The reference point.
     * @param type The type of the lines.
     * @return The distance.
     */
    double getClosestDistance(const Vector2<double>& v,
                              LinesPercept::LineType type) const;

    /**
     * The function returns the distance between a point and the closest point on a line of a certain type in a certain direction.
     * @param pose The reference point and direction.
     * @param type The type of the lines.
     * @return The distance. It is -1 if no line of that type exists in the certain direction.
     */
    double getDistance(const Pose2D& pose,
                       LinesPercept::LineType type) const;

    /**
     * The function returns the distance between a point and the closest point on a line in a certain direction.
     * @param pose The reference point and direction.
     * @param ignoreFieldLines The field lines are ignored during the search.
     * @return The distance. It is -1 if no line of that type exists in the certain direction.
     */
    double getDistance(const Pose2D& pose,bool ignoreFieldLines = false) const;

    /**
     * The function returns the distance to the robots own penalty area for a given pose.
     * @param pose The reference point and direction (robot pose).
     * @return The distance. It is -1 if no line of that type exists in the certain direction.
     */
    double getDistanceToOwnPenaltyArea(const Pose2D& pose) const;

    /**
     * The function returns the distance between a point and the closest obstacle point in a certain direction.
     * @param pose The reference point and direction.
     * @return The distance. It is -1 if no line of that type exists in the certain direction.
     */
    double getObstacleDistance(const Pose2D& pose, ObstaclesPercept::ObstacleType& obstacleType) const;

    /**
     * The function places all robots as obstacles in an internal data structure.
     * @param ppc The poses of all other players on the field.
     */
    void placePlayers(const PlayerPoseCollection& ppc);

    /**
     * The function returns a random pose inside the field.
     * @return The random pose.
     */
    Pose2D randomPose() const;

    /**
     * The function draws all lines of a cetain type.
     * It is use to control the correctness of the model.
     * @param color The color the lines will be drawn in.
     * @param type The type of the lines.
     */
    void draw(const Drawings::Color color, LinesPercept::LineType type) const;
};

#endif // __Field_h_

/*
 * Change log :
 * 
 * $Log: Field.h,v $
 * Revision 1.2  2004/06/24 18:26:38  roefer
 * Lines table redesign, should not influence the performance of GT2003SL
 *
 * Revision 1.1.1.1  2004/05/22 17:35:50  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.7  2004/03/16 14:00:23  juengel
 * Integrated Improvments from "Gnne"
 * -ATH2004ERS7Behavior
 * -ATHHeadControl
 * -KickSelectionTable
 * -KickEditor
 *
 * Revision 1.2  2004/03/15 17:11:41  hoffmann
 * - added ATH2004HeadControl
 * - added ATH2004LEDControl
 * - headmotiontester shows "tilt2"
 * - motion process updates odometry while no new robotPose is received, added to motion request
 * - some ui adjustments
 * - added member function to "field" to find out if robot is in own penalty area for use in the obstacles locator
 *
 * Revision 1.6  2004/03/01 11:47:16  juengel
 * Moved enum ObstacleType to class ObstaclesPercept.
 *
 * Revision 1.5  2004/02/28 14:00:43  juengel
 * Added ObstacleType to ObstaclesModel.
 *
 * Revision 1.4  2003/12/03 23:59:26  roefer
 * Compatibility with VC2003.NET, GUI works now
 *
 * Revision 1.3  2003/11/29 07:41:28  roefer
 * Possible heap problem fixed
 *
 * Revision 1.2  2003/10/23 15:41:40  roefer
 * Oracled obstacle model added
 *
 * Revision 1.1  2003/10/07 10:13:20  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.3  2003/09/26 15:28:10  juengel
 * Renamed DataTypes to representations.
 *
 * Revision 1.2  2003/09/26 11:40:39  juengel
 * - sorted tools
 * - clean-up in DataTypes
 *
 * Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.6  2003/05/23 15:51:09  roefer
 * Doxygen comment corrected
 *
 * Revision 1.5  2003/05/11 23:56:15  dueffert
 * doxygen bugs fixed
 *
 * Revision 1.4  2003/05/05 11:55:59  dueffert
 * spelling corrected
 *
 * Revision 1.3  2003/04/16 07:00:17  roefer
 * Bremen GO checkin
 *
 * Revision 1.3  2003/04/08 18:23:07  roefer
 * LinesHeadControl speed increased
 *
 * Revision 1.2  2003/04/06 21:03:29  roefer
 * LinesHeadControl, first draft
 *
 * Revision 1.1  2003/03/31 21:01:44  roefer
 * Moved class Field to Tools
 *
 * Revision 1.6  2002/12/13 11:25:19  dueffert
 * doxygen bugs fixed
 *
 * Revision 1.5  2002/12/12 22:09:41  roefer
 * Progress in LinesSelfLocator
 *
 * Revision 1.4  2002/11/26 12:26:38  dueffert
 * doxygen docu added
 *
 * Revision 1.3  2002/11/12 10:49:02  juengel
 * New debug drawing macros - second step
 * -moved /Tools/Debugging/PaintMethods.h and . cpp
 *  to /Visualization/DrawingMethods.h and .cpp
 * -moved DebugDrawing.h and .cpp from /Tools/Debugging/
 *  to /Visualization
 *
 * Revision 1.2  2002/09/22 18:40:53  risler
 * added new math functions, removed GTMath library
 *
 * Revision 1.1  2002/09/10 15:36:15  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.2  2002/08/22 14:41:03  risler
 * added some doxygen comments
 *
 * Revision 1.1  2002/06/02 23:21:09  roefer
 * Single color table and progress in LinesSelfLocator
 *
 *
 */
