/**
 * @file Tools/Field.cpp
 * 
 * This file contains the implementation of a class representing the field boundary.
 *
 * @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
 */

#include "Field.h"
#include "Tools/Debugging/Debugging.h"
#include "Tools/FieldDimensions.h"
#include "Tools/Player.h"
#include "Tools/RobotConfiguration.h"
#include "Platform/GTAssert.h"

Field::Field()
: Boundary<double>(0,0)
{
  add(Vector2<double>(xPosOpponentGoal,yPosLeftSideline));
  add(Vector2<double>(xPosOwnGoal,yPosRightSideline));
  initBoundary(boundary);
  initLines(lines[LinesPercept::field], lines[LinesPercept::numberOfLineTypes + 2], lines[LinesPercept::numberOfLineTypes + 3]);
  initBorder(lines[LinesPercept::border]);
  initOwnGoal(lines[LinesPercept::yellowGoal]); // for red player
  initOpponentGoal(lines[LinesPercept::skyblueGoal]); // for red player
  initSimpleLines(lines[LinesPercept::numberOfLineTypes]);
}

void Field::initBoundary(Table& table)
{
  double x[] =
  {
    xPosOwnGroundline,
    xPosOwnSideCorner,
    xPosOpponentSideCorner,
    xPosOpponentGroundline,
    xPosOpponentGroundline,
    xPosOpponentGoal - 100,
    xPosOpponentGoal - 100,
    xPosOpponentGroundline,
    xPosOpponentGroundline,
    xPosOpponentSideCorner,
    xPosOwnSideCorner,
    xPosOwnGroundline,
    xPosOwnGroundline,
    xPosOwnGoal + 100,
    xPosOwnGoal + 100,
    xPosOwnGroundline,
    xPosOwnGroundline
  },
         y[] =
  {
    yPosLeftGroundline,
    yPosLeftSideline,
    yPosLeftSideline,
    yPosLeftGroundline,
    yPosLeftGoal,
    yPosLeftGoal,
    yPosRightGoal,
    yPosRightGoal,
    yPosRightGroundline,
    yPosRightSideline,
    yPosRightSideline,
    yPosRightGroundline,
    yPosRightGoal,
    yPosRightGoal,
    yPosLeftGoal,
    yPosLeftGoal,
    yPosLeftGroundline
  };

  table.setSize(sizeof(x) / sizeof(x[0]) - 1);
  addCoords(table,table.numberOfEntries,x,y);
}

void Field::initLines(Table& table, Table& xTable, Table& yTable)
{
  double x[] =
  {
    // own ground line
    xPosOwnGroundline,
    xPosOwnGroundline,
    xPosOwnGroundline - 25,
    xPosOwnGroundline - 25,

    // opponent ground line
    xPosOpponentGroundline,
    xPosOpponentGroundline,
    xPosOpponentGroundline + 25,
    xPosOpponentGroundline + 25,

    // own inner penalty line border
    xPosOwnGroundline,
    xPosOwnPenaltyArea - 25,
    xPosOwnPenaltyArea - 25,
    xPosOwnGroundline,

    // own outer penalty line border
    xPosOwnGroundline,
    xPosOwnPenaltyArea,
    xPosOwnPenaltyArea,
    xPosOwnGroundline,

    // opponent inner penalty line border
    xPosOpponentGroundline,
    xPosOpponentPenaltyArea + 25,
    xPosOpponentPenaltyArea + 25,
    xPosOpponentGroundline,

    // opponent outer penalty line border
    xPosOpponentGroundline,
    xPosOpponentPenaltyArea,
    xPosOpponentPenaltyArea,
    xPosOpponentGroundline,

    // inner center, taken from GT2003.scn
    -12.5,-12.5,-52.6,-97.2,-127,-137.5,-127,-97.2,-52.6,-12.5,
    12.5,12.5,52.6,97.2,127,137.5,127,97.2,52.6,12.5,

    // outer circle and center line, taken from GT2003.scn
    -12.5,-12.5,-62.2,-114.9,-150.1,-162.5,-150.1,-114.9,-62.2,-12.5,-12.5,
    12.5,12.5,62.2,114.9,150.1,162.5,150.1,114.9,62.2,12.5,12.5
  },
         y[] =
  {
    // own ground line
    yPosLeftGoal,
    yPosRightGoal,
    yPosLeftGoal,
    yPosRightGoal,

    // opponent ground line
    yPosLeftGoal,
    yPosRightGoal,
    yPosLeftGoal,
    yPosRightGoal,

    // own inner penalty line border
    yPosLeftPenaltyArea - 25,
    yPosLeftPenaltyArea - 25,
    yPosRightPenaltyArea + 25,
    yPosRightPenaltyArea + 25,

    // own outer penalty line border
    yPosLeftPenaltyArea,
    yPosLeftPenaltyArea,
    yPosRightPenaltyArea,
    yPosRightPenaltyArea,

    // opponent inner penalty line border
    yPosLeftPenaltyArea - 25,
    yPosLeftPenaltyArea - 25,
    yPosRightPenaltyArea + 25,
    yPosRightPenaltyArea + 25,

    // opponent outer penalty line border
    yPosLeftPenaltyArea,
    yPosLeftPenaltyArea,
    yPosRightPenaltyArea,
    yPosRightPenaltyArea,

    // inner center, taken from GT2003.scn
    137.5,-137.5,-127,-97.2,-52.6,0,52.6,97.2,127,137.5,
    137.5,-137.5,-127,-97.2,-52.6,0,52.6,97.2,127,137.5,

    // outer circle and center line, taken from GT2003.scn
    yPosRightSideline,-162.5,-150.1,-114.9,-62.2,0,62.2,114.9,150.1,162.5,yPosLeftSideline,
    yPosRightSideline,-162.5,-150.1,-114.9,-62.2,0,62.2,114.9,150.1,162.5,yPosLeftSideline
  };

  table.setSize(54);
  xTable.setSize(54);
  yTable.setSize(54);

  // ground lines
  for(int i = 0; i < 4; ++i)
    addCoords(table,xTable,yTable,1,x + 2 * i,y + 2 * i);

  // own penalty
  addCoords(table,xTable,yTable,3,x + 8,y + 8);
  addCoords(table,xTable,yTable,3,x + 12,y + 12);

  // opponent penalty
  addCoords(table,xTable,yTable,3,x + 16,y + 16);
  addCoords(table,xTable,yTable,3,x + 20,y + 20);

  // inner center
  addCoords(table,xTable,yTable,9,x + 24,y + 24);
  addCoords(table,xTable,yTable,9,x + 34,y + 34);

  // outer center and center line
  addCoords(table,xTable,yTable,10,x + 44,y + 44);
  addCoords(table,xTable,yTable,10,x + 55,y + 55);

  xTable.numberOfEntries = xTable.index;
  yTable.numberOfEntries = yTable.index;
}

void Field::initSimpleLines(Table& table)
{
  double x[] =
  {
    xPosOwnGroundline,
    xPosOwnGroundline,

    xPosOpponentGroundline,
    xPosOpponentGroundline,

    xPosOwnGroundline + 12,
    xPosOwnPenaltyArea,
    xPosOwnPenaltyArea,
    xPosOwnGroundline + 12,

    xPosOpponentGroundline - 12,
    xPosOpponentPenaltyArea,
    xPosOpponentPenaltyArea,
    xPosOpponentGroundline - 12,

    xPosHalfWayLine,
    xPosHalfWayLine
  },
         y[] =
  {
    yPosLeftGoal,
    yPosRightGoal,

    yPosLeftGoal,
    yPosRightGoal,

    yPosLeftPenaltyArea,
    yPosLeftPenaltyArea,
    yPosRightPenaltyArea,
    yPosRightPenaltyArea,

    yPosLeftPenaltyArea,
    yPosLeftPenaltyArea,
    yPosRightPenaltyArea,
    yPosRightPenaltyArea,

    yPosRightSideline,yPosLeftSideline
  };

  table.setSize(9);

  // ground lines
  addCoords(table,1,x,y);
  addCoords(table,1,x + 2,y + 2);

  // penalty area
  addCoords(table,3,x + 4,y + 4);
  addCoords(table,3,x + 8,y + 8);

  // halfWay line
  addCoords(table,1,x + 12,y + 12);
}

void Field::initBorder(Table& table)
{
  double x[] =
  {
    xPosOwnGoalpost,
    xPosOwnGroundline,
    xPosOwnGroundline,
    xPosOwnSideCorner,
    xPosOpponentSideCorner,
    xPosOpponentGroundline,
    xPosOpponentGroundline,
    xPosOpponentGoalpost,
    // opponent goal
    xPosOpponentGoalpost,
    xPosOpponentGroundline,
    xPosOpponentGroundline,
    xPosOpponentSideCorner,
    xPosOwnSideCorner,
    xPosOwnGroundline,
    xPosOwnGroundline,
    xPosOwnGoalpost
    // own goal
  },
         y[] =
  {
    yPosLeftGoal,
    yPosLeftGoal,
    yPosLeftGroundline,
    yPosLeftSideline,
    yPosLeftSideline,
    yPosLeftGroundline,
    yPosLeftGoal,
    yPosLeftGoal,
    // opponent goal
    yPosRightGoal,
    yPosRightGoal,
    yPosRightGroundline,
    yPosRightSideline,
    yPosRightSideline,
    yPosRightGroundline,
    yPosRightGoal,
    yPosRightGoal
    // own goal
  };

  table.setSize(sizeof(x) / sizeof(x[0]) - 2);
  addCoords(table,7,x,y);
  addCoords(table,7,x+8,y+8);
}

void Field::initOpponentGoal(Table& table)
{
  double x[] =
  {
    xPosOpponentGoalpost,
    xPosOpponentGoal,
    xPosOpponentGoal,
    xPosOpponentGoalpost
  },
         y[] =
  {
    yPosLeftGoal,
    yPosLeftGoal,
    yPosRightGoal,
    yPosRightGoal
  };

  table.setSize(sizeof(x) / sizeof(x[0]) - 1);
  addCoords(table,table.numberOfEntries,x,y);
}

void Field::initOwnGoal(Table& table)
{
  double x[] =
  {
    xPosOwnGoalpost,
    xPosOwnGoal,
    xPosOwnGoal,
    xPosOwnGoalpost
  },
         y[] =
  {
    yPosRightGoal,
    yPosRightGoal,
    yPosLeftGoal,
    yPosLeftGoal
  };

  table.setSize(sizeof(x) / sizeof(x[0]) - 1);
  addCoords(table,table.numberOfEntries,x,y);
}

void Field::addCoords(Table& table,int number,double* x,double* y)
{
  Vector2<double> v1(x[0],y[0]);
  for(int i = 1; i <= number; ++i)
  {
    Vector2<double> v2(x[i],y[i]),
                    v3 = v2 - v1;
    table.push(Pose2D(atan2(v3.y,v3.x),v1), v3.abs());
    v1 = v2;
  }
}

void Field::addCoords(Table& table,Table& xTable,Table& yTable,int number,double* x,double* y)
{
  Vector2<double> v1(x[0],y[0]);
  for(int i = 1; i <= number; ++i)
  {
    Vector2<double> v2(x[i],y[i]),
                    v3 = v2 - v1;
    if(fabs(v3.x) > 1)
      xTable.push(Pose2D(atan2(v3.y,v3.x),v1), v3.abs());
    if(fabs(v3.y) > 1)
      yTable.push(Pose2D(atan2(v3.y,v3.x),v1), v3.abs());
    table.push(Pose2D(atan2(v3.y,v3.x),v1), v3.abs());
    v1 = v2;
  }
}

bool Field::isReallyInside(const Vector2<double>& v) const
{
  if(!isInside(v))
    return false;
  for(int i = 0; i < boundary.numberOfEntries; ++i)
  {
    Pose2D diff = Pose2D(v) - boundary.corner[i];
    if(diff.translation.y > 0 && diff.translation.x >= 0 && diff.translation.x < boundary.length[i])
      return false;
  }
  return true;
}

double Field::clip(Vector2<double>& v) const
{
  if(isReallyInside(v))
    return 0;
  else
  {
    Vector2<double> old = v,
                          v2;
    double minDist = 100000;
    for(int i = 0; i < boundary.numberOfEntries; ++i)
    {
      Vector2<double> diff = (Pose2D(old) - boundary.corner[i]).translation;
      if(diff.x < 0)
        v2 = boundary.corner[i].translation;
      else if(diff.x > boundary.length[i])
        v2 = (boundary.corner[i] + Pose2D(Vector2<double>(boundary.length[i],0))).translation;
      else
        v2 = (boundary.corner[i] + Pose2D(Vector2<double>(diff.x,0))).translation;
      double dist = (old - v2).abs();
      if(minDist > dist)
      {
        minDist = dist;
        v = v2;
      }
    }
    return (v - old).abs();
  }
}

Vector2<double> Field::getClosestPoint(const Vector2<double>& v,
                                       LinesPercept::LineType type) const
{
  const Table& table = lines[type];
  Vector2<double> vMin,
                  v2;
  double minDist = 100000;
  for(int i = 0; i < table.numberOfEntries; ++i)
  {
    Vector2<double> diff = (Pose2D(v) - table.corner[i]).translation;
    if(diff.x < 0)
      v2 = table.corner[i].translation;
    else if(diff.x > table.length[i])
      v2 = (table.corner[i] + Pose2D(Vector2<double>(table.length[i],0))).translation;
    else
      v2 = (table.corner[i] + Pose2D(Vector2<double>(diff.x,0))).translation;
    Vector2<double> vDiff = v2 - v;
    double dist = vDiff.abs();
    if(minDist > dist)
    {
      minDist = dist;
      vMin = v2;
    }
  }
  return vMin;
}

Vector2<double> Field::getClosestPoint(const Pose2D& p, int numberOfRotations,
                                       LinesPercept::LineType type) const
{
  // target angle -> target index
  double r = p.rotation / pi2 * numberOfRotations + 0.5;
  if(r < 0)
    r += numberOfRotations;
  int targetRot = int(r);
  ASSERT(targetRot >= 0 && targetRot < numberOfRotations);
  int targetRot2 = targetRot - numberOfRotations / 2;
  if(targetRot2 < 0)
    targetRot2 += numberOfRotations;
  const Table& table = lines[type];
  Vector2<double> vMin,
                  v2;
  double minDist = 100000;
  for(int i = 0; i < table.numberOfEntries; ++i)
  {
    // angle -> index
    double r = (table.corner[i].rotation - pi_2) / pi2 * numberOfRotations + 0.5;
    if(r < 0)
      r += numberOfRotations;
    int rot = int(r);
    ASSERT(rot >= 0 && rot < numberOfRotations);

    // index must be target index. Exception: field lines can be seen from both sides
    if(rot == targetRot || type == LinesPercept::field && rot == targetRot2)
    {
      double a = normalize(table.corner[i].rotation + pi_2);
      Vector2<double> diff = (p - table.corner[i]).translation;
      if(diff.x < 0)
        v2 = table.corner[i].translation;
      else if(diff.x > table.length[i])
        v2 = (table.corner[i] + Pose2D(Vector2<double>(table.length[i],0))).translation;
      else
        v2 = (table.corner[i] + Pose2D(Vector2<double>(diff.x,0))).translation;
      Vector2<double> vDiff = v2 - p.translation;
      double dist = vDiff.abs();
      if(minDist > dist)
      {
        minDist = dist;
        vMin = v2;
      }
    }
  }
  return vMin;
}

double Field::getClosestDistance(const Vector2<double>& v,
                                 LinesPercept::LineType type) const
{
  return (v - getClosestPoint(v,type)).abs();
}

double Field::getDistance(const Pose2D& pose,
                          LinesPercept::LineType type) const
{
  const Table& table = lines[type];
  double minDist = 100000;
  for(int i = 0; i < table.numberOfEntries; ++i)
  {
    Vector2<double> v1 = (table.corner[i] - pose).translation,
                    v2 = (table.corner[i] + Pose2D(Vector2<double>(table.length[i],0)) 
                          - pose).translation;
    if(v1.y < 0 && v2.y > 0 ||
       v1.y > 0 && v2.y < 0)
    {
      double dist = v1.x + (v2.x - v1.x) * -v1.y / (v2.y - v1.y);
      if(dist >= 0 && dist < minDist)
        minDist = dist;
    }
  }
  return minDist == 100000 ? -1 : minDist;
}

double Field::getDistance(const Pose2D& pose,bool ignoreFieldLines) const
{
  double minDist = 100000;
  for(int i = LinesPercept::numberOfLineTypes - (ignoreFieldLines ? 1 : 0);
      i > 0; --i) // use simpleLines instead of lines
  {
    double dist = getDistance(pose,LinesPercept::LineType(i));
    if(dist != -1 && dist < minDist)
      minDist = dist;
  } 
  return minDist == 100000 ? -1 : minDist;
}




double Field::getDistanceToOwnPenaltyArea(const Pose2D& pose) const
{
  const Table& table = lines[LinesPercept::numberOfLineTypes];
  double minDist = 100000;
  for(int i = 2; i < 5; ++i) // these are the 
  {
    Vector2<double> v1 = (table.corner[i] - pose).translation,
                    v2 = (table.corner[i] + Pose2D(Vector2<double>(table.length[i],0)) 
                          - pose).translation;
    if(v1.y < 0 && v2.y > 0 ||
       v1.y > 0 && v2.y < 0)
    {
      double dist = v1.x + (v2.x - v1.x) * -v1.y / (v2.y - v1.y);
      if(dist >= 0 && dist < minDist)
        minDist = dist;
    }
  }
  return minDist == 100000 ? -1 : minDist;
}
  

double Field::getObstacleDistance(const Pose2D& pose, ObstaclesPercept::ObstacleType& obstacleType) const
{
  double minDist = getDistance(pose,LinesPercept::LineType(LinesPercept::numberOfLineTypes + 1));
  LinesPercept::LineType minDistType = LinesPercept::LineType(LinesPercept::numberOfLineTypes + 1);
  if(minDist == -1)
    minDist = 100000;
 
  for(int i = LinesPercept::numberOfLineTypes - 1; 
    i > 0; // everything but field lines 
    --i)
  {
    double dist = getDistance(pose,LinesPercept::LineType(i));
    if(dist != -1 && dist < minDist)
    {
      minDist = dist;
      minDistType = LinesPercept::LineType(i);
    }
  } 
  
  //minDist = getDistanceToOwnPenaltyArea(pose);
    
  switch(minDistType)
  {
  case LinesPercept::border: obstacleType = ObstaclesPercept::border; break;
  case LinesPercept::redRobot: obstacleType = ObstaclesPercept::opponent; break;
  case LinesPercept::blueRobot: obstacleType = ObstaclesPercept::teammate; break;
  case LinesPercept::yellowGoal: obstacleType = ObstaclesPercept::goal; break;
  case LinesPercept::skyblueGoal: obstacleType = ObstaclesPercept::goal; break;
  default: obstacleType = ObstaclesPercept::unknown;
  }
  return minDist == 100000 ? -1 : minDist;
}

void Field::placePlayers(const PlayerPoseCollection& ppc)
{
  lines[LinesPercept::numberOfLineTypes + 1].setSize(
    (ppc.numberOfOwnPlayers + ppc.numberOfOpponentPlayers) * 4); // for player boundaries
  for(int i = 0; i < ppc.numberOfOwnPlayers; ++i)
    addPlayer(ppc.getOwnPlayerPose(i).getPose());
  for(int j = 0; j < ppc.numberOfOpponentPlayers; ++j)
    addPlayer(ppc.getOpponentPlayerPose(j).getPose());
}

void Field::addPlayer(const Pose2D& pose)
{
  // @todo These should be the dimensions of the other robot!
  const RobotDimensions& robotDimensions = getRobotConfiguration().getRobotDimensions();
  double front = robotDimensions.bodyLength / 2 - robotDimensions.lengthNeckToBodyCenter,
         back = front - robotDimensions.bodyLength,
         side = robotDimensions.overallBodyWidth / 2;
  Vector2<double> fl = (pose + Pose2D(Vector2<double>(front, side))).translation,
                  fr = (pose + Pose2D(Vector2<double>(front, -side))).translation,
                  bl = (pose + Pose2D(Vector2<double>(back, side))).translation,
                  br = (pose + Pose2D(Vector2<double>(back, -side))).translation;
  double x[] = {fl.x, fr.x, br.x, bl.x, fl.x},
         y[] = {fl.y, fr.y, br.y, bl.y, fl.y};
  addCoords(lines[LinesPercept::numberOfLineTypes + 1], 4, x, y);
}

Pose2D Field::randomPose() const
{ 
  Pose2D pose;
  do
    pose = Pose2D::random(x,y,Range<double>(-pi,pi));
  while(!isReallyInside(pose.translation));
  return pose;
}

void Field::draw(const Drawings::Color color, LinesPercept::LineType type) const
{
  const Table& table = type == LinesPercept::numberOfLineTypes ? boundary : lines[type];
  for(int i = 0; i < table.numberOfEntries; ++i)
  {
    Vector2<double> p = (table.corner[i] + 
      Pose2D(Vector2<double>(table.length[i],0))).translation;
    LINE(selfLocatorField,
       (int) table.corner[i].translation.x,
       (int) table.corner[i].translation.y,
       (int) p.x,
       (int) p.y,
       1, Drawings::ps_solid, color);    
  }
}

/*
 * Change log :
 * 
 * $Log: Field.cpp,v $
 * Revision 1.2  2004/06/24 18:26:38  roefer
 * Lines table redesign, should not influence the performance of GT2003SL
 *
 * Revision 1.1.1.1  2004/05/22 17:35:50  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.8  2004/03/16 14:00:23  juengel
 * Integrated Improvments from "Gnne"
 * -ATH2004ERS7Behavior
 * -ATHHeadControl
 * -KickSelectionTable
 * -KickEditor
 *
 * Revision 1.2  2004/03/15 17:11:41  hoffmann
 * - added ATH2004HeadControl
 * - added ATH2004LEDControl
 * - headmotiontester shows "tilt2"
 * - motion process updates odometry while no new robotPose is received, added to motion request
 * - some ui adjustments
 * - added member function to "field" to find out if robot is in own penalty area for use in the obstacles locator
 *
 * Revision 1.7  2004/03/01 11:47:15  juengel
 * Moved enum ObstacleType to class ObstaclesPercept.
 *
 * Revision 1.6  2004/02/28 14:00:43  juengel
 * Added ObstacleType to ObstaclesModel.
 *
 * Revision 1.5  2004/01/01 10:58:52  roefer
 * RobotDimensions are in a class now
 *
 * Revision 1.4  2003/11/08 18:48:32  roefer
 * placePlayers now also works with less than 7 players
 *
 * Revision 1.3  2003/10/24 08:27:57  schumann
 * removed compiler error (renamed variable i to j in method Field::placePlayers in second for statement)
 *
 * Revision 1.2  2003/10/23 15:41:40  roefer
 * Oracled obstacle model added
 *
 * Revision 1.1  2003/10/07 10:13:20  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.2  2003/09/01 10:23:14  juengel
 * DebugDrawings clean-up 2
 * DebugImages clean-up
 * MessageIDs clean-up
 * Stopwatch clean-up
 *
 * Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.5  2003/05/12 12:03:39  roefer
 * New penalty area size, corrected sample poses for blue players
 *
 * Revision 1.4  2003/05/11 23:56:15  dueffert
 * doxygen bugs fixed
 *
 * Revision 1.3  2003/04/16 07:00:16  roefer
 * Bremen GO checkin
 *
 * Revision 1.4  2003/04/10 21:38:35  roefer
 * Improvements in self localization
 *
 * Revision 1.3  2003/04/08 18:23:07  roefer
 * LinesHeadControl speed increased
 *
 * Revision 1.2  2003/04/06 21:03:29  roefer
 * LinesHeadControl, first draft
 *
 * Revision 1.1  2003/03/31 21:01:44  roefer
 * Moved class Field to Tools
 *
 * Revision 1.9  2003/03/16 17:54:45  roefer
 * Severe error removed
 *
 * Revision 1.8  2003/03/16 13:20:52  roefer
 * Minor errors corrected
 *
 * Revision 1.7  2003/03/15 13:57:34  roefer
 * Bug removed
 *
 * Revision 1.6  2003/03/04 17:43:54  roefer
 * New penalty area
 *
 * Revision 1.5  2003/02/21 22:20:12  roefer
 * LinesSelfLocator is working
 *
 * Revision 1.4  2002/12/12 22:09:41  roefer
 * Progress in LinesSelfLocator
 *
 * Revision 1.3  2002/11/12 10:49:02  juengel
 * New debug drawing macros - second step
 * -moved /Tools/Debugging/PaintMethods.h and . cpp
 *  to /Visualization/DrawingMethods.h and .cpp
 * -moved DebugDrawing.h and .cpp from /Tools/Debugging/
 *  to /Visualization
 *
 * Revision 1.2  2002/09/22 18:40:53  risler
 * added new math functions, removed GTMath library
 *
 * Revision 1.1  2002/09/10 15:36:15  cvsadm
 * Created new project GT2003 (M.L.)
 * - Cleaned up the /Src/DataTypes directory
 * - Removed challenge related source code
 * - Removed processing of incoming audio data
 * - Renamed AcousticMessage to SoundRequest
 *
 * Revision 1.1  2002/06/02 23:21:09  roefer
 * Single color table and progress in LinesSelfLocator
 *
 *
 */
