/**
* @file GenericDebugData.h
* 
* Declaration of class GenericDebugData used for adjusting parameters on the robot.
*
* @author <a href="mailto:jhoffman@informatik.hu-berlin.de">Jan Hoffmann</a>
*/ 

#ifndef __GenericDebugData_h_
#define __GenericDebugData_h_

#include "Tools/Streams/InOut.h"
#include "Tools/MessageQueue/MessageIDs.h"


/**
* 
*/
class GenericDebugData
{
public:
  /** Constructor */ 
  GenericDebugData();

  /** Destructor */ 
  ~GenericDebugData();

  enum GenericDebugDataID 
  {
    numberOfFourierCoefficients = 0, 
    braitenbergPIDs = 1, 
    danielTest = 2,
    defaultObstaclesLocator = 3, 
    linesImageProcessor = 4,
    MotionDetector = 5,
    gridImageProcessor2DebugParameters = 6,
    linesSelfLocator = 7,
    ballLocatorPIDs = 8,
    bodyOffsets = 9,
    odometryScale = 10,
    motionStabilizer = 11,
    yetAnotherInvKinParams = 12,
    yetAnotherInvKinPaws = 13,
    noOdometrySelfLocator = 14,
	teamMessageSendDelay = 15,
    // insert above this line
    numOfGenericDebugDataIDs,
    unknown
  };

  static const char* getGenericDebugDateName(GenericDebugDataID genericDebugDataID)
  {
    switch(genericDebugDataID)
    {
      case braitenbergPIDs: return "braitenbergPIDs";
      case defaultObstaclesLocator: return "DefaultObstaclesLocator";
      case MotionDetector: return "MotionDetector";
      case numberOfFourierCoefficients: return "numberOfFourierCoefficients";
      case danielTest: return "danielTest";
      case linesImageProcessor: return "linesImageProcessor";
      case gridImageProcessor2DebugParameters: return "gridImageProcessor2DebugParameters";
      case linesSelfLocator: return "linesSelfLocator";
      case ballLocatorPIDs: return "ballLocatorPIDs";
      case bodyOffsets: return "bodyOffsets";
      case odometryScale: return "odometryScale";
      case motionStabilizer: return "motionStabilizer";
      case yetAnotherInvKinParams: return "yetAnotherInvKinParams";
      case yetAnotherInvKinPaws: return "yetAnotherInvKinPaws";
      case noOdometrySelfLocator: return "noOdometrySelfLocator";
	  case teamMessageSendDelay: return "teamMessageSendDeay";
      default: return "unnamed";
    }
  }

  static const char* getParameterName(GenericDebugDataID genericDebugDataID, int index)
  {
    char noNames[10][100] = {"","","","","","","","","",""};
    char linesImageProcessorNames[10][100] = {"gridSpacing","yThreshold","vThreshold","","","","","","",""};
    char gridImageProcessor2DebugParametersNames[10][100] = {"scanLineSpacing","numOfLines","useFixedHorizon","returnCondition","-", "-", "-", "-", "-", "-"};
    char linesSelfLocatorNames[10][100] = {"up","down","delay","height","z","y","trans","rot","",""};
    char ballLocatorPIDNames[10][100] = {"ballX P","ballX I","ballX D","speedX P","speedX I","speedX D","","","",""};
    char bodyOffsetNames[10][100] = {"bodyTiltOffset","bodyRollOffset","headTiltOffset","headRollOffset","tiltFactor","panFactor","tile2Factor","","",""};
    char odometryScaleNames[10][100] = {"scaleX","scaleY","scaleRotation","counterRotation","","","","","",""};
    char yetAnotherInvKinParamNames[10][100] = {"x","y","z","tilt","pan","roll","kick lateral","kick speed","body extra","kick length"};
    char yetAnotherInvKinPawsNames[10][100] = {"fore x","fore y","hind x","hind y","kick height","use paw","smoothing","","",""};

    switch(genericDebugDataID)
    {
      case linesImageProcessor: return linesImageProcessorNames[index];
      case gridImageProcessor2DebugParameters: return gridImageProcessor2DebugParametersNames[index];
      case linesSelfLocator: return linesSelfLocatorNames[index];
      case ballLocatorPIDs: return ballLocatorPIDNames[index];
      case bodyOffsets: return bodyOffsetNames[index];
      case odometryScale: return odometryScaleNames[index];
      case yetAnotherInvKinParams: return yetAnotherInvKinParamNames[index];
      case yetAnotherInvKinPaws: return yetAnotherInvKinPawsNames[index];
      default: return noNames[index];
    }
  }

  static MessageID getMessageID(GenericDebugDataID genericDebugDataID)
  {
    switch(genericDebugDataID)
    {
      case linesSelfLocator: return idLinesSelfLocatorParameters;
      case bodyOffsets: return idBodyOffsets;
      case odometryScale: return idOdometryScale;
      case yetAnotherInvKinParams: return idYetAnotherInvKinParams;
      case yetAnotherInvKinPaws: return idYetAnotherInvKinPaws;
      default: return idGenericDebugData;
    }
  }

  static double getDefaultValue(GenericDebugDataID genericDebugDataID, int index)
  {
    double linesImageProcessorValues[10] = {5,15,8,0,0,0,0,0,0,0};
    double gridImageProcessor2DebugParametersValues[10] = {5,40,0,0,0,0,0,0,0,0};
    double linesSelfLocatorValues[10] = {0.01,0.005,2,150,10,2,200,0.5,0,0};
    double yetAnotherInvKinParamsValues[10] = {23.0,0.0,106.0,10.2,10,10,0,0,0,0};
    double yetAnotherInvKinPawsValues[10] = {36.0,7.0,111.0,32.2,0,0,0,0,0,0};

    switch(genericDebugDataID)
    {
      case linesImageProcessor: return linesImageProcessorValues[index];
      case gridImageProcessor2DebugParameters: return gridImageProcessor2DebugParametersValues[index];
      case linesSelfLocator: return linesSelfLocatorValues[index];
      case yetAnotherInvKinParams: return yetAnotherInvKinParamsValues[index];
      case yetAnotherInvKinPaws: return yetAnotherInvKinPawsValues[index];
      default: return 0;
    }
  }

  static double getMinValue(GenericDebugDataID genericDebugDataID, int index)
  {
    double gridImageProcessor2DebugParametersValues[10] = {1,1,0,0,0,0,0,0,0,0};
    double yetAnotherInvKinParamsValues[10] = {0,0,60,9.5,8.5,8.5,0,0,0,0};
    double yetAnotherInvKinPawsValues[10] = {0,0,60,9.5,8.5,8.5,0,0,0,0};

    switch(genericDebugDataID)
    {
      case gridImageProcessor2DebugParameters: return gridImageProcessor2DebugParametersValues[index];
      case yetAnotherInvKinParams: return yetAnotherInvKinParamsValues[index];
      case yetAnotherInvKinPaws: return yetAnotherInvKinPawsValues[index];
      default: return 0;
    }
  }

  static double getMaxValue(GenericDebugDataID genericDebugDataID, int index)
  {
    double gridImageProcessor2DebugParametersValues[10] = {100,200,1,100,0,0,0,0,0,0};
    double yetAnotherInvKinParamsValues[10] = {100,100,200,10.5,12,12,80,150,100,100};
    double yetAnotherInvKinPawsValues[10] = {100,100,200,100,100,100,0,0,0,0};

    switch(genericDebugDataID)
    {
      case gridImageProcessor2DebugParameters: return gridImageProcessor2DebugParametersValues[index];
      case yetAnotherInvKinParams: return yetAnotherInvKinParamsValues[index];
      case yetAnotherInvKinPaws: return yetAnotherInvKinPawsValues[index];
      default: return 0;
    }
  }

  int id;
  double data[10];
};

/**
 * Streaming operator that reads GenericDebugData from a stream.
 * @param stream The stream from which is read.
 * @param d The GenericDebugData object.
 * @return The stream.
 */ 
In& operator>>(In& stream, GenericDebugData& d);
 
/**
 * Streaming operator that writes GenericDebugData to a stream.
 * @param stream The stream to write on.
 * @param d The GenericDebugData object.
 * @return The stream.
 */ 
Out& operator<<(Out& stream, GenericDebugData& d);

#endif //__GenericDebugData_h_

/*
* Change log :
* 
* $Log: GenericDebugData.h,v $
* Revision 1.3  2004/05/24 17:03:45  wachter
* Added GenerigDebugData for setting team-communication delay.
*
* Revision 1.2  2004/05/22 20:46:34  juengel
* Removed idLinesImageProcessorParameters.
*
* Revision 1.1.1.1  2004/05/22 17:36:06  cvsadm
* created new repository GT2004_WM
*
* Revision 1.9  2004/03/22 21:58:15  roefer
* True odometry
*
* Revision 1.8  2004/02/26 10:28:18  jhoffman
* configuration toolbar removed
*
* Revision 1.7  2004/02/24 19:01:06  roefer
* Additional calibration parameters added
*
* Revision 1.6  2004/02/16 19:22:47  jhoffman
* added debug parameters for yet another inv kin
*
* Revision 1.5  2003/12/03 21:03:07  spranger
* added noOdometry to GenericDebugDataId-enum and getGenericDebugDataName
*
* Revision 1.4  2003/11/24 09:31:39  jhoffman
* worked on walking engine (paw placement,  trajectories, ...)
*
* Revision 1.3  2003/11/22 15:59:03  jhoffman
* parameter sending using generic debug data to
* walking engine now works if it is explicitely given
* a messageID
*
* Revision 1.2  2003/11/22 09:33:01  jhoffman
* added "Yet Another Inv Kin Walking Engine"
* experimental stage, doesn't do much yet
* no Fourier inside!
*
* Revision 1.1  2003/10/07 10:13:22  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1  2003/09/26 11:40:40  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.2  2003/08/18 12:10:08  juengel
* Changed GridImageProcessor2Parameters.
*
* Revision 1.1.1.1  2003/07/02 09:40:22  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.29  2003/06/13 17:29:44  jhoffman
* added debug message for stabilizer
*
* Revision 1.28  2003/05/14 13:08:38  risler
* removed DefaultObstaclesLocator
* renamed MicroSectorsObstaclesLocator to DefaultObstaclesLocator
* ObstaclesModel contains increased number of sectors
* DefaultObstaclesLocator clean up
*
* Revision 1.27  2003/05/05 12:16:17  juengel
* Removed GridImageProcessor2Thresholds
*
* Revision 1.26  2003/05/03 16:20:42  roefer
* bodyOffsets generic debug data
*
* Revision 1.25  2003/04/06 17:14:37  jhoffman
* added headTilt to PSDpercept and added debugging capabilities to microsectors
*
* Revision 1.24  2003/03/28 14:31:17  juengel
* Added "circleCalculation" to gridImageProcessor2Parameters.
*
* Revision 1.23  2003/03/19 15:38:22  jhoffman
* no message
*
* Revision 1.22  2003/03/06 14:27:58  juengel
* Added value in GridImageProcessor2DebugParameters
*
* Revision 1.21  2003/03/05 14:07:09  roefer
* LinesPerceptor2 is now LinesImageProcessor
*
* Revision 1.20  2003/03/05 10:35:57  jhoffman
* added debugging capabilities for ball locator
*
* Revision 1.19  2003/03/04 14:54:09  juengel
* Renamed gridImageProcessor2 in gridImageProcessor2DebugParameters.
*
* Revision 1.18  2003/02/27 12:56:25  juengel
* Changed GridImageProcessor2 parameters.
*
* Revision 1.17  2003/02/25 14:00:00  juengel
* added gridImageProcessor2Values
*
* Revision 1.16  2003/02/21 22:20:12  roefer
* LinesSelfLocator is working
*
* Revision 1.15  2003/02/17 11:06:02  dueffert
* warnings removed
*
* Revision 1.14  2003/02/09 20:51:15  roefer
* sg console command added to SimGT2003
*
* Revision 1.13  2003/02/09 15:11:11  juengel
* Added parameter "detect"
*
* Revision 1.12  2003/02/09 13:01:16  juengel
* Parameters green auto and green manual added.
*
* Revision 1.11  2003/02/09 10:07:37  juengel
* Added min max and default value.
*
* Revision 1.10  2003/02/08 18:36:44  juengel
* added idGridImageProcessor2Thresholds
*
* Revision 1.9  2003/01/30 22:29:30  juengel
* Added methods getParameterName and getParameterValue to GenericDebugData.
*
* Revision 1.8  2003/01/30 11:30:03  juengel
* Added method getMessageID()
*
* Revision 1.7  2003/01/28 18:07:17  juengel
* added string for danielTest
*
* Revision 1.6  2003/01/27 14:47:12  juengel
* TestDataGenerator generates GenericDebugData now.
*
* Revision 1.5  2003/01/23 19:07:52  juengel
* Improved TestDataGeneratorDlg.
* Removed ParameterStruct.
* Moved implementation and declaration of TestDataGenerator to
* TestDataGeneratorDlg.h and TestDataGeneratorDlg.cpp.
*
* Revision 1.4  2003/01/23 15:51:44  jhoffman
* no message
*
* Revision 1.3  2003/01/17 14:43:23  dueffert
* doxygen docu corrected
*
* Revision 1.2  2003/01/17 12:48:19  jhoffman
* added GenericDebugData
* changes to the TestDataGenerator towards using sliders, min/max, and other cool stuff
*
* Revision 1.1  2003/01/16 09:54:49  jhoffman
* Added "Generic Debug Data" data type. This can be used
* for quick and dirty optimization and debugging, e.g. to send
* parameters to a module through WLAN to adjust it's settings.
* The DebugMessageGenerator is used to for parsing and
* sendig the data
*
*
*/
