/**
* @file Tools/Debugging/DebugDrawings.h
*
* Macros for drawing
* 
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Jngel</a>
*/ 
#ifndef __DebugDrawings_h_
#define __DebugDrawings_h_

#include "Tools/Debugging/Debugging.h"
#include "Tools/Debugging/DebugKeyTable.h"
//#include "Platform/GTAssert.h"

/**
* This class includes all enumeration types and methods that are necessary
* to transmit messages for debug drawings
*/
class Drawings
{
public:

  /** IDs for shape types 
  * shapes are the basic drawings that can be sent.
  */
  enum ShapeType
  {
    circle, filledCircle, line, dot, largeDot, octangle, arrow
  };

  /**
  * IDs for debug drawings on the field.
  * enter new debug drawings here and add the corresponding string in getDrawingName.
  */
  enum FieldDrawing
  {
    // the field
    fieldPolygons, /*!< the polygons of the 2D field view */
    fieldLines, /*!< the lines of the 2D field view  */

    // percepts
    percepts_ballFlagsGoalsField, /*!< player, ball and landmarks percepts on the field view. */
    percepts_obstaclesAndPSDField, /*!< obstacles and psd percepts on the field view. */
    percepts_specialField, /*!< special percept on field */

    // models
    worldStateOracle, /*!< the oracled world state (displayed ligter) */
    worldState, /*!< the world state as calculated by the modules */
    models_obstaclesField, /*!< the obstacles model */

    // modules
    selfLocatorField,/*!< visualization of the particles in the monte carlo selflocalization */
    ballLocatorField,/*!< visualization of hypothesis regarding the ball */
	goaliePositionField,/*!< visualization of the goalie position */
    headControlField,/*!< visualization what feature of the environment HeadControl is aiming at */
    teamBallLocatorField,

    //      obstaclesLocatorField, /*! Drawings for the SensorFusionPlayersLocator */
    behavior_CBBVector, /*! Drawings for ContinuosBasicBehavior1 */
    behavior_CBBWay, /*! Drawings for ContinuosBasicBehavior2 */
    behavior_kickAngles, /*! Drawings for several kick angles */
    behavior_aStarSearch, /*! Drawings for A* search tree */

    dribbleBall, /*! Drawings for dribble ball */

    // radar viewer      
    firstDrawingInRadarViewer, /*!< all drawings below here are shown in the radar view */
    percepts_ballFlagsGoalsRadar = firstDrawingInRadarViewer, /*!< player, ball and landmarks percepts on the radar view. */
    percepts_obstaclesAndPSDRadar, /*!< obstacles and psd percepts on the radar view. */
    models_obstaclesRadar, /*!< obstacles model in radar view */
    models_corridorsRadar, /*! All corridors that are calculated */
    behavior_kickAnglesRadar,  /*!< kick angles in radar view */

    numberOfFieldDrawings
  };

  /**
  * IDs for debug drawings on the field.
  * enter new debug drawings here and add the corresponding string in getDrawingName.
  */
  enum ImageDrawing
  {
    percepts_ball,
    percepts_flagsGoals,
    percepts_freePartOfGoal,
    percepts_lines,
    percepts_edges,
    percepts_obstacles,
    percepts_psd,
    percepts_special,
    models_obstacles, 
    models_freePartOfGoal, 
    selfLocator,
    imageProcessor_general,
    imageProcessor_horizon,
    imageProcessor_scanLines,
    imageProcessor_coloredSegments1,
    imageProcessor_coloredSegments2,
    imageProcessor_coloredSegments3,
    imageProcessor_calibration1,
    imageProcessor_calibration2,
    imageProcessor_ball1,
    imageProcessor_ball2,
    imageProcessor_ball3,
    imageProcessor_ball4,
    imageProcessor_ground,
    imageProcessor_flagsAndGoals,
    imageProcessor_obstacles,
    imageProcessor_gradients, 
    imageProcessor_edges,
    sketch,
    numberOfImageDrawings
  };

  enum TypeOfDrawing 
  {
    drawingOnField,
    drawingOnImage
  };

  static TypeOfDrawing getTypeOfDrawing(FieldDrawing) {return drawingOnField;}
  static TypeOfDrawing getTypeOfDrawing(ImageDrawing) {return drawingOnImage;}

  /**
  * Returns a description for an indexed fieldDrawing.
  * \attention Add descriptions for new fieldDrawings here.
  */
  static const char* getDrawingName(FieldDrawing fieldDrawing)
  {
    switch (fieldDrawing) 
    {
    case fieldPolygons: return "field polygons";
    case fieldLines: return "field lines";

    case percepts_ballFlagsGoalsField: return "percepts: ball; flags; goals";
    case percepts_obstaclesAndPSDField: return "percepts: obstacles; PSD";
    case percepts_specialField: return "percepts: special";

    case worldStateOracle: return "world state oracle";
    case worldState: return "world state";
    case models_obstaclesField: return "models: obstacles";

    case selfLocatorField: return "self locator";
    case ballLocatorField: return "ball locator";
    case goaliePositionField: return "goalie position";	
    case headControlField: return "head control";
    case teamBallLocatorField: return "team ball locator";
      //    case obstaclesLocatorField: return "obstacles locator";
    case behavior_CBBVector: return "behavior: CBBVector";
    case behavior_CBBWay: return "behavior: CBBWay";
    case behavior_kickAngles: return "behavior: kick angles";
    case behavior_aStarSearch: return "behavior: A* search tree";
    case dribbleBall: return "dribble Ball";

    case percepts_ballFlagsGoalsRadar: return "radar - percepts: ball; flags; goals";
    case percepts_obstaclesAndPSDRadar: return "radar - percepts: obstacles; PSD";
    case models_obstaclesRadar: return "radar - models: obstacles";
    case models_corridorsRadar: return "radar - models: corridors";
    case behavior_kickAnglesRadar: return "radar - behavior: kick angles";
    default: return "check available drawings";
    }
  }

  /**
  * Returns a description for an indexed imageDrawing.
  * \attention Add descriptions for new image drawings here.
  */
  static const char* getDrawingName(ImageDrawing imageDrawing)
  {
    switch (imageDrawing) 
    {
    case percepts_ball: return "percepts: ball";
    case percepts_flagsGoals: return "percepts: flags; goals";
    case percepts_freePartOfGoal: return "percepts: free part of goal";
    case percepts_lines: return "percepts: lines";
    case percepts_edges: return "percepts: edges";
    case percepts_obstacles: return "percepts: obstacles";
    case percepts_psd: return "percepts: psd";
    case percepts_special: return "percepts: special";
    case models_obstacles: return "models: obstacles";
    case models_freePartOfGoal: return "models: free part of goal";
    case selfLocator: return "self locator";
    case imageProcessor_general: return "image processor: general";
    case imageProcessor_horizon: return "image processor: horizon";
    case imageProcessor_scanLines: return "image processor: scan lines";
    case imageProcessor_coloredSegments1: return "image processor: colored segments1";
    case imageProcessor_coloredSegments2: return "image processor: colored segments2";
    case imageProcessor_coloredSegments3: return "image processor: colored segments3";
    case imageProcessor_calibration1: return "image processor: calibration 1";
    case imageProcessor_calibration2: return "image processor: calibration 2";
    case imageProcessor_ball1: return "image processor: ball 1";
    case imageProcessor_ball2: return "image processor: ball 2";
    case imageProcessor_ball3: return "image processor: ball 3";
    case imageProcessor_ball4: return "image processor: ball 4";
    case imageProcessor_ground: return "image processor: ground";
    case imageProcessor_flagsAndGoals: return "image processor: flags; goals";
    case imageProcessor_obstacles: return "image processor: obstacles";
    case imageProcessor_gradients: return "image processor: gradients";
    case imageProcessor_edges: return "image processor: edges";
    case sketch: return "sketch";
    default: return "check available drawings";
    }
  }

  static bool getDebugKeyID(enum FieldDrawing fieldDrawing, DebugKeyTable::debugKeyID& debugKeyID)
  {
    switch (fieldDrawing) 
    {
    case selfLocatorField: debugKeyID = DebugKeyTable::send_selfLocatorField_drawing; return true;
    case ballLocatorField: debugKeyID = DebugKeyTable::send_ballLocatorField_drawing; return true;
    case goaliePositionField: debugKeyID = DebugKeyTable::send_goaliePositionField_drawing; return true;
    case headControlField: debugKeyID = DebugKeyTable::send_headControlField_drawing; return true;
    case teamBallLocatorField: debugKeyID = DebugKeyTable::send_teamBallLocatorField_drawing; return true;
      //    case obstaclesLocatorField: debugKeyID = DebugKeyTable::send_obstaclesLocatorField_drawing; return true;
    case behavior_CBBVector: debugKeyID = DebugKeyTable::send_behavior_CBBVector_drawing; return true;
    case behavior_CBBWay: debugKeyID = DebugKeyTable::send_behavior_CBBWay_drawing; return true;
    case behavior_kickAngles: debugKeyID = DebugKeyTable::send_behavior_kickAngles_drawing; return true;
    case behavior_aStarSearch: debugKeyID = DebugKeyTable::send_behavior_aStarSearch_drawing; return true;
    case dribbleBall: debugKeyID = DebugKeyTable::send_dribbleBall_drawing; return true;

    case models_corridorsRadar: debugKeyID = DebugKeyTable::send_models_corridorsRadar_drawing; return true;
    case behavior_kickAnglesRadar: debugKeyID = DebugKeyTable::send_behavior_kickAnglesRadar_drawing; return true;

    default: return false;
    }
  }

  static bool getDebugKeyID(enum ImageDrawing imageDrawing, DebugKeyTable::debugKeyID& debugKeyID)
  {
    switch (imageDrawing) 
    {
    case selfLocator: debugKeyID = DebugKeyTable::send_selfLocator_drawing; return true;
    case imageProcessor_general: debugKeyID = DebugKeyTable::send_imageProcessor_general_drawing; return true;
    case imageProcessor_horizon: debugKeyID = DebugKeyTable::send_imageProcessor_horizon_drawing; return true;
    case imageProcessor_scanLines: debugKeyID = DebugKeyTable::send_imageProcessor_scanLines_drawing; return true;
    case imageProcessor_coloredSegments1: debugKeyID = DebugKeyTable::send_imageProcessor_coloredSegments1_drawing; return true;
    case imageProcessor_coloredSegments2: debugKeyID = DebugKeyTable::send_imageProcessor_coloredSegments2_drawing; return true;
    case imageProcessor_coloredSegments3: debugKeyID = DebugKeyTable::send_imageProcessor_coloredSegments3_drawing; return true;
    case imageProcessor_calibration1: debugKeyID = DebugKeyTable::send_imageProcessor_calibration1_drawing; return true;
    case imageProcessor_calibration2: debugKeyID = DebugKeyTable::send_imageProcessor_calibration2_drawing; return true;
    case imageProcessor_ball1: debugKeyID = DebugKeyTable::send_imageProcessor_ball1_drawing; return true;
    case imageProcessor_ball2: debugKeyID = DebugKeyTable::send_imageProcessor_ball2_drawing; return true;
    case imageProcessor_ball3: debugKeyID = DebugKeyTable::send_imageProcessor_ball3_drawing; return true;
    case imageProcessor_ball4: debugKeyID = DebugKeyTable::send_imageProcessor_ball4_drawing; return true;
    case imageProcessor_ground: debugKeyID = DebugKeyTable::send_imageProcessor_ground_drawing; return true;
    case imageProcessor_flagsAndGoals: debugKeyID = DebugKeyTable::send_imageProcessor_flagsAndGoals_drawing; return true;
    case imageProcessor_obstacles: debugKeyID = DebugKeyTable::send_imageProcessor_obstacles_drawing; return true;
    case imageProcessor_gradients: debugKeyID = DebugKeyTable::send_imageProcessor_gradients_drawing; return true;
    case imageProcessor_edges: debugKeyID = DebugKeyTable::send_imageProcessor_edges_drawing; return true;
    case sketch: debugKeyID = DebugKeyTable::send_sketch_drawing; return true;
    default: return false;
    }
  }

  /** The pen style that is used for basic shapes*/
  enum PenStyle
  {
    ps_solid, ps_dash, ps_dot, ps_null
  };

  /** The fill style that is used for basic shapes*/
  enum FillStyle
  {
    bs_solid, bs_null
  };

  /** The color that is used for basic shapes*/
  enum Color
  {
    red, green, blue, yellow, orange, pink, skyblue, white, light_gray, gray, dark_gray, black, yellowOrange, numberOfColors
  };

};

#ifdef NDEBUG
#define NODEBUGDRAWINGS
#endif
#ifdef NO_DEBUG_DRAWING
#define NODEBUGDRAWINGS
#endif

#ifdef NODEBUGDRAWINGS
#define CIRCLE(id, center_x, center_y, radius, penWidth, penStyle, penColor) /**/
#define DOT(id, x, y, penColor, fillColor) /**/
#define LARGE_DOT(id, x, y, penColor, fillColor) /**/
#define ARROW(id, x1, y1, x2, y2, penWidth, penStyle, penColor) /**/
#define LINE(id, x1, y1, x2, y2, penWidth, penStyle, penColor) /**/
#define POSE_2D_SAMPLE(id, p, color) /**/
#define OCTANGLE(id, x1,y1,x2,y2,x3,y3,x4,y4,x5,y5,x6,y6,x7,y7,x8,y8, color, fill) /**/
#define QUADRANGLE(id, x1,y1,x2,y2,x3,y3,x4,y4, penWidth, penStyle, penColor) /**/
#define RECTANGLE(id, x1,y1,x2,y2, penWidth, penStyle, penColor) /**/

#define COMPLEX_DRAWING(id,expression) /**/
#define DEBUG_DRAWING_FINISHED(id) /**/

#else //NODEBUGDRAWINGS

/** 
* A macro that sends a circle
* @param id A drawing id (Drawings::FieldDrawing or Drawings::ImageDrawing)
* @param center_x The x coordinate of the center of the circle
* @param center_y The y coordinate of the center of the circle
* @param radius The radius of the circle
* @param penWidth The width of the arc of the circle
* @param penStyle The pen style of the arc of the circle (Drawings::PenStyle)
* @param penColor The color of the arc of the circle (Drawings::Color)
*/
#define CIRCLE(id, center_x, center_y, radius, penWidth, penStyle, penColor) \
  INFO(send_##id##_drawing, idDebugDrawing2, bin, \
  (char)Drawings::circle << \
  (char)Drawings::id << \
  (char)Drawings::getTypeOfDrawing(Drawings::id) << \
  (int)(center_x) << (int)(center_y) << (int)(radius) << (char)(penWidth) << \
  (char)(penStyle) << (char)(penColor) \
  );

/** 
* A macro that sends a dot (a quadratic box with a border)
* @param id A drawing id (Drawings::FieldDrawing or Drawings::ImageDrawing)
* @param x The x coordinate of the center of the box
* @param y The y coordinate of the center of the box
* @param penColor The color of the border of the dot (Drawings::Color)
* @param fillColor The color of the dot (Drawings::Color)
*/
#define DOT(id, x, y, penColor, fillColor) \
  INFO(send_##id##_drawing, idDebugDrawing2, bin, \
  (char)Drawings::dot << \
  (char)Drawings::id << \
  (char)Drawings::getTypeOfDrawing(Drawings::id) << \
  (int)(x) << (int)(y) << (char)(penColor) << (char)(fillColor) \
  );


/** 
* A macro that sends a dot (a quadratic box with a border)
* @param id A drawing id (Drawings::FieldDrawing or Drawings::ImageDrawing)
* @param x The x coordinate of the center of the box
* @param y The y coordinate of the center of the box
* @param penColor The color of the border of the dot (Drawings::Color)
* @param fillColor The color of the dot (Drawings::Color)
*/
#define LARGE_DOT(id, x, y, penColor, fillColor) \
  INFO(send_##id##_drawing, idDebugDrawing2, bin, \
  (char)Drawings::largeDot << \
  (char)Drawings::id << \
  (char)Drawings::getTypeOfDrawing(Drawings::id) << \
  (int)(x) << (int)(y) << (char)(penColor) << (char)(fillColor) \
  );

/**
* A macro that sends a line
* @param id A drawing id (Drawings::FieldDrawing or Drawings::ImageDrawing)
* @param x1 The x coordinate of the starting point.
* @param y1 The y coordinate of the starting point.
* @param x2 The x coordinate of the end point.
* @param y2 The y coordinate of the end point.
* @param penWidth The width of the line
* @param penStyle The pen style of the line (Drawings::PenStyle)
* @param penColor The color of the line (Drawings::Color)
*/
#define LINE(id, x1, y1, x2, y2, penWidth, penStyle, penColor) \
  INFO(send_##id##_drawing, idDebugDrawing2, bin, \
  (char)Drawings::line << \
  (char)Drawings::id << \
  (char)Drawings::getTypeOfDrawing(Drawings::id) << \
  (int)(x1) << (int)(y1) << (int)(x2) << (int)(y2) << (char)(penWidth) << (char)(penStyle) << (char)(penColor) \
  );


/**
* A macro that sends an arrow
* @param id A drawing id (Drawings::FieldDrawing or Drawings::ImageDrawing)
* @param x1 The x coordinate of the starting point.
* @param y1 The y coordinate of the starting point.
* @param x2 The x coordinate of the end point.
* @param y2 The y coordinate of the end point.
* @param penWidth The width of the line
* @param penStyle The pen style of the line (Drawings::PenStyle)
* @param penColor The color of the line (Drawings::Color)
*/
#define ARROW(id, x1, y1, x2, y2, penWidth, penStyle, penColor) \
  INFO(send_##id##_drawing, idDebugDrawing2, bin, \
  (char)Drawings::arrow << \
  (char)Drawings::id << \
  (char)Drawings::getTypeOfDrawing(Drawings::id) << \
  (int)(x1) << (int)(y1) << (int)(x2) << (int)(y2) << (char)(penWidth) << (char)(penStyle) << (char)(penColor) \
  );


/**
* A macro that sends an arrow for a pose
* @param id A drawing id (Drawings::FieldDrawing or Drawings::ImageDrawing)
* @param p A Pose2D describing the arrow
* @param color The color of the line arrow (Drawings::Color)
*/
#define POSE_2D_SAMPLE(id, p, color) \
  Pose2D current = p; current += Pose2D(-100,0); \
  LINE(id, int(current.translation.x),int(current.translation.y), int(p.translation.x), int(p.translation.y), \
  1, Drawings::ps_solid, color); \
  current = p; current += Pose2D(-40,-40); \
  LINE(id, int(current.translation.x), int(current.translation.y), int(p.translation.x), int(p.translation.y), \
  1, Drawings::ps_solid, color); \
  current = p; current += Pose2D(-40,40); \
  LINE(id, int(current.translation.x),int(current.translation.y),int(p.translation.x), int(p.translation.y), \
  1, Drawings::ps_solid, color); \

/** 
* A macro that sends an octangle
* @param id A drawing id (Drawings::FieldDrawing or Drawings::ImageDrawing)
* @param x1,y1,x2,y2,x3,y3,x4,y4,x5,y5,x6,y6,x7,y7,x8,y8 The coordinates of the 8 octagon vertices
* @param color The color of the octangle
* @param fill A boolean indicating if the octangle is filled
*/
#define OCTANGLE(id, x1,y1,x2,y2,x3,y3,x4,y4,x5,y5,x6,y6,x7,y7,x8,y8, color, fill) \
  INFO(send_##id##_drawing, idDebugDrawing2, bin, \
  (char)Drawings::octangle << \
  (char)Drawings::id << \
  (char)Drawings::getTypeOfDrawing(Drawings::id) << \
  (int)(x1) << (int)(y1) << (int)(x2) << (int)(y2) << (int)(x3) << (int)(y3) << (int)(x4) << (int)(y4) << \
  (int)(x5) << (int)(y5) << (int)(x6) << (int)(y6) << (int)(x7) << (int)(y7) << (int)(x8) << (int)(y8) << \
  (char)ColorClasses::colorClassToDrawingsColor(color) << (char)fill \
  );

/** 
* A macro that sends an quadrangle
* @param x1,y1,x2,y2,x3,y3,x4,y4 The coordinates of the 4 quadrangle vertices
* @param id A drawing id (Drawings::FieldDrawing or Drawings::ImageDrawing)
* @param penWidth The line width of the quadrangle
* @param penStyle The line style, e.g. dotted
* @param penColor The color of the quadrangle
*/
#define QUADRANGLE(id, x1,y1,x2,y2,x3,y3,x4,y4, penWidth, penStyle, penColor) \
  LINE(id, x1, y1, x2, y2, penWidth, penStyle, penColor); \
  LINE(id, x2, y2, x3, y3, penWidth, penStyle, penColor); \
  LINE(id, x3, y3, x4, y4, penWidth, penStyle, penColor); \
  LINE(id, x4, y4, x1, y1, penWidth, penStyle, penColor); 

/** 
* A macro that sends an rectangle
* @param x1,y1,x2,y2 The coordinates of 2 opposite corners
* @param id A drawing id (Drawings::FieldDrawing or Drawings::ImageDrawing)
* @param penWidth The line width of the rectangle
* @param penStyle The line style, e.g. dotted
* @param penColor The color of the quadrangle
*/
#define RECTANGLE(id, x1,y1,x2,y2, penWidth, penStyle, penColor) \
  LINE(id, x1, y1, x1, y2, penWidth, penStyle, penColor); \
  LINE(id, x1, y2, x2, y2, penWidth, penStyle, penColor); \
  LINE(id, x2, y2, x2, y1, penWidth, penStyle, penColor); \
  LINE(id, x2, y1, x1, y1, penWidth, penStyle, penColor); 

/** 
* Complex drawings should be encapsuled by this macro.
*/
#define COMPLEX_DRAWING(id,expression) \
  if (getDebugKeyTable().isActive(DebugKeyTable::send_##id##_drawing)) {\
  expression;\
  }

/**
* Indicates that all elements of the specified drawing are drawn.
* @param id A drawing id (Drawings::FieldDrawing or Drawings::ImageDrawing)
*/
#define DEBUG_DRAWING_FINISHED(id) \
  INFO(send_##id##_drawing, idDebugDrawingFinished, bin, (char)Drawings::id << (char)Drawings::getTypeOfDrawing(Drawings::id) );

#endif //NODEBUGDRAWINGS











#endif //__DebugDrawings_h_

/*
* Change log :
* 
* $Log: DebugDrawings.h,v $
* Revision 1.4  2004/06/27 14:53:51  altmeyer
* new GoaliePositionBenjamin and new debug key send_goaliePositionField
*
* Revision 1.3  2004/06/15 10:58:28  thomas
* added edge-specialist, edges-percept, debug-drawings etc. (not yet called from image-processor)
*
* Revision 1.2  2004/06/14 20:12:10  jhoffman
* - numerous changes and additions to headcontrol
* - cameraInfo default constructor now creates ERS7 info
* - debug drawing "headcontrolfield" added
*
* Revision 1.1.1.1  2004/05/22 17:36:03  cvsadm
* created new repository GT2004_WM
*
* Revision 1.12  2004/04/27 19:39:37  roefer
* Support for debug images added to simulator
*
* Revision 1.11  2004/04/08 15:33:08  wachter
* GT04 checkin of Microsoft-Hellounds
*
* Revision 1.10  2004/03/21 19:07:49  roefer
* More realistic camera images for ERS-7
*
* Revision 1.9  2004/02/12 14:29:44  juengel
* Added drawing ball4.
*
* Revision 1.8  2004/02/06 10:16:23  juengel
* percepts for image view are painted in different debug drawings now
*
* Revision 1.7  2004/01/24 18:35:03  wachter
* Reactivated debug-drawings in SensorFusionTeamBallLocator
*
* Revision 1.6  2004/01/20 15:46:17  tim
* Added drawing for potential field A* search
*
* Revision 1.5  2003/12/01 16:24:18  juengel
* Added drawing ball3.
*
* Revision 1.4  2003/11/05 16:44:00  juengel
* Added drawing imageProcessor_ball2
*
* Revision 1.3  2003/11/03 20:19:22  juengel
* Added color class yellowOrange
*
* Revision 1.2  2003/10/29 13:31:34  juengel
* added drawing imageProcessor_coloredSegments
*
* Revision 1.1  2003/10/07 10:13:22  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.7  2003/09/30 17:42:01  juengel
* changed names of drawings
*
* Revision 1.6  2003/09/01 15:18:21  juengel
* Added drawings scanLines, calibration1, calibration2.
*
* Revision 1.5  2003/09/01 10:23:14  juengel
* DebugDrawings clean-up 2
* DebugImages clean-up
* MessageIDs clean-up
* Stopwatch clean-up
*
* Revision 1.4  2003/08/30 10:16:41  juengel
* DebugDrawings clean-up 1
*
* Revision 1.3  2003/07/29 12:43:50  juengel
* Changed order
*
* Revision 1.2  2003/07/03 10:50:17  juengel
* Added Drawing kickAnglesRadar.
*
* Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.49  2003/06/24 11:42:22  jhoffman
* empty define for ARROW-makro  for "debug no debugdrawings" added
*
* Revision 1.48  2003/06/24 10:10:16  jhoffman
* added arrow so it can be sent
*
* Revision 1.47  2003/06/19 18:35:14  juengel
* Added drawing kickAngles.
*
* Revision 1.46  2003/06/13 14:49:34  juengel
* Added jointStateDrawing.
*
* Revision 1.45  2003/06/10 18:01:53  juengel
* Added LARGE_DOT.
*
* Revision 1.44  2003/06/05 08:15:10  juengel
* Added "corridors" drawing.
*
* Revision 1.43  2003/05/26 08:25:36  juengel
* Added drawing obstaclesModelImage
* COMPLEX_DRAWINGs are only drawn when requested under WIN_32
*
* Revision 1.42  2003/05/16 14:07:48  risler
* added radar view obstacles drawing
*
* Revision 1.41  2003/05/09 16:21:58  engel
* DebugKey changed for the GaussBellTeamBallLocator
*
* Revision 1.40  2003/05/05 11:56:29  dueffert
* now using ifdef instead of if
*
* Revision 1.39  2003/05/01 23:33:56  roefer
* Anchors for flag specialist added
*
* Revision 1.38  2003/04/30 09:17:08  kallnik
* renamed some DebugKeys and defines
*
* Revision 1.37  2003/04/25 12:57:14  kallnik
* new version of ContinuousBasicBehavior released
*
* Revision 1.36  2003/04/15 15:52:07  risler
* DDD GO 2003 code integrated
*
* Revision 1.35  2003/04/09 15:51:57  kallnik
* ContinuousBasicBehavior Debug Drawing   for   ContinuousBasicBehaviorTester
*
* Revision 1.35  2003/04/05 16:49:39  juengel
* Added image drawing "specialPercept"
*
* Revision 1.34  2003/03/29 21:06:41  juengel
* Added obstaclesAndPSDPercept.
*
* Revision 1.33  2003/03/28 15:07:41  juengel
* Added preprocessor directive NO_DEBUG_DRAWINGS
*
* Revision 1.32  2003/03/28 14:13:58  loetzsch
* added debug drawing areaOfResponsibility
*
* Revision 1.31  2003/03/25 10:56:40  juengel
* DebugDrawings have to be requested in configuration "WIN_32" now.
*
* Revision 1.30  2003/03/24 11:09:10  schumann
* added missing descriptor of StrategicalDatabaseMove drawing
*
* Revision 1.29  2003/03/21 12:22:21  thiel
* added hexAreaPotentialfieldDrawing
*
* Revision 1.28  2003/03/20 14:06:51  roefer
* Localization experiment and view for segmented image added
*
* Revision 1.27  2003/03/19 20:16:19  schumann
* added strategical database drawing
*
* Revision 1.26  2003/03/19 16:41:27  jhoffman
* fixed debug drawing error
*
* Revision 1.25  2003/03/19 15:38:50  jhoffman
* changed drawing name
*
* Revision 1.24  2003/03/15 13:25:49  juengel
* Added descriptions for several debug drawings and debug keys.
*
* Revision 1.23  2003/03/11 11:31:11  juengel
* Added obstaclesModel
*
* Revision 1.22  2003/03/07 15:35:59  wachter
* Added debug-drawings for SensorFusionBall and PlayerLocator.
*
* Revision 1.21  2003/03/05 14:07:10  roefer
* LinesPerceptor2 is now LinesImageProcessor
*
* Revision 1.20  2003/03/05 10:35:35  jhoffman
* added debugdrawing
*
* Revision 1.19  2003/02/27 12:56:35  juengel
* Added some drawings.
*
* Revision 1.18  2003/02/19 15:01:23  roefer
* LinesPerceptor2 added
*
* Revision 1.17  2003/02/08 18:40:06  juengel
* added imageProcessorDrawing
*
* Revision 1.16  2003/01/15 13:47:29  roefer
* Progress in LinesSelfLocator, new debug drawing
*
* Revision 1.15  2003/01/13 18:26:04  juengel
* Added LineFollowerField drawing
*
* Revision 1.14  2002/12/18 16:49:30  dueffert
* doxygen docu corrected
*
* Revision 1.13  2002/12/18 16:22:56  dueffert
* doxygen docu corrected
*
* Revision 1.12  2002/12/17 20:58:38  roefer
* COMPLEX_DRAWING changed
*
* Revision 1.11  2002/12/14 19:44:55  roefer
* Lines drawing in image coordinates removed
*
* Revision 1.10  2002/12/01 17:35:05  juengel
* Added QUADRANGLE(...)
*
* Revision 1.9  2002/11/28 20:42:47  risler
* added walking drawing
*
* Revision 1.8  2002/11/20 15:46:27  juengel
* PathPlanner added.
*
* Revision 1.7  2002/11/19 17:38:31  dueffert
* doxygen bugs corrected
*
* Revision 1.6  2002/11/18 19:24:16  juengel
* Line Follower added.
*
* Revision 1.5  2002/11/18 10:33:01  juengel
* Comments added.
*
* Revision 1.4  2002/11/12 16:08:11  loetzsch
* Added (char) and (int) casts to DebugDrawing Macros
*
* Revision 1.3  2002/11/12 10:49:02  juengel
* New debug drawing macros - second step
* -moved /Tools/Debugging/PaintMethods.h and . cpp
*  to /Visualization/DrawingMethods.h and .cpp
* -moved DebugDrawing.h and .cpp from /Tools/Debugging/
*  to /Visualization
*
* Revision 1.2  2002/11/11 11:27:10  juengel
* First Step: New debug drawing macros.
*
* Revision 1.1  2002/09/19 23:49:08  juengel
* Changed debug image mechanisms.
*
*/
