/**
* @file ContinuousRule.h
*
* the base class for all continuous rules
* 
* @author Martin Kallnik
*/

#ifndef __ContinuousRule_h_
#define __ContinuousRule_h_

#include "Modules/BehaviorControl/BehaviorControl.h"
#include "Tools/Xabsl2/Xabsl2Engine/Xabsl2Array.h"
#include "Tools/RobotConfiguration.h"

/**@class ContinuousRuleBase
 * Base class for all continuous rules.
 */
class ContinuousRule: public BehaviorControlInterfaces, public Xabsl2NamedItem{
public:

	/** Constructor */
  ContinuousRule(const BehaviorControlInterfaces& interfaces, const char* name)
    :BehaviorControlInterfaces(interfaces), Xabsl2NamedItem(name),
     bodyLength(getRobotConfiguration().getRobotDimensions().bodyLength)
  {};
  
	/** Destructor */
  ~ContinuousRule(){};

	/** executes the basic behavior rule
   * @param robotPose the Robotpose, needed for DebugDrawings
	 * @param walk the direction and speed of the suggested motion as an vector (output)
	 * @param rotationAngle the rotation angle, the direction the robot should be directed (output)
	 * @param rotationWeight the rotation weight the weight of the succested rotation (output)
	 */
  virtual void execute(const RobotPose& robotPose,
                       Vector2<double>& walk,
                       double& rotationAngle,
                       double& rotationWeight) = 0;

  /**
   * Returns whether this rule generates absolute or robot relative coordinates.
   */
  virtual bool isRelative() {return false;}

  /**
   * Returns whether this rule affect the walk speed of the basic behavior.
   * Overwrite and return false if rule should only influence direction. 
   */
  virtual bool affectSpeed() {return true;}

protected:
  /**
   * Calculates the force of a repelling object.
   * Implemented with a simple linear function.
   * @param distance the distance from the repelling object (in mm)
	 * @param maxForce the maximum force
	 * @param maxForceLength the distance below which the force is maxForce (in mm)
	 * @param influenceLength the distance above which the force is 0 (in mm)
   */
  double getRepellingForce(double distance, double maxForce, double maxForceLength, double influenceLength)
  {
    if (distance < maxForceLength)
      return maxForce;
    else if (distance < influenceLength)
      return maxForce * (influenceLength - distance) / (influenceLength - maxForceLength);
    else
      return 0.0;
  }

  /**
   * Calculates the force of an attracting object.
   * Implemented with a simple linear function.
   * @param distance the distance to the attracting object (in mm)
	 * @param maxForce the maximum force
	 * @param minDistance the distance below which the force is 0 (in mm)
	 * @param maxForceLength the distance above which the force is maxForce (in mm)
   */
  double getAttractingForce(double distance, double maxForce, double minDistance, double maxForceLength)
  {
    if (distance < minDistance)
      return 0.0;
    else if (distance < maxForceLength)
      return maxForce * (distance - minDistance) / (maxForceLength - minDistance);
    else
      return maxForce;
  }

  const double bodyLength; /**< The length of the robot's body, used in many rules. */
};

#endif //__ContinuousRule_h_


/*
* Change log:
*
* $Log: ContinuousRule.h,v $
* Revision 1.1.1.1  2004/05/22 17:36:02  cvsadm
* created new repository GT2004_WM
*
* Revision 1.3  2004/03/08 01:07:18  roefer
* Interfaces should be const
*
* Revision 1.2  2004/01/01 10:58:51  roefer
* RobotDimensions are in a class now
*
* Revision 1.1  2003/10/07 10:13:22  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.9  2003/06/03 15:51:02  risler
* separated calculation of speed and direction => rules may now affect direction only
*
* Revision 1.8  2003/06/03 10:51:24  risler
* ContinuousRules can be in relative coordinates
* no rotation rule active => no rotation
*
* Revision 1.7  2003/05/19 10:29:47  risler
* added max speeds to continuousbasicbehaviors
*
* Revision 1.6  2003/05/09 15:25:08  risler
* ContinuousBasicBehavior uses Xabsl2Array,
* ContinuousRules inherit from Xabsl2NamedItem
*
* Revision 1.5  2003/05/07 11:31:04  risler
* comments fixed
*
* Revision 1.4  2003/05/07 11:05:55  risler
* ContinuousBasicBehaviors:
* removed classes Attracting/RepellingFieldRule, replaced with getAttracting/RepellingForce functions
* some clean up
*
* Revision 1.3  2003/05/07 10:46:38  kallnik
* warnings removed
*
* Revision 1.2  2003/05/07 10:02:09  kallnik
* ContinuousBasicBehavior:
* -DebugDrawings fixed
* -copyfiles: do no copy Config/ContBeh (files deleted)
*
* Revision 1.1  2003/05/06 17:03:43  kallnik
* ContinuousBasicBehaviors code review
*
* Revision 1.7  2003/04/25 12:57:14  kallnik
* new version of ContinuousBasicBehavior released
*
* Revision 1.6  2003/04/15 15:52:11  risler
* DDD GO 2003 code integrated
*
* Revision 1.7  2003/04/07 22:06:29  dthomas
* fix: changed limitRotationSpeed to protected
*
* Revision 1.6  2003/04/07 22:01:12  dthomas
* updated: rules with linear function and fixed rotation limit
*
* Revision 1.5  2003/03/04 14:17:31  kallnik
* NEW CONTINUOUS RULES
*
* Revision 1.4  2003/02/25 12:54:20  dueffert
* spelling mistakes and Greenhills compatibility fixed
*
* Revision 1.3  2003/02/25 11:43:44  dueffert
* line feed and doxygen bugs fixed
*
* Revision 1.2  2003/02/25 10:11:28  dueffert
* doxygen bugs fixed
*
* Revision 1.1  2003/02/20 12:02:28  kallnik
* ContinuousBasicBehavior added
*
*
*/
