/**
* @file ContinuousBasicBehavior.cpp
*
* @author Martin Kallnik
*/


#include "ContinuousBasicBehavior.h"
#include "Tools/Debugging/DebugDrawings.h"
#include "Tools/FieldDimensions.h"
//#include "Platform/GTAssert.h"
 


//too much points leads to wrong behavior (do not know why)

#define CBB_PARTS_X 5 /*number of vectors in x-direction per half field*/
#define CBB_PARTS_Y 3 /*number of vectors in y-direction per half field*/
#define CBB_LENGTH  100 /*length of vectors in mm*/

#define CBB_WAY_LENGTH 20 /*maximum number of steps in CBBWay drawing*/
#define CBB_WAY_STILL 20 /*if calculated step is shorter Way stops*/

ContinuousBasicBehavior::ContinuousBasicBehavior(const BehaviorControlInterfaces& interfaces,
                                                 Xabsl2ErrorHandler& errorHandler,
                                                 char* basicBehaviorName)
    :Xabsl2BasicBehavior(basicBehaviorName,errorHandler),
    BehaviorControlInterfaces(interfaces),
    walkType(WalkRequest::normal),
    maxSpeed(walkMaxForwardSpeed),
    maxRotationSpeed(walkMaxRotationSpeed),
    evaluationMode(discrete),
    smooth(1),
    backwards(true),
    maxAccX(300.0),
    maxAccY(300.0),
    maxAccRot(180.0),
    accelerationRestrictor(motionRequest)
{}

void ContinuousBasicBehavior::setWalkType(WalkRequest::WalkType walkType)
{
  this->walkType = walkType;
}

void ContinuousBasicBehavior::setMaxSpeed(double maxSpeed)
{
  if (maxSpeed == 0 || maxSpeed > walkMaxForwardSpeed)
    this->maxSpeed = walkMaxForwardSpeed;
  else
    this->maxSpeed = maxSpeed;
}

void ContinuousBasicBehavior::setMaxRotationSpeed(double maxRotationSpeed)
{
  if (maxRotationSpeed == 0 || maxRotationSpeed > walkMaxRotationSpeed)
    this->maxRotationSpeed = walkMaxRotationSpeed;
  else
    this->maxRotationSpeed = maxRotationSpeed;
}

void ContinuousBasicBehavior::setEvaluationMode(EvaluationMode evaluationMode)
{
  this->evaluationMode = evaluationMode;
}


void ContinuousBasicBehavior::setSmoothing(int smooth)
{
  this->smooth = smooth;
}


void ContinuousBasicBehavior::enableBackwards(bool backwards)
{
  this->backwards = backwards;
}

void ContinuousBasicBehavior::setAccelerations(double maxAccX, double maxAccY, double maxAccRot)
{
  this->maxAccX = maxAccX;
  this->maxAccY = maxAccY;
  this->maxAccRot = maxAccRot;
}


void ContinuousBasicBehavior::addRule(ContinuousRule *pRule)
{
  if (pRule->isRelative())
    pRulesRelative.append(pRule->n, pRule);
  else
    pRules.append(pRule->n, pRule);
}

ContinuousBasicBehavior::~ContinuousBasicBehavior()
{
  int i;

  for(i=0; i<pRules.getSize(); i++)
  {
    delete pRules[i];
  }

  for(i=0; i<pRulesRelative.getSize(); i++)
  {
    delete pRulesRelative[i];
  }
}

void ContinuousBasicBehavior::execute()
{
   accelerationRestrictor.saveLastWalkParameters();

 // it is not useful to make these debug drawings in sample mode!
  if (evaluationMode!=sample)
  {
    //DebugDrawings
    COMPLEX_DRAWING(behavior_CBBVector,drawVectorField(););
    DEBUG_DRAWING_FINISHED(behavior_CBBVector); 

    COMPLEX_DRAWING(behavior_CBBWay,drawWay(););
    DEBUG_DRAWING_FINISHED(behavior_CBBWay); 
  }

  
  if (smooth<=1)
    executeAt(robotPose, motionRequest, false); 
  else
  {
    RobotPose tmpRobotPose = robotPose;
    Vector2<double> walk;
 

    for (int i=0 ; i < smooth; i++){
      executeAt(tmpRobotPose, motionRequest, false); 
  
      walk=motionRequest.walkRequest.walkParams.translation;

      if (walk.abs()>walkMaxForwardSpeed) walk.normalize(walkMaxForwardSpeed);

      walk /= (double) smooth * 0.5;


      tmpRobotPose.translation.x += walk.x * cos(tmpRobotPose.rotation) - walk.y * sin(tmpRobotPose.rotation);
      tmpRobotPose.translation.y += walk.x * sin(tmpRobotPose.rotation) + walk.y * cos(-tmpRobotPose.rotation);
      tmpRobotPose.rotation=normalize(motionRequest.walkRequest.walkParams.rotation+tmpRobotPose.rotation);

    }

    //way = tmpRobotPose.translation - robotPose.translation
    //    = endpoint after smooth steps - startposition

    //rotation =  tmpRobotPose.rotation - robotPose.rotation

    walk= tmpRobotPose.translation -robotPose.translation;

    motionRequest.walkRequest.walkParams.translation.x=  walk.x * cos(robotPose.rotation) + walk.y * sin(robotPose.rotation);
    motionRequest.walkRequest.walkParams.translation.y= -walk.x * sin(robotPose.rotation) + walk.y * cos(robotPose.rotation);

    motionRequest.walkRequest.walkParams.rotation= 4.0 * normalize(tmpRobotPose.rotation - robotPose.rotation);
  }

  accelerationRestrictor.restrictAccelerations(maxAccX,maxAccY,maxAccRot);

}

void ContinuousBasicBehavior::executeAt(const RobotPose& robotPose, MotionRequest& motionRequest, const bool suppressPrintRules){
  // scratch head if no rules were set
  if (pRules.getSize() == 0 && pRulesRelative.getSize() == 0)
  {
    motionRequest.motionType = MotionRequest::specialAction;
    motionRequest.specialActionRequest.specialActionType = SpecialActionRequest::demoScratchHead;
    return;
  }
  
  int i; //counter for rules
  
  Vector2<double> walk,wrobot, rotation, rotrobot, speed, speedrobot;
  
  walk=rotation=speed=Vector2<double>(0.0,0.0);
  
  double dRotationAngle, dRotationWeight;
  Vector2<double> dWalk;
  
  if (!suppressPrintRules) 
  {  
    INFO(printCBBRules,idText,text,"Results for rules of "<<n<<" :");
  }
  
  // evaluate absolute rules

  if ((evaluationMode==sample) && (selfLocatorSamples.getNumberOfSamples()>0)) 
  {
    // execution in sample mode
    
    double sumOfProbabilities=0;
    Vector2<double> tempWalk, tempRotation, tempSpeed;
    for (int j=0;j<selfLocatorSamples.getNumberOfSamples();j++)
    {
      tempWalk = tempRotation = tempSpeed = Vector2<double>(0.0,0.0);
      sumOfProbabilities+=selfLocatorSamples[j].probability;
      for(i=0; i<pRules.getSize(); i++)
      {
        dWalk=Vector2<double>(0.0,0.0);
        dRotationAngle=0.0;
        dRotationWeight=0.0;
        
        pRules[i]->execute(selfLocatorSamples[j],dWalk,dRotationAngle,dRotationWeight);
        
        if (!suppressPrintRules)
        {
          INFO(printCBBRules,idText,text,
            "Rule "<<pRules.getName(i)<<": walk=("<<dWalk.x<<","<<dWalk.y<<")");
        }
        
        tempWalk+=dWalk;
        if (pRules[i]->affectSpeed()) 
          tempSpeed+=dWalk;
        tempRotation+=Vector2<double>(cos(dRotationAngle)*dRotationWeight,
          sin(dRotationAngle)*dRotationWeight);
      }
      
      walk += tempWalk * selfLocatorSamples[j].probability;
      speed += tempSpeed * selfLocatorSamples[j].probability;
      rotation += tempRotation * selfLocatorSamples[j].probability;
    }
    
    walk /= sumOfProbabilities;
    speed /= sumOfProbabilities;
    
  } else {
    // normal (discrete) execution
    for(i=0; i<pRules.getSize(); i++)
    {
      dWalk=Vector2<double>(0.0,0.0);
      dRotationAngle=0.0;
      dRotationWeight=0.0;
      
      pRules[i]->execute(robotPose,dWalk,dRotationAngle,dRotationWeight);
      
      if (!suppressPrintRules)
      {
        INFO(printCBBRules,idText,text,
          "Rule "<<pRules.getName(i)<<": walk=("<<dWalk.x<<","<<dWalk.y<<")");
      }
      
      walk+=dWalk;
      if (pRules[i]->affectSpeed()) speed+=dWalk;
      rotation+=Vector2<double>(cos(dRotationAngle)*dRotationWeight,
        sin(dRotationAngle)*dRotationWeight);
    }
    
  }


  // transform from worldcoordinates to robotcoordinates
  
  double c = cos(robotPose.rotation);
  double s = sin(robotPose.rotation);

  wrobot.x= walk.x * c + walk.y * s;
  wrobot.y= -walk.x * s + walk.y * c;

  speedrobot.x= speed.x * c + speed.y * s;
  speedrobot.y= -speed.x * s + speed.y * c;

  rotrobot.x= rotation.x * c + rotation.y * s;
  rotrobot.y= -rotation.x * s + rotation.y * c;
  

  // evaluate relative rules

  for(i=0; i<pRulesRelative.getSize(); i++)
  {
    dWalk=Vector2<double>(0.0,0.0);
    dRotationAngle=0.0;
    dRotationWeight=0.0;
    
    pRulesRelative[i]->execute(robotPose,dWalk,dRotationAngle,dRotationWeight);
    
    if (!suppressPrintRules)
    {
      INFO(printCBBRules,idText,text,
        "Relative Rule "<<pRulesRelative.getName(i)<<": walk=("<<dWalk.x<<","<<dWalk.y<<")");
    }
    
    wrobot+=dWalk;
    if (pRulesRelative[i]->affectSpeed()) 
      speedrobot+=dWalk;
    rotrobot+=Vector2<double>(cos(dRotationAngle)*dRotationWeight,
      sin(dRotationAngle)*dRotationWeight);
  }
  
  motionRequest.motionType=MotionRequest::walk;
  motionRequest.walkRequest.walkType=walkType;
  
  wrobot.normalize();
  motionRequest.walkRequest.walkParams.translation= wrobot * speedrobot.abs() * walkMaxForwardSpeed;
  
  if (rotrobot.x == 0 && rotrobot.y == 0)
    motionRequest.walkRequest.walkParams.rotation = 0;
  else
    motionRequest.walkRequest.walkParams.rotation = normalize(rotrobot.angle());

  // clip to max speed
  if ( !backwards && motionRequest.walkRequest.walkParams.translation.x < 0)
    motionRequest.walkRequest.walkParams.translation.x = 0;

  if ( motionRequest.walkRequest.walkParams.translation.abs() > maxSpeed )
  {
    motionRequest.walkRequest.walkParams.translation.normalize();
    motionRequest.walkRequest.walkParams.translation *= maxSpeed;
  }

  if ( motionRequest.walkRequest.walkParams.rotation > maxRotationSpeed )
    motionRequest.walkRequest.walkParams.rotation = maxRotationSpeed;
  else if ( motionRequest.walkRequest.walkParams.rotation < -maxRotationSpeed )
    motionRequest.walkRequest.walkParams.rotation = -maxRotationSpeed;

  // complete stop on minimal motion
  if ( fabs(motionRequest.walkRequest.walkParams.rotation) < fromDegrees(10) 
    && motionRequest.walkRequest.walkParams.translation.abs() < 20 )
  {
    motionRequest.walkRequest.walkParams.rotation = 0;
    motionRequest.walkRequest.walkParams.translation.x = 0;
    motionRequest.walkRequest.walkParams.translation.y = 0;
  }
  
  if (!suppressPrintRules)
  {
    INFO(printCBBRules,idText,text,
      "Result of ContinuousBasicBehavior "<<n<<": walk=("
      <<walk.x<<","<<walk.y<<"), angle="<<toDegrees(rotation.angle())<<"");
  }
  
}


//vectorfield
void ContinuousBasicBehavior::drawVectorField(){
  int i,j;
  //Vector2<int> start, endr,endw,arrow1,arrow2,at;
  Vector2<double> start, endr,endw,arrow1,arrow2,at;
  Vector2<double> walk;
  RobotPose tmpRobotPose;
  double rot;
  
  tmpRobotPose = robotPose;
  
  
  
  for(i= -CBB_PARTS_X; i<= CBB_PARTS_X; i++){
    start.x = i* xPosOpponentGroundline/CBB_PARTS_X;
    tmpRobotPose.translation.x=start.x;
    
    for (j= -CBB_PARTS_Y ; j <= CBB_PARTS_Y; j++){
      start.y = j* yPosLeftSideline/CBB_PARTS_Y;
      tmpRobotPose.translation.y=start.y;
      
      executeAt(tmpRobotPose, motionRequest, true);
      
      //endw berechnen
      
      walk=motionRequest.walkRequest.walkParams.translation / walkMaxForwardSpeed;
      
      
      endw.x=( walk.x * cos(tmpRobotPose.rotation) - walk.y * sin(tmpRobotPose.rotation))*CBB_LENGTH;
      endw.y=( walk.x * sin(tmpRobotPose.rotation) + walk.y * cos(-tmpRobotPose.rotation))*CBB_LENGTH;
      
      //Pfeile berechnen
      
      at=endw;
      at.normalize(CBB_LENGTH/2);
      arrow2=arrow1=at=endw-at;
      
      at.normalize(CBB_LENGTH/2);
      at.transpose();
      
      arrow1=arrow1+at;
      arrow2=arrow2-at;
      
      //werte absulot machen
      
      arrow1+=start;
      arrow2+=start;
      endw+=start;
      
      //endr berechnen
      
      rot=normalize(motionRequest.walkRequest.walkParams.rotation+robotPose.rotation);
      endr.x =(int)( cos(rot) *CBB_LENGTH )+ start.x;
      endr.y =(int)( sin(rot) *CBB_LENGTH )+ start.y;
      
      
      //paint 
      
      //main line
      LINE(behavior_CBBVector,
        start.x,start.y, endw.x, endw.y, 2,
        Drawings::ps_solid, Drawings::black);
      
      // orientation
      LINE(behavior_CBBVector,
        start.x,start.y, endr.x, endr.y, 2,
        Drawings::ps_solid, Drawings::red);
      
      //wings
      LINE(behavior_CBBVector,
        arrow1.x,arrow1.y, endw.x, endw.y, 2,
        Drawings::ps_solid, Drawings::black);
      
      LINE(behavior_CBBVector,
        arrow2.x,arrow2.y, endw.x, endw.y, 2,
        Drawings::ps_solid, Drawings::black);
      
    }
  }
}


//way
void ContinuousBasicBehavior::drawWay(){
  bool end= false;
  RobotPose tmpRobotPose;

  int runnings=0;

  Vector2<int> start, endr,endw;
  Vector2<double> walk, walkabs;
 
  double rot;

  tmpRobotPose=robotPose;
  start.x=(int)(tmpRobotPose.translation.x+0.5);
  start.y=(int)(tmpRobotPose.translation.y+0.5);


  while(!end){
 
    executeAt(tmpRobotPose, motionRequest, true);

    //endw berechnen
 
    walk=motionRequest.walkRequest.walkParams.translation;

    if (walk.abs()>walkMaxForwardSpeed) walk.normalize(walkMaxForwardSpeed);

    walkabs.x= walk.x * cos(tmpRobotPose.rotation) - walk.y * sin(tmpRobotPose.rotation);
    walkabs.y= walk.x * sin(tmpRobotPose.rotation) + walk.y * cos(-tmpRobotPose.rotation);

    //testing 
    walkabs /=4.0;

    endw.x=(int)( walkabs.x+0.5);
    endw.y=(int)( walkabs.y+0.5);

    //werte absolut machen
    endw+=start;


    //endr berechnen

    rot=normalize(motionRequest.walkRequest.walkParams.rotation+robotPose.rotation);
    endr.x =(int)( cos(rot) *CBB_LENGTH )+ start.x;
    endr.y =(int)( sin(rot) *CBB_LENGTH )+ start.y;


    //paint 
    // orientation
 /*   LINE(CBBWay,
    start.x,start.y, endr.x, endr.y, 1,
    Drawings::ps_solid, Drawings::red);
*/
  
    //main line
    LINE(behavior_CBBWay,
         start.x,start.y, endw.x, endw.y, 2,
         Drawings::ps_solid, Drawings::black);


    //update tmpRobotPose 

    tmpRobotPose.translation+=walkabs;
    tmpRobotPose.rotation=rot;

    //end?
    runnings++; 
    end=(walkabs.abs()<=CBB_WAY_LENGTH)||(runnings>CBB_WAY_LENGTH);
    
    start=endw;

  }
 
}

/*
* Change Log:
*
* $Log: ContinuousBasicBehavior.cpp,v $
* Revision 1.4  2004/06/02 17:18:23  spranger
* MotionRequest cleanup
*
* Revision 1.3  2004/05/27 19:37:17  loetzsch
* compilation fixed
*
* Revision 1.2  2004/05/27 13:53:39  loetzsch
* bug fix
*
* Revision 1.1.1.1  2004/05/22 17:36:02  cvsadm
* created new repository GT2004_WM
*
* Revision 1.2  2004/03/08 01:07:17  roefer
* Interfaces should be const
*
* Revision 1.1  2003/10/07 10:13:21  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.3  2003/09/01 10:23:13  juengel
* DebugDrawings clean-up 2
* DebugImages clean-up
* MessageIDs clean-up
* Stopwatch clean-up
*
* Revision 1.2  2003/08/09 14:53:10  dueffert
* files and docu beautified
*
* Revision 1.1.1.1  2003/07/02 09:40:28  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.17  2003/06/19 16:08:02  risler
* increased continuous behavior rotation speed
*
* Revision 1.16  2003/06/05 09:22:25  kallnik
* ContinuousBasicBehavior:
* Smoothing added
*
* Revision 1.15  2003/06/04 11:16:47  risler
* added acceleration restriction, enableBackwards
*
* Revision 1.14  2003/06/03 15:51:02  risler
* separated calculation of speed and direction => rules may now affect direction only
*
* Revision 1.13  2003/06/03 10:51:24  risler
* ContinuousRules can be in relative coordinates
* no rotation rule active => no rotation
*
* Revision 1.12  2003/05/25 11:35:39  risler
* maxSpeed added to cont-go-to-point,cont-go-forward-to-point
*
* Revision 1.11  2003/05/19 11:12:56  brunn
* moved routines for sample mode from execute to executeAt to avoid some trigonometric functions
* disabled debug drawings in sample mode
*
* Revision 1.10  2003/05/19 10:29:46  risler
* added max speeds to continuousbasicbehaviors
*
* Revision 1.9  2003/05/13 14:45:16  brunn
* added evaluationMode
* default evaluationMode is discrete which uses only the robotPose
* added sample evaluation Mode which uses all selflocatorSamples
*
* Revision 1.8  2003/05/11 16:56:42  risler
* delete rules in destructor
*
* Revision 1.7  2003/05/11 09:40:07  risler
* no message
*
* Revision 1.6  2003/05/09 15:25:08  risler
* ContinuousBasicBehavior uses Xabsl2Array,
* ContinuousRules inherit from Xabsl2NamedItem
*
* Revision 1.5  2003/05/09 12:59:36  loetzsch
* removed useless #include statements
*
* Revision 1.4  2003/05/07 11:05:55  risler
* ContinuousBasicBehaviors:
* removed classes Attracting/RepellingFieldRule, replaced with getAttracting/RepellingForce functions
* some clean up
*
* Revision 1.3  2003/05/07 10:46:38  kallnik
* warnings removed
*
* Revision 1.2  2003/05/07 10:02:09  kallnik
* ContinuousBasicBehavior:
* -DebugDrawings fixed
* -copyfiles: do no copy Config/ContBeh/ (files deleted)
*
* Revision 1.1  2003/05/06 17:03:43  kallnik
* ContinuousBasicBehaviors code review
*
* Revision 1.24  2003/04/30 09:17:08  kallnik
* renamed some DebugKeys and defines
*
* Revision 1.23  2003/04/29 13:35:30  kallnik
* CBB Bugfix
*
* Revision 1.22  2003/04/28 11:01:02  kallnik
* new ContinuousBasicBehavior version (NORules is taken from number of lines in Rules-section
* and some bugfixes
*
* Revision 1.21  2003/04/25 12:57:14  kallnik
* new version of ContinuousBasicBehavior released
*
* Revision 1.20  2003/04/15 15:52:11  risler
* DDD GO 2003 code integrated
*
* Revision 1.28  2003/04/09 16:44:45  kallnik
* DebugDrawing moved from ContinuousBasicBehaviorTester to ContinuousBasicBehavior
*
* Revision 1.27  2003/04/09 15:51:57  kallnik
* ContinuousBasicBehavior Debug Drawing   for   ContinuousBasicBehaviorTester
*
* Revision 1.26  2003/04/07 21:59:39  dthomas
* added: no move if vector and angle are too small
*
* Revision 1.25  2003/04/07 08:41:48  max
* dont turn when no rule has rotation weight
*
* Revision 1.24  2003/04/03 09:41:39  max
* minor cleanup
*
* Revision 1.23  2003/04/02 19:03:07  max
* bugfix: wrong config files wont crash anymore
* bugfix: crash due to uninitaliased dra in execute removed
*
* Revision 1.22  2003/04/02 15:19:04  dthomas
* modified: continuous behaviour of goalie
*
* Revision 1.21  2003/04/02 12:11:14  max
* some rules added, modified
*
* Revision 1.20  2003/03/28 14:43:56  max
* tamara update
*
* Revision 1.19  2003/03/28 14:21:52  risler
* rotation rule weight bug fixed
*
* Revision 1.18  2003/03/20 17:40:04  risler
* sign bug fixed
*
* Revision 1.17  2003/03/12 22:58:41  roefer
* Destructor added, memory leak closed
*
* Revision 1.16  2003/03/12 12:35:27  kallnik
* new continuousBasicBehavior config files
*
* Revision 1.15  2003/03/11 14:59:01  kallnik
* Fehlerresistenz erhht
*
* Revision 1.14  2003/03/11 14:36:16  kallnik
* ContinuousBasicBehavior:
* copyfiles angepasst (Konfigurationsdateien)
* Konstruktor angepasst
*
* Revision 1.13  2003/03/10 17:23:51  risler
* bug fixed reading configuration
*
* Revision 1.12  2003/03/10 17:20:13  risler
* bug fixed reading configuration
*
* Revision 1.11  2003/03/06 12:01:00  dueffert
* Greenhills compilability restored, typo fixed, semicolon warnings removed, initialization added
*
* Revision 1.10  2003/03/04 15:11:33  kallnik
* bugs fixed
*
* Revision 1.9  2003/03/04 14:51:52  kallnik
* addition because of problems with InConfigFile
*
* Revision 1.8  2003/03/04 14:18:02  kallnik
* NEW CONTINUOUS RULES
*
* Revision 1.7  2003/03/03 15:08:51  dueffert
* warnings removed
*
* Revision 1.6  2003/02/25 12:54:20  dueffert
* spelling mistakes and Greenhills compatibility fixed
*
* Revision 1.5  2003/02/24 14:48:17  kallnik
* some bugs in continuousbasicbehaviors fixed
*
* Revision 1.4  2003/02/20 15:55:17  dueffert
* bugs fixed
*
* Revision 1.3  2003/02/20 15:39:58  loetzsch
* ein Paar Sachen auskommentiert, weil es nicht kompiliert hat.
*
*/
