/**
 * @file Boundary.h
 *
 * This file contains the template class Boundary.
 *
 * @author <A href=mailto:roefer@tzi.de>Thomas Rfer</A>
 */

#ifndef __BOUNDARY_H__
#define __BOUNDARY_H__

#include "Tools/Math/Vector2.h"
#include "Range.h"

/**
 * The template class represents rectangular boundaries.
 */
template<class T> class Boundary
{
  public:
    Range<T> x, /**< The range in x-direction. */
             y; /**< The range in y-direction. */

    /**
     * Constructor.
     */
    Boundary() : x(0,0), y(0,0) {}

    /**
     * Constructor.
     * This constructor allows to specify the minimum and maximum values
     * for type T, e.g. -HUGE_VAL and HUGE_VAL for double. These limits 
     * are used to construct the object, so that it can adapt to any boundary later.
     * @param min The minimum value of T.
     * @param max The maximum value of T.
     */
    Boundary(T min,T max)
    : x(max,min), y(max,min) {}

    /**
     * The function enlarges the boundary so that it also includes the specified point.
     * @param p The point.
     */
    void add(const Vector2<T>& p)
    {
      x.add(p.x),
      y.add(p.y);
    }

    /**
     * The function enlarges the boundary so that it also includes another boundary.
     * @param b The other boundary.
     */
    void add(const Boundary<T>& b)     // add function for adding Boundaries
    {
      x.add(b.x);
      y.add(b.y);
    }

    /**
     * The function checks whether a certain point is enclosed by the boundary 
     * @param p The point.
     * @return Lies the point inside the boundary?
     */
    bool isInside(const Vector2<T>& p) const 
      {return x.isInside(p.x) && y.isInside(p.y);}
};

#endif // __BOUNDARY_H_

/*
 * Changelog:
 * 
 * $Log: Boundary.h,v $
 * Revision 1.1.1.1  2004/05/22 17:35:49  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.2  2003/12/15 11:44:33  juengel
 * Parameters for add function changed.
 *
 * Revision 1.1  2003/10/07 10:13:20  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.1  2003/09/26 11:40:39  juengel
 * - sorted tools
 * - clean-up in DataTypes
 *
 * Revision 1.1.1.1  2003/07/02 09:40:22  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.5  2002/12/16 14:53:53  dueffert
 * changelog added
 *
 */
