/**
 * @file Simulation.h
 * 
 * Definition of class Simulation
 *
 * @author <A href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</A>
 */ 

#ifndef SIMULATION_H_
#define SIMULATION_H_

#include "Platform/Globals.h"
#include <map>
#include <string>
#include <vector>
#include <list>
#include "APIDatatypes.h"
#include "Sensorport.h"
#include "Errors.h"
#include "SimMath.h"

class Actuatorport;
class Actuator;
class View;
class Surface;
class SAX2Parser;
class SimObject;


/**
* @class Simulation
*
* The main class of SimRobotCore.
* Simulation controls and wraps all simulation tasks:
* - loading, starting, stoppping
* - visualization
* - object management
* - error management
*/
class Simulation
{
private:
  /**
  * @class Selection
  * Encapsulates information about a selection in the scene
  */
  class Selection
  {
  public:
    /** Information about the user's perspective*/
    double xAngle,
           yAngle,
           zAngle, 
           zoom, 
           fovy;
    /** Information about the display resolution*/
    int width,
        height;
    /** The selection plane and an offset*/
    Vector3d plane,
             clickOffset;
    /** The selected object and the root object*/
    SimObject* object,
             * root;

    /** Constructor*/
    Selection() : object(0) {}
  };

  /** The current selection*/
  Selection selection;

  /** A type for mapping object names to pointers*/
  typedef std::map<std::string, SimObject*> NameToPointerMap;
    
  /** A list of all sensorports*/
  std::vector<Sensorport*> sensorportList;
  /** A list of all actuatorports*/
  std::vector<Actuatorport*> actuatorportList;
  /** A list of all actuators*/
  std::vector<Actuator*> actuatorList;
  /** A list of all object surfaces*/
  std::vector<Surface*> surfaces;
  /** A list of all views to be drawn*/
  std::list<View*> viewList;
  /** A fast look-up table for object pointers*/
  NameToPointerMap objectMap;
  /** The root node of the object tree*/
  SimObject* objectTree;
  /** The file to be opened*/
  std::string filename;
  /** The dtd to be used*/
  std::string dtdname;
  /** The current status of the simulation*/
  SimulationStatus simulationStatus;
  /** A pointer to a parser for XML files*/
  SAX2Parser* parser;
  /** The error management object*/
  ErrorManager errorManager;
  /** The current simulation step*/
  int simulationStep;
  /** The time, one step takes*/
  double stepLength;
  /** A flag that stores whether the scene graph changed. */
  bool sceneGraphChanged;
  
  //TODO: Remove these two:
  double sceneViewNear;
  double sceneViewFar;

  /** The background color of the scene view*/
  Vector3d backgroundColor;

  /** The color of the ambient light*/
  Vector3d ambientColor;

  /** Helper function for object selection.
  * @param x The selected x position on the display
  * @param y The selected y position on the display
  * @param xAngle The rotation of the viewer's x-axis
  * @param yAngle The rotation of the viewer's y-axis
  * @param zAngle The rotation of the viewer's z-axis
  * @param zoom The distance from the viewer to the center 
  * @param width The width of the view (in pixels)
  * @param height The height of the view (in pixels)
  * @param fovy The vertical openingAngle in degrees
  * @param viewerPosition Returns the position of the viewer.
  * @param vecToClick Returns a ray from viewer through the scene
  *                   representing the user's click.
  */
  void projectClick(int x, int y,Vector3d& viewerPosition, Vector3d& vecToClick) const;

public:
  /** Constructor*/
  Simulation();
  /** Destructor */
  ~Simulation() {closeSimulation();}
  
  /** Draws the complete scene (or a subset) with OpenGL
  * (An OpenGL environment has to be set up before calling draw)
  * @param xAngle The rotation of the viewer's x-axis
  * @param yAngle The rotation of the viewer's y-axis
  * @param zAngle The rotation of the viewer's z-axis
  * @param zoom The distance from the viewer to the center 
  * @param width The width of the view (in pixels)
  * @param height The height of the view (in pixels)
  * @param fovy The vertical openingAngle in degrees
  * @param visParams Parameters for drawing
  * @param objectName The name of an object to be drawn instead of the scene
  */
  void draw(double xAngle, double yAngle, double zAngle, double zoom, 
            int width, int height, double fovy,
            const VisualizationParameterSet& visParams,
            const std::string& objectName="");

  /** Selects a drawn object
  * @param x The selected x position on the display
  * @param y The selected y position on the display
  * @param xAngle The rotation of the viewer's x-axis
  * @param yAngle The rotation of the viewer's y-axis
  * @param zAngle The rotation of the viewer's z-axis
  * @param zoom The distance from the viewer to the center 
  * @param width The width of the view (in pixels)
  * @param height The height of the view (in pixels)
  * @param fovy The vertical openingAngle in degrees
  * @param objectName The root object of the scene displayed
  * @param selectButtons Flag: Also select interactive buttons, if true
  * @return The type of the selection result
  */
  InteractiveSelectionType selectObject(
                    int x, int y, double xAngle, double yAngle, double zAngle, 
                    double zoom, int width, int height, double fovy, DragAndDropPlane plane,
                    const std::string& objectName, bool selectButtons=false);

  /** Removes a previous selection*/
  void unselectObject();

  /** Moves a selected object to a new position
  * @param x The selected x position on the display
  * @param y The selected y position on the display
  * @param xAngle The rotation of the viewer's x-axis
  * @param yAngle The rotation of the viewer's y-axis
  * @param zAngle The rotation of the viewer's z-axis
  * @param zoom The distance from the viewer to the center  
  * @param width The width of the view (in pixels)
  * @param height The height of the view (in pixels)
  * @param fovy The vertical openingAngle in degrees
  */
  void translateObject(int x, int y);

  /** Rotates a selected object
  * @param x The selected x position on the display
  * @param y The selected y position on the display
  * @param xAngle The rotation of the viewer's x-axis
  * @param yAngle The rotation of the viewer's y-axis
  * @param zAngle The rotation of the viewer's z-axis
  * @param zoom The distance from the viewer to the center  
  * @param width The width of the view (in pixels)
  * @param height The height of the view (in pixels)
  * @param fovy The vertical openingAngle in degrees
  */
  void rotateObject(int x, int y);

  /** Returns the selected object
  * @return The selected object or 0 if no object is selected
  */
  SimObject* getSelectedObject() const {return selection.object;}

  /** Determines a zoom distance to fit an object or the whole scene in the current display
  * @param zoomType The type of fitting the object into the display
  * @param width The width of the display (in pixels)
  * @param height The height of the display (in pixels)
  * @param fovy The vertical openingAngle in degrees
  * @param objectName The name of an object to be fit in instead of the wohle scene
  * @return The distance
  */
  double getOptimalZoomForObject(ZoomFitType zoomType,
                                 int width, int height, double fovy,
                                 const std::string& objectName="");

  /** Sets the color of the background of the scene
  * (The color has to be a triple of (R,G,B) with values [0.0,..,1.0]
  * @param color The color
  */
  void setBackgroundColor(const Vector3d& color) {backgroundColor = color;}

  /** Sets the color of the ambient light
  * (The color has to be a triple of (R,G,B) with values [0.0,..,1.0]
  * @param color The color
  */
  void setAmbientColor(const Vector3d& color) {ambientColor = color;}

  /** Set the value of visualizeSensors
  * @param visualizeSensors The new value
  */
  void setVisualizeSensors(bool visualizeSensors); 
  
  /** Loads a file and initializes the simulation
  * @param filename The name of the file
  */
  bool loadFile(const std::string& filename, const std::string& dtdname);

  /** Closes the simulation and destroys all object*/
  void closeSimulation();

  /** Returns the current status of the simulation
  * @return The status
  */
  SimulationStatus getSimulationStatus() const {return simulationStatus;}
  
  /** Returns the file name of the simulation
  * @return The file name
  */
  std::string getSimulationFileName() const {return filename;}
  
  /** Resets the simulation*/
  void resetSimulation();

  /** Executes one simulation step*/
  void doSimulationStep();

  /** Sets the length of one simulation step (in milliseconds)
  * @param stepLength The time which is simulated by one step
  */
  void setStepLength(double stepLength);

  /** Returns the tep length
  * @return The time which is simulated by one step
  */
  double getStepLength() const {return stepLength;}

  /** Returns the current simulation step
  * @return The step
  */
  unsigned int getSimulationStep() const {return simulationStep;}
  
  /** Returns a pointer to an object (Hacking interface ;-)
  * @param objectName The name of the object
  * @return A pointer to an object
  */
  SimObject* getObjectReference(const std::string& objectName);

  /** Returns the id of an actuatorport
  * @param actuatorportName The name of the actuatorport
  * @return The id
  */
  int getActuatorportId(const std::string& actuatorportName) const;

  /** Returns the minimum value to be set for an actuatorport
  * @param id The id of the port
  * @return The minimum value
  */
  double getActuatorportMinValue(int id) const;

  /** Returns the maximum value to be set for an actuatorport
  * @param id The id of the port
  * @return The maximum value
  */
  double getActuatorportMaxValue(int id) const;

  /** Sets the value for an actuatorport
  * @param id The id of the port
  * @param value The new value
  */
  void setActuatorport(int id, double value);

  /** Sets the value for an actuatorport
  * @param id The id of the port
  * @param value The new value
  */
  void setActuatorport(int id, bool value);

  /** Returns the id for a sensorport
  * @param sensorportName The name of the sensorport
  * @return The id
  */
  int getSensorportId(const std::string& sensorName) const;

  /** Returns the dimensions of the values returned by a sensorport
  * @param id The id of the sensorport
  * @return An array. The number of values equals the number of dimension, each value is the size of the dimension
  */
  const std::vector<int>& getSensorDimensions(int id) const;

  /** Returns the current value of a sensorport
  * @param id The id of the port
  * @param value The returned value
  */
  void getSensorportValue(int id, bool& value);

  /** Returns the current value of a sensorport
  * @param id The id of the port
  * @param value The returned value
  */
  void getSensorportValue(int id, double& value);

  /** Returns the current value of a sensorport
  * @param id The id of the port
  * @param value The returned value
  */
  void getSensorportValue(int id, int& value);

  /** Returns a pointer to the current values of a sensorport
  * @param id The id of the port
  * @param value The returned value
  */
  void getSensorportValue(int id, double*& value);

  /** Returns a pointer to the current values of a sensorport
  * @param id The id of the port
  * @param value The returned value
  */
  void getSensorportValue(int id, unsigned char*& value);

  /** Returns the lowest possible value of a sensorport
  * @param id The id of the sensorport
  * @return The value
  */
  double getSensorportMinValue(int id) const;

  /** Returns the highest possible value of a sensorport
  * @param id The id of the sensorport
  * @return The value
  */
  double getSensorportMaxValue(int id) const;

  /** Returns the type of a sensorport
  * @param id The id of the sensorport
  * @return The type
  */
  SensorType getSensorportType(int id) const;
  
  /** Returns the surface for a given name.
  * @param name The name of the surface searched for.
  * @return A pointer to the surface, or 0 if it does not exist.
  */
  Surface* getSurface(const std::string& name) const;

  /** Returns a list of object descriptions
  * @param objectDesciptionTree The list to be returned
  */
  void getObjectDescriptionTree(std::vector<ObjectDescription>& objectDesciptionTree) const;
  
  /** Returns the first (the oldest) error and deletes it
  * @param The error to be returned
  */
  void getFirstError(ErrorDescription& error);

  /** Returns all current errors
  * @return errors A returned list of errors
  */
  void getAllErrors(std::vector<ErrorDescription>& errors);

  /** Deletes all current errors*/
  void deleteAllErrors();
  
  /** Adds a new view to be managed and updated
  * @param view The view
  * @param name The name of the view
  */
  void addView(View* view, const std::string& name);

  /** Returns the view with a certain sensorport id
  * @param id The sensorport id
  * @return The view
  */
  View* getView(int id);

  /** Return whether the scene graph has changed.
  * @return Has the scene graph changed sinse the last 
  *         call of doSimulationStep?
  */
  bool hasSceneGraphChanged() const {return sceneGraphChanged;}

  /** Resets that the scene graph has changed. */
  void resetSceneGraphChanged() {sceneGraphChanged = false;}
};

#endif //SIMULATION_H_

/*
 * $Log: Simulation.h,v $
 * Revision 1.1.1.1  2004/05/22 17:35:48  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.3  2004/04/20 13:14:53  roefer
 * All console commands now also work outside the start script
 *
 * Revision 1.2  2003/12/09 13:40:53  roefer
 * href attribute corrected
 *
 * Revision 1.27  2003/12/09 12:38:28  roefer
 * href attribute corrected
 *
 * Revision 1.26  2003/10/25 13:06:12  roefer
 * getSimulationFileName() added
 *
 * Revision 1.25  2003/10/21 22:32:56  roefer
 * Documented controller
 *
 * Revision 1.24  2003/10/21 12:56:32  roefer
 * onSelected added
 *
 * Revision 1.23  2003/10/20 17:11:18  roefer
 * Ambient light added
 *
 * Revision 1.22  2003/10/18 20:07:31  roefer
 * DirectView added
 *
 * Revision 1.21  2003/10/13 05:10:50  roefer
 * Progress with SimGT2004
 *
 * Revision 1.20  2003/10/12 13:19:13  tim
 * - added interactive buttons
 *
 * Revision 1.19  2003/10/12 10:24:51  roefer
 * getSurface added to Simulation
 *
 * Revision 1.18  2003/10/01 18:57:27  roefer
 * DTD is automatically selected now
 *
 * Revision 1.17  2003/09/28 14:50:04  roefer
 * Planes changed, initialValue for joints added
 *
 * Revision 1.16  2003/09/25 13:41:19  roefer
 * Drag and drop GUI, corrections for sub-object views
 *
 * Revision 1.15  2003/09/25 07:54:57  roefer
 * Drag and drop/rotate in all three axes
 *
 * Revision 1.14  2003/09/23 07:41:35  roefer
 * Drag and rotate added
 *
 * Revision 1.13  2003/09/18 05:27:37  roefer
 * simulationStep is an int now, so -1 is a possible value
 *
 * Revision 1.12  2003/09/18 01:52:09  tim
 * - changed OpenGL surface computation
 * - added stepLength
 *
 * Revision 1.11  2003/09/12 11:34:14  tim
 * - added sensor visualization framework
 * - implemented visualization for whisker
 *
 * Revision 1.10  2003/09/11 22:10:57  tim
 * - added new zoom functions
 * - added changing of opening angle
 *
 * Revision 1.9  2003/09/10 10:09:14  tim
 * - added comments
 * - small interface changes
 *
 * Revision 1.8  2003/09/08 22:32:08  tim
 * - removed files
 * - added some doxygen documentation
 * - added some const qualifiers
 * - partial code clean-up
 * - minor code changes
 * - remove __ from guards (__ should only be used by compiler)
 *
 * Revision 1.7  2003/09/08 14:46:59  tim
 * - added surface style selection
 *
 * Revision 1.6  2003/09/08 11:46:54  tim
 * - added Simulation::getSensorportMinValue
 * - added Simulation::getSensorportMaxValue
 *
 * Revision 1.5  2003/09/07 17:51:08  roefer
 * Sensor and actuator view added, interface changes for simulation
 *
 * Revision 1.4  2003/09/05 20:40:52  tim
 * - changed getObjectReference
 *
 * Revision 1.3  2003/09/05 20:20:24  tim
 * - added background color for scene
 *
 * Revision 1.2  2003/09/04 13:34:22  tim
 * - better parsing of numbers
 * - fixed macro bug
 * - better integration of macros in the object tree
 * - added getObjectReference() to Simulation
 * - faster object look-up in Simulation
 * - added changed log
 *
 */