/**
 * @file SimMath.h
 * 
 * Implementation of two basic classes: Matrix3d and Vector3d
 * Collection of functions in Functions
 *
 * @author <A href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</A>
 */ 

#ifndef SIMMATH_H_
#define SIMMATH_H_

#include <cmath>

#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif
#define EPSILON 0.0000000001

class Matrix3d;

/**
* @class Functions
* A collection of some mathematical functions
*/
class Functions
{
public:
  /** Checks if a value is between two other values
  * @param b1 The lower border
  * @param b2 The upper border
  * @param val The value to be checked
  * @return true, if b1<=v<=b2
  */
  static bool between(double b1, double b2, double val)
  { return ((b1<=val) && (val<=b2));}
};


/**
* @class Vector3d
* A class describing a vector in 3D-space
*/
class Vector3d
{
  public:
    double v[3];
  
    Vector3d();

    Vector3d(double x, double y, double z);
    Vector3d(const Vector3d& vec);

    void rotate(const Matrix3d& m);
    void rotateX(double angle);
    void rotateY(double angle);
    void rotateZ(double angle);
    void init(double angleX, double angleY, double angleZ);
    double getXRotation() const;
    double getYRotation() const;

    void normalize();
    void normalizeToAngles();
    void toRad();

    double getLength() const
      {return sqrt(v[0] * v[0] + v[1] * v[1] + v[2] * v[2]);}
    double getQuadLength() const
      {return (v[0] * v[0] + v[1] * v[1] + v[2] * v[2]);}

    void operator+=(const Vector3d& vec);
    void operator-=(const Vector3d& vec);
    void operator*=(const double& scalar);
    Vector3d operator*(const Vector3d& vec) const;
    Vector3d operator*(double n) const;
    Vector3d operator+(const Vector3d& vec) const;
    Vector3d operator-(const Vector3d& vec) const;
    bool operator==(const Vector3d& vec) const;
    bool operator!=(const Vector3d& vec) const;
    void operator=(const Vector3d& vec);

    double vecmul (const Vector3d& vec) const
      {return (v[0] * vec.v[0] + v[1] * vec.v[1] + v[2] * vec.v[2]);}
};


/**
* @class Matrix3d
* A class representing a 3x3-Matrix
*/
class Matrix3d
{
  public:
    Vector3d col[3];

    Matrix3d()
    {
      col[0].v[0] = 1; col[1].v[0] = 0; col[2].v[0] = 0;
      col[0].v[1] = 0; col[1].v[1] = 1; col[2].v[1] = 0;
      col[0].v[2] = 0; col[1].v[2] = 0; col[2].v[2] = 1;
    }

    Matrix3d(const Matrix3d& mat)
    {
      col[0] = mat.col[0];
      col[1] = mat.col[1];
      col[2] = mat.col[2];
    }

    Matrix3d(const Vector3d& rotations)
    {
      Matrix3d newMat;
      newMat.setRotation(rotations.v[0], rotations.v[1], rotations.v[2]);
      *this = newMat;
    }

    inline void setXRotation(const double angle)
    {
      double c = cos(angle);
      double s = sin(angle);
      col[0].v[0] = 1.0; col[1].v[0] = 0.0; col[2].v[0] = 0.0;
      col[0].v[1] = 0.0; col[1].v[1] = c;   col[2].v[1] = -s;
      col[0].v[2] = 0.0; col[1].v[2] = s;   col[2].v[2] = c;
    }

    inline void setYRotation(const double angle)
    {
      double c = cos(angle);
      double s = sin(angle);
      col[0].v[0] = c;   col[1].v[0] = 0.0; col[2].v[0] = s;
      col[0].v[1] = 0.0; col[1].v[1] = 1.0; col[2].v[1] = 0.0;
      col[0].v[2] = -s;   col[1].v[2] = 0.0; col[2].v[2] = c;
    }

    inline void setZRotation(const double angle)
    {
      double c = cos(angle);
      double s = sin(angle);
      col[0].v[0] = c;   col[1].v[0] = -s;  col[2].v[0] = 0.0;
      col[0].v[1] = s;   col[1].v[1] = c;   col[2].v[1] = 0.0;
      col[0].v[2] = 0.0; col[1].v[2] = 0.0; col[2].v[2] = 1.0;
    }

    inline void rotateX(double angle)
    {
      Matrix3d xMat;
      xMat.setXRotation(angle);
      xMat *= *this;
      *this = xMat;
    }

    inline void rotateY(double angle)
    {
      Matrix3d yMat;
      yMat.setYRotation(angle);
      yMat *= *this;
      *this = yMat;
    }

    inline void rotateZ(double angle)
    {
      Matrix3d zMat;
      zMat.setZRotation(angle);
      zMat *= *this;
      *this = zMat;
    }

    inline void setRotation(const double x, const double y, const double z)
    {
      Matrix3d newMatrix;
      newMatrix.rotateX(x);
      newMatrix.rotateY(y);
      newMatrix.rotateZ(z);
      *this = newMatrix;
    }

    inline void operator*=(const Matrix3d& mat)
    {
      (*this) = ((*this)*mat);
    }

    inline Matrix3d operator*(const Matrix3d& mat) const
    {
      Matrix3d m(*this);
      m.col[0].v[0] = mat.col[0].v[0] * col[0].v[0] 
                  + mat.col[0].v[1] * col[1].v[0]
                  + mat.col[0].v[2] * col[2].v[0];

      m.col[0].v[1] = mat.col[0].v[0] * col[0].v[1] 
                  + mat.col[0].v[1] * col[1].v[1]
                  + mat.col[0].v[2] * col[2].v[1];

      m.col[0].v[2] = mat.col[0].v[0] * col[0].v[2] 
                  + mat.col[0].v[1] * col[1].v[2]
                  + mat.col[0].v[2] * col[2].v[2];

      m.col[1].v[0] = mat.col[1].v[0] * col[0].v[0] 
                  + mat.col[1].v[1] * col[1].v[0]
                  + mat.col[1].v[2] * col[2].v[0];

      m.col[1].v[1] = mat.col[1].v[0] * col[0].v[1] 
                  + mat.col[1].v[1] * col[1].v[1]
                  + mat.col[1].v[2] * col[2].v[1];

      m.col[1].v[2] = mat.col[1].v[0] * col[0].v[2] 
                  + mat.col[1].v[1] * col[1].v[2]
                  + mat.col[1].v[2] * col[2].v[2];

      m.col[2].v[0] = mat.col[2].v[0] * col[0].v[0] 
                  + mat.col[2].v[1] * col[1].v[0]
                  + mat.col[2].v[2] * col[2].v[0];

      m.col[2].v[1] = mat.col[2].v[0] * col[0].v[1] 
                  + mat.col[2].v[1] * col[1].v[1]
                  + mat.col[2].v[2] * col[2].v[1];

      m.col[2].v[2] = mat.col[2].v[0] * col[0].v[2] 
                  + mat.col[2].v[1] * col[1].v[2]
                  + mat.col[2].v[2] * col[2].v[2];
      return m;
    }

    void operator=(const Matrix3d& m)
    {
      col[0] = m.col[0];
      col[1] = m.col[1];
      col[2] = m.col[2];
    }

    void invert()
    {
      Matrix3d newMatrix;
      newMatrix.col[0].v[0] = col[0].v[0];
      newMatrix.col[0].v[1] = col[1].v[0];
      newMatrix.col[0].v[2] = col[2].v[0];
      newMatrix.col[1].v[0] = col[0].v[1];
      newMatrix.col[1].v[1] = col[1].v[1];
      newMatrix.col[1].v[2] = col[2].v[1];
      newMatrix.col[2].v[0] = col[0].v[2];
      newMatrix.col[2].v[1] = col[1].v[2];
      newMatrix.col[2].v[2] = col[2].v[2];
      *this = newMatrix;
    }

    double getXAngle() const
    { 
      double h = sqrt(col[2].v[1] * col[2].v[1] + col[2].v[2] * col[2].v[2]);
      if(h)
      {
        h = col[2].v[2] / h;
        if(h > 1)
          h = 1;
        else if(h < -1)
          h = -1;
        return acos(h) * (col[2].v[1] > 0 ? -1 : 1);
      }
      else
        return 0;
    }

    double getYAngle() const
    { 
      double h = sqrt(col[0].v[0] * col[0].v[0] + col[0].v[1] * col[0].v[1]);
      if(h > 1)
        h = 1;
      return (h ? -acos(h) : -M_PI) * (col[0].v[2] < 0 ? -1 : 1);
    }

    double getZAngle() const
    {
      double h = sqrt(col[0].v[0] * col[0].v[0] + col[0].v[1] * col[0].v[1]);
      if(h)
      {
        h = col[0].v[0] / h;
        if(h > 1)
          h = 1;
        else if(h < -1)
          h = -1;
        return acos(h) * (col[0].v[1] < 0 ? -1 : 1);
      }
      else
        return 0;
    }

    void setRotationAroundAxis(const Vector3d& axis, double angle)
    {
      Matrix3d newRotation;
      Matrix3d xAxisRotation;
      Matrix3d yAxisRotation;
      double d = sqrt(axis.v[1]*axis.v[1] + axis.v[2]*axis.v[2]);
      if(d != 0)
      {
        double sina = axis.v[1] / d;
        double cosa = axis.v[2] / d;
        xAxisRotation.col[1].v[1] = cosa;
        xAxisRotation.col[1].v[2] = sina;
        xAxisRotation.col[2].v[1] = -sina;
        xAxisRotation.col[2].v[2] = cosa;
      }
      double sinb = -axis.v[0];
      double cosb = d;
      yAxisRotation.col[0].v[0] = cosb;
      yAxisRotation.col[0].v[2] = -sinb;
      yAxisRotation.col[2].v[0] = sinb;
      yAxisRotation.col[2].v[2] = cosb;

      Matrix3d invXAxisRotation = xAxisRotation;
      invXAxisRotation.invert();
      Matrix3d invYAxisRotation = yAxisRotation;
      invYAxisRotation.invert();

      newRotation = xAxisRotation * newRotation;
      newRotation = yAxisRotation * newRotation;
      newRotation.rotateZ(angle);
      newRotation = invYAxisRotation * newRotation;
      newRotation = invXAxisRotation * newRotation;
      *this = newRotation;
    }

};

#endif //SIMMATH_H_

/*
 * $Log: SimMath.h,v $
 * Revision 1.1.1.1  2004/05/22 17:35:45  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.5  2004/01/28 15:33:36  tim
 * merged corrections from SimRobXP repository
 *
 * Revision 1.11  2004/01/28 13:49:55  tim
 * - corrected y-rotation
 * - merged changes from GT2004 repository
 *
 * Revision 1.10  2004/01/11 18:29:16  tim
 * no message
 *
 * Revision 1.9  2003/12/09 12:38:28  roefer
 * href attribute corrected
 *
 * Revision 1.8  2003/10/26 14:47:03  tim
 * - fixed cylinder backtransformation problem
 * - fixed wireframe polyeder drawing problem
 *
 * Revision 1.7  2003/10/26 12:09:31  tim
 * - changed polygon rendering to vertex arrays
 * - improved polygon intersection test
 * - removed backtransformation stuff
 *
 * Revision 1.6  2003/10/18 11:25:44  tim
 * - fixed intersection tests
 * - faster intersection test
 * - reimplementation of SimGeometry
 * - added portId for sensor calls
 * - finished sensor interfaces for joint and movableObject
 *
 * Revision 1.5  2003/10/13 05:10:50  roefer
 * Progress with SimGT2004
 *
 * Revision 1.4  2003/09/10 10:09:14  tim
 * - added comments
 * - small interface changes
 *
 * Revision 1.3  2003/09/08 22:32:08  tim
 * - removed files
 * - added some doxygen documentation
 * - added some const qualifiers
 * - partial code clean-up
 * - minor code changes
 * - remove __ from guards (__ should only be used by compiler)
 *
 * Revision 1.2  2003/09/04 13:34:22  tim
 * - better parsing of numbers
 * - fixed macro bug
 * - better integration of macros in the object tree
 * - added getObjectReference() to Simulation
 * - faster object look-up in Simulation
 * - added changed log
 *
 */
