/**
 * @file SimMath.h
 * 
 * Implementation of two basic classes: Matrix3d and Vector3d
 *
 * @author <A href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</A>
 */ 

#include "SimMath.h"


Vector3d::Vector3d(double x, double y, double z)
{
  v[0] = x;
  v[1] = y;
  v[2] = z;
}

void Vector3d::init(double x, double y, double z)
{
  v[0] = x;
  v[1] = y;
  v[2] = z;
}

Vector3d::Vector3d()
{
  v[0] = 0.0;
  v[1] = 0.0;
  v[2] = 0.0;
}

Vector3d::Vector3d(const Vector3d& vec)
{
  v[0] = vec.v[0];
  v[1] = vec.v[1];
  v[2] = vec.v[2];
}

void Vector3d::rotate(const Matrix3d& m)
{
  Vector3d v_old = (*this);
  v[0] = m.col[0].v[0]*v_old.v[0] + 
         m.col[1].v[0]*v_old.v[1] + 
         m.col[2].v[0]*v_old.v[2];
  v[1] = m.col[0].v[1]*v_old.v[0] + 
         m.col[1].v[1]*v_old.v[1] + 
         m.col[2].v[1]*v_old.v[2];
  v[2] = m.col[0].v[2]*v_old.v[0] + 
         m.col[1].v[2]*v_old.v[1] + 
         m.col[2].v[2]*v_old.v[2];
}

void Vector3d::rotateX(double angle)
{
  double ca = cos(angle);
  double sa = sin(angle);
  //v[0] does not change
  double x2 = ca * v[1] - sa * v[2];
  double x3 = sa * v[1] + ca * v[2];
  v[1] = x2;
  v[2] = x3;
}

void Vector3d::rotateY(double angle)
{
  double ca = cos(angle);
  double sa = sin(angle);
  double x1 = ca * v[0] + sa * v[2];
  //v[1] does not change
  double x3 = - sa * v[0] + ca * v[2];
  v[0] = x1;
  v[2] = x3;
}

void Vector3d::rotateZ(double angle)
{
  double ca = cos(angle);
  double sa = sin(angle);
  double x1 = ca * v[0] - sa * v[1];
  double x2 = sa * v[0] + ca * v[1];
  //v[2] does not change
  v[0] = x1;
  v[1] = x2;
}
 
double Vector3d::getXRotation() const
{
  return atan2(v[1],v[2]);
}

double Vector3d::getYRotation() const
{
  return atan2(v[0],v[2]);
}

void Vector3d::normalize()
{
  double length = sqrt(v[0] * v[0] + v[1] * v[1] + v[2] * v[2]);
  if(length)
  {
    v[0] /= length;
    v[1] /= length;
    v[2] /= length;
  }
}

void Vector3d::normalizeToAngles()
{
  while(v[0] > M_PI) v[0] -= 2*M_PI;
  while(v[0] < -M_PI) v[0] += 2*M_PI;
  while(v[1] > M_PI) v[1] -= 2*M_PI;
  while(v[1] < -M_PI) v[1] += 2*M_PI;
  while(v[2] > M_PI) v[2] -= 2*M_PI;
  while(v[2] < -M_PI) v[2] += 2*M_PI;
}

void Vector3d::toRad()
{
  double factor = M_PI / 180.0;
  v[0] *= factor;
  v[1] *= factor;
  v[2] *= factor;
}

void Vector3d::operator+=(const Vector3d& vec)
{
  v[0] += vec.v[0];
  v[1] += vec.v[1];
  v[2] += vec.v[2];
}

void Vector3d::operator-=(const Vector3d& vec)
{
  v[0] -= vec.v[0];
  v[1] -= vec.v[1];
  v[2] -= vec.v[2];
}

void Vector3d::operator=(const Vector3d& vec)
{
  v[0] = vec.v[0];
  v[1] = vec.v[1];
  v[2] = vec.v[2];
}

Vector3d Vector3d::operator*(const Vector3d& vec) const
{
  Vector3d newVec;
  newVec.v[0] = v[1] * vec.v[2] - vec.v[1] * v[2];
  newVec.v[1] = vec.v[0] * v[2] - v[0] * vec.v[2];
  newVec.v[2] = v[0] * vec.v[1] - vec.v[0] * v[1];
  return newVec;
}

Vector3d Vector3d::operator*(double n) const
{
  Vector3d newVec(*this);
  newVec.v[0] *= n;
  newVec.v[1] *= n;
  newVec.v[2] *= n;
  return newVec;
}

void Vector3d::operator*=(const double& scalar)
{
  v[0] *= scalar;
  v[1] *= scalar;
  v[2] *= scalar;
}

Vector3d Vector3d::operator+(const Vector3d& vec) const
{
  Vector3d newVec(*this);
  newVec += vec;
  return newVec;
}

Vector3d Vector3d::operator-(const Vector3d& vec) const
{
  Vector3d newVec(*this);
  newVec -= vec;
  return newVec;
}

bool Vector3d::operator == (const Vector3d& vec) const
{
  return (v[0] == vec.v[0]) && (v[1] == vec.v[1]) && (v[2] == vec.v[2]);
}

bool Vector3d::operator!= (const Vector3d& vec) const
{
  return !(*this == vec);
}

/*
 * $Log: SimMath.cpp,v $
 * Revision 1.1.1.1  2004/05/22 17:35:45  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.4  2004/01/28 15:33:36  tim
 * merged corrections from SimRobXP repository
 *
 * Revision 1.9  2004/01/28 13:49:55  tim
 * - corrected y-rotation
 * - merged changes from GT2004 repository
 *
 * Revision 1.8  2003/12/09 12:38:27  roefer
 * href attribute corrected
 *
 * Revision 1.7  2003/10/26 14:47:03  tim
 * - fixed cylinder backtransformation problem
 * - fixed wireframe polyeder drawing problem
 *
 * Revision 1.6  2003/10/26 12:09:31  tim
 * - changed polygon rendering to vertex arrays
 * - improved polygon intersection test
 * - removed backtransformation stuff
 *
 * Revision 1.5  2003/10/18 11:25:44  tim
 * - fixed intersection tests
 * - faster intersection test
 * - reimplementation of SimGeometry
 * - added portId for sensor calls
 * - finished sensor interfaces for joint and movableObject
 *
 * Revision 1.4  2003/09/18 01:50:37  tim
 * - changed initialization for some variables
 *
 * Revision 1.3  2003/09/10 10:09:14  tim
 * - added comments
 * - small interface changes
 *
 * Revision 1.2  2003/09/04 13:34:22  tim
 * - better parsing of numbers
 * - fixed macro bug
 * - better integration of macros in the object tree
 * - added getObjectReference() to Simulation
 * - faster object look-up in Simulation
 * - added changed log
 *
 */