/**
 * @file SimGeometry.h
 *
 * Definition of static class SimGeometry
 * A collection of functions for geometric computations
 *
 * @author <A href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</A>
 */ 

#ifndef SIMGEOMETRY_H_
#define SIMGEOMETRY_H_

#include "SimMath.h"
#include <vector>


/**
* @class Intersection
* A class describing an intersection point
*/
class Intersection
{
public:
  /** The name of the intersected object*/
  std::string objectName;
  /** True, if the intersection has a known position*/
  bool hasPosition;
  /** The position of the intersection, if known*/
  Vector3d position;
};


/**
 * @class SimGeometry
 *
 * A collection of functions for geometric computations
 */
class SimGeometry
{
public:
  /** Tests if a point is inside a polygon
  * @param p The point
  * @param polygon The polygon
  * @param numOfPoints The number of points of the polygon
  * @param maxDistance The maximum distance of any point inside to point[0]
  * @return true, if the point is inside the polygon
  */
  static bool pointInsidePolygon(const Vector3d& p, 
                                 const std::vector<Vector3d>& polygon,
                                 const unsigned int numOfPoints,
                                 double maxDistance);
  
  /** Tests if a ray and a plane have an intersection point and
  *   returns that point.
  * @param point The starting point of the ray
  * @param v The ray
  * @param plane The position of the plane
  * @param n The normal vector of the plane
  * @param intersection The intersection point, if any exists
  * @return true, if any intersection point exists
  */
  static bool intersectRayAndPlane(const Vector3d& point, const Vector3d& v, 
                                   const Vector3d& plane, const Vector3d& n, 
                                   Vector3d& intersection);

  /** Checks if a point b is in the same plane as the points a,b and c.
  * @param p The point p
  * @param a The point a
  * @param b The point b
  * @param c The point c
  * @return true, if all four points are in one plane
  */
  static bool fourPointsInPlane(const Vector3d& p, const Vector3d& a,
                                const Vector3d& b, const Vector3d& c);

private:
  /** Intersects a ray with a line
  * @param p The starting position of the ray
  * @param v The ray
  * @param p1 The first point of the line
  * @param p2 The second point of the line
  * @return true, if there is any intersection
  */
  static bool intersectRayAndLine(const Vector3d& p, const Vector3d& v,
                                  const Vector3d& p1, const Vector3d& p2);
};


#endif //SIMGEOMETRY_H_

/*
 * $Log: SimGeometry.h,v $
 * Revision 1.1.1.1  2004/05/22 17:35:44  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.3  2003/12/09 13:40:52  roefer
 * href attribute corrected
 *
 * Revision 1.6  2003/12/09 12:38:27  roefer
 * href attribute corrected
 *
 * Revision 1.5  2003/10/26 12:09:31  tim
 * - changed polygon rendering to vertex arrays
 * - improved polygon intersection test
 * - removed backtransformation stuff
 *
 * Revision 1.4  2003/10/18 11:25:44  tim
 * - fixed intersection tests
 * - faster intersection test
 * - reimplementation of SimGeometry
 * - added portId for sensor calls
 * - finished sensor interfaces for joint and movableObject
 *
 * Revision 1.3  2003/09/08 22:32:08  tim
 * - removed files
 * - added some doxygen documentation
 * - added some const qualifiers
 * - partial code clean-up
 * - minor code changes
 * - remove __ from guards (__ should only be used by compiler)
 *
 * Revision 1.2  2003/09/04 13:34:22  tim
 * - better parsing of numbers
 * - fixed macro bug
 * - better integration of macros in the object tree
 * - added getObjectReference() to Simulation
 * - faster object look-up in Simulation
 * - added changed log
 *
 */