/**
 * @file SAX2Parser.h
 * 
 * Definition of class SAX2Parser
 *
 * @author <A href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</A>
 */ 

#ifndef SAX2PARSER_H_
#define SAX2PARSER_H_

#include "SAX2Handler.h"
#include <xercesc/framework/LocalFileInputSource.hpp>
#include "Platform/Globals.h"
#include "Platform/DebugOutput.h"
#include <string>
#include <vector>

class Sensorport;
class Actuatorport;
class Simulation;


/**
* @class XMLChTranscoder
* Utility class for converting strings in XMLCh* format
* to strings in std::string format
*/
class XMLChTranscoder
{
private:
  /** A buffer for a transcoded text*/
  std::string text;
  
public:
  /** Constructor
  * @param toTranscode The string to be transcoded
  */
  XMLChTranscoder(const XMLCh* const toTranscode)
  {
    // This is strange: if linked to the release version of xerces,
    // the string returned ist free automatically. However, the
    // debug version of xerces does not free the string at all!
    text = XMLString::transcode(toTranscode);
  }
  
  /** Returns a transcoded string
  * @return The string
  */
  std::string getValue() const
  {
    return text;
  }
};

/** 
* @class SAX2Parser
*
* A SAX2 parser for parsing the XML style *.scn files
*/
class SAX2Parser
{
private:
  class SAX2EntityResolver : public EntityResolver
  {
  private:
    InputSource* dtd;

  public:
    SAX2EntityResolver(InputSource* dtd) 
    {
      this->dtd= dtd;
    }

    virtual InputSource* resolveEntity(const XMLCh* const publicId, const XMLCh* const systemId)
    {
      if(dtd)
      {
        InputSource* is = dtd;
        dtd = 0;
        return is;
      }
      else
        return new LocalFileInputSource(systemId);
    }
  };

public:
  /** The parse function. Parses a file and builds all internal
  * data structures.
  * @param filename The name of the file to parse
  * @param dtdname The name a DTD to use for validation
  * @param sim A pointer to the simulation object
  * @param objectTree A pointer to the tree (call with empty pointer)
  * @param errorManager A pointer to the error handler
  * @param sensorportList A list of all sensorports
  * @param actuatorportList A list of all actuatorports
  * @param actuatorList A list of all actuators
  * @param surfaces A list of all object surfaces
  */
  void parse(const std::string& filename, 
             const std::string& dtdname,
             Simulation* sim,
             SimObject*& objectTree,
             ErrorManager* errorManager,
             std::vector<Sensorport*>* sensorportList,
             std::vector<Actuatorport*>* actuatorportList,
             std::vector<Actuator*>* actuatorList,
             std::vector<Surface*>* surfaces);
};


#endif //SAX2PARSER_H_

/*
 * $Log: SAX2Parser.h,v $
 * Revision 1.1.1.1  2004/05/22 17:35:43  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.3  2004/01/21 00:52:25  roefer
 * Include files and first version of simulated ERS-7
 *
 * Revision 1.2  2003/12/09 13:40:51  roefer
 * href attribute corrected
 *
 * Revision 1.6  2003/12/09 12:38:27  roefer
 * href attribute corrected
 *
 * Revision 1.5  2003/10/01 18:57:27  roefer
 * DTD is automatically selected now
 *
 * Revision 1.4  2003/09/08 22:32:08  tim
 * - removed files
 * - added some doxygen documentation
 * - added some const qualifiers
 * - partial code clean-up
 * - minor code changes
 * - remove __ from guards (__ should only be used by compiler)
 *
 * Revision 1.3  2003/09/05 20:20:24  tim
 * - added background color for scene
 *
 * Revision 1.2  2003/09/04 13:34:21  tim
 * - better parsing of numbers
 * - fixed macro bug
 * - better integration of macros in the object tree
 * - added getObjectReference() to Simulation
 * - faster object look-up in Simulation
 * - added changed log
 *
 */