/**
 * @file SAX2Parser.cpp
 * 
 * Implementation of class SAX2Parser
 *
 * @author <A href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</A>
 */ 

#include <xercesc/util/PlatformUtils.hpp>
#include <xercesc/util/TransService.hpp>
#include <xercesc/sax2/SAX2XMLReader.hpp>
#include <xercesc/sax2/XMLReaderFactory.hpp>

#include "SAX2Parser.h"
#include "SimObject.h"
#include "Errors.h"
#include "Simulation.h"

static const char* encodingName = "LATIN1";
static const char*  xmlFile = 0;
static SAX2XMLReader::ValSchemes valScheme = SAX2XMLReader::Val_Auto;
static bool	expandNamespaces = false ;
static bool doNamespaces = true;
static bool doSchema = true;
static bool schemaFullChecking = false;
static bool namespacePrefixes = false;



void SAX2Parser::parse(const std::string& filename, 
                       const std::string& dtdname,
                       Simulation* sim,
                       SimObject*& objectTree,
                       ErrorManager* errorManager,
                       std::vector<Sensorport*>* sensorportList,
                       std::vector<Actuatorport*>* actuatorportList,
                       std::vector<Actuator*>* actuatorList,
                       std::vector<Surface*>* surfaces)
{
  setlocale(LC_NUMERIC, "en_US");
  objectTree = new SimObject();
  objectTree->setName("Scene");
  objectTree->setRootNode(objectTree);
  
  // Initialize the XML4C2 system
  try
  {
    XMLPlatformUtils::Initialize();
  }
  catch (const XMLException& toCatch)
  {
    errorManager->addError("Error during initialization!",
                           XMLChTranscoder(toCatch.getMessage()).getValue());
    delete objectTree;
    objectTree = 0;
    return;
  }
  valScheme = SAX2XMLReader::Val_Always;
  expandNamespaces = true;
  schemaFullChecking = true;      
  xmlFile = filename.c_str();
  
  //  Create a SAX parser object. Then, according to what we were told on
  SAX2XMLReader* parser = XMLReaderFactory::createXMLReader();
  parser->setFeature(XMLUni::fgSAX2CoreValidation, true);
  parser->setFeature(XMLUni::fgXercesDynamic, false);
  parser->setFeature(XMLUni::fgSAX2CoreNameSpaces, doNamespaces);
  parser->setFeature(XMLUni::fgXercesSchema, doSchema);
  parser->setFeature(XMLUni::fgXercesSchemaFullChecking, true);
  parser->setFeature(XMLUni::fgSAX2CoreNameSpacePrefixes, namespacePrefixes);
  parser->setFeature(XMLUni::fgXercesStandardUriConformant, true);

  //  Create the handler object and install it as the document and error
  //  handler for the parser. Then parse the file and catch any exceptions
  //  that propogate out
  try
  {
    SAX2Handler handler(encodingName, sim, objectTree, errorManager, sensorportList, 
                        actuatorportList, actuatorList, surfaces);
    SAX2EntityResolver resolver(new LocalFileInputSource(XMLString::transcode(dtdname.c_str())));
    parser->setContentHandler(&handler);
    parser->setErrorHandler(&handler);
    parser->setEntityResolver(&resolver);
    LocalFileInputSource inputSource(XMLString::transcode(xmlFile));
    parser->parse(inputSource);
  }
  catch (const XMLException& toCatch)
  {
    errorManager->addError("Parse error!",
                           XMLChTranscoder(toCatch.getMessage()).getValue());
  }
  //  Delete the parser itself.  Must be done prior to calling Terminate, below
  delete parser;
  // And call the termination method
  XMLPlatformUtils::Terminate();

  if(errorManager->getNumberOfErrors())
  {
    delete objectTree;
    objectTree = 0;
  }
}

/*
 * $Log: SAX2Parser.cpp,v $
 * Revision 1.1.1.1  2004/05/22 17:35:43  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.3  2004/01/21 00:52:25  roefer
 * Include files and first version of simulated ERS-7
 *
 * Revision 1.2  2003/12/09 13:40:51  roefer
 * href attribute corrected
 *
 * Revision 1.9  2003/12/09 12:38:27  roefer
 * href attribute corrected
 *
 * Revision 1.8  2003/10/01 18:57:27  roefer
 * DTD is automatically selected now
 *
 * Revision 1.7  2003/09/08 22:32:08  tim
 * - removed files
 * - added some doxygen documentation
 * - added some const qualifiers
 * - partial code clean-up
 * - minor code changes
 * - remove __ from guards (__ should only be used by compiler)
 *
 * Revision 1.6  2003/09/08 11:49:14  tim
 * - removed DTD hack
 *
 * Revision 1.5  2003/09/08 11:46:54  tim
 * - added Simulation::getSensorportMinValue
 * - added Simulation::getSensorportMaxValue
 *
 * Revision 1.4  2003/09/06 21:20:22  roefer
 * Memory leaks in error case removed
 *
 * Revision 1.3  2003/09/05 20:20:24  tim
 * - added background color for scene
 *
 * Revision 1.2  2003/09/04 13:34:21  tim
 * - better parsing of numbers
 * - fixed macro bug
 * - better integration of macros in the object tree
 * - added getObjectReference() to Simulation
 * - faster object look-up in Simulation
 * - added changed log
 *
 */