/**
 * @file SAX2Handler.h
 * 
 * Definition of class SAX2Handler
 *
 * @author <A href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</A>
 */ 

#ifndef SAXHANDLER_H_
#define SAXHANDLER_H_

#include <xercesc/sax2/DefaultHandler.hpp>
#include <stack>
#include <vector>

class Polyeder;
class Sphere;
class Cylinder;
class Camera;
class Whisker;
class SimMacro;
class SimObject;
class Sensorport;
class Actuatorport;
class Actuator;
class Bumper;
class InteractiveButton;
class Joint;
class MovableObject;
class Surface;
class Vector3d;
class ErrorManager;
class Simulation;

XERCES_CPP_NAMESPACE_USE


class SAX2Handler : public DefaultHandler
{
  public:
    SAX2Handler(const char* const encodingName, Simulation* sim, 
                SimObject* objectTree,
                ErrorManager* errorManager,
                std::vector<Sensorport*>* sensorportList,
                std::vector<Actuatorport*>* actuatorportList,
                std::vector<Actuator*>* actuatorList,
                std::vector<Surface*>* surfaces);

    ~SAX2Handler();

    //  Implementations of the SAX DocumentHandler interface
    void setDocumentLocator(const Locator *const locator)
    {
       this->locator = locator;
    }

    void endDocument();

    void endElement(const XMLCh* const uri, 
                    const XMLCh* const localname, 
                    const XMLCh* const qname);

    void characters(const XMLCh* const chars, 
                    const unsigned int length);

    void processingInstruction(const XMLCh* const target,
                               const XMLCh* const data);

    void startDocument();

    void startElement(const XMLCh* const uri,
                      const XMLCh* const localname,
                      const XMLCh* const qname,
                      const Attributes& attributes);

    //  Implementations of the SAX ErrorHandler interface
    void warning(const SAXParseException& exception);
    void error(const SAXParseException& exception);
    void fatalError(const SAXParseException& exception);

    //  Implementation of the SAX DTDHandler interface
    void notationDecl(const XMLCh* const name,
                      const XMLCh* const publicId,
                      const XMLCh* const systemId);

    void unparsedEntityDecl(const XMLCh* const name,
                            const XMLCh* const publicId,
                            const XMLCh* const systemId,
                            const XMLCh* const notationName);

    //Implementation of the SAX EntityResolver interface
   /* InputSource* resolveEntity( const XMLCh *const publicId,
		                            const XMLCh *const systemId);*/

  private:
    Simulation* sim;
    std::stack<SimObject*> treeStack;
    ErrorManager* errorManager;
    SimObject* currentNode;
    SimObject* rootNode;
    std::vector<Sensorport*>* sensorportList;
    std::vector<Actuatorport*>* actuatorportList;
    std::vector<Actuator*>* actuatorList;
    std::vector<Surface*>* surfaces;
    std::vector<SimMacro*> macros;

    const Locator* locator;

    void parseSimObjectAttributes(const Attributes& attributes, SimObject* object);
    void parsePolyederAttributes(const Attributes& attributes, Polyeder* polyeder);
    void parseCameraAttributes(const Attributes& attributes, Camera* camera);
    void parseBumperAttributes(const Attributes& attributes, Bumper* bumper);
    void parseInteractiveButtonAttributes(const Attributes& attributes, InteractiveButton* interactiveButton);
    void parseWhiskerAttributes(const Attributes& attributes, Whisker* whisker);
    void parseSphereAttributes(const Attributes& attributes, Sphere* sphere);
    void parseCylinderAttributes(const Attributes& attributes, Cylinder* cylinder);
    void parseGroupAttributes(const Attributes& attributes, SimObject* group);
    void parseSceneAttributes(const Attributes& attributes, SimObject* scene);
    void parseMacroAttributes(const Attributes& attributes, SimMacro* macro);
    void parseJointAttributes(const Attributes& attributes, Joint* joint);
    void parseMovableObjectAttributes(const Attributes& attributes, MovableObject* movableObject);
    bool parseUseAttributes(const Attributes& attributes, SimObject*& object);
    void parseSurfaceAttributes(const Attributes& attributes, Surface*& surface);
    void parseDouble(const std::string& s, double& d);
    void parseInteger(const std::string& s, int& in);
    void parseBoolean(const std::string& s, bool& value);
    void parsePair(const std::string& s, double& a, double& b);
    void parsePair(const std::string& s, int& a, int& b);
    void parsePoint(const std::string& s, Vector3d& point);
    void parsePoints(const std::string& s, std::vector<Vector3d>& points);
    void parseSingleSurface(const std::string& s, std::vector<unsigned int>& surface);
    void parseSurfaces(const std::string& s, std::vector< std::vector<unsigned int> >& surfaces);

    void extractNumbersFromPair(const std::string& pair, 
      std::string& a, std::string& b);
    void extractNumbersFromTriple(const std::string& triple, 
      std::string& a, std::string& b, std::string& c);
    bool onlyWhitespaces(const std::string& s, 
                         unsigned int firstToCheck, 
                         unsigned int firstNotToCheck);
    bool checkPointsAndSurfaces(const std::vector<Vector3d>& points, 
                                const std::vector< std::vector<unsigned int> >& surfaces);
    void unsignedIntToString(unsigned int number, std::string& str);
    void getSurface(const std::string& name, Surface*& surface);
    void postprocessObjectTree(SimObject* element, const std::string& path);
};

#endif //SAX2HANDLER_H_

/*
 * $Log: SAX2Handler.h,v $
 * Revision 1.1.1.1  2004/05/22 17:35:43  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.2  2003/12/09 13:40:51  roefer
 * href attribute corrected
 *
 * Revision 1.11  2003/12/09 12:38:27  roefer
 * href attribute corrected
 *
 * Revision 1.10  2003/10/12 13:19:13  tim
 * - added interactive buttons
 *
 * Revision 1.9  2003/10/11 14:53:57  tim
 * - added standard length
 * - added invisibility for objects
 * - changed parser implementation
 *
 * Revision 1.8  2003/09/08 22:32:08  tim
 * - removed files
 * - added some doxygen documentation
 * - added some const qualifiers
 * - partial code clean-up
 * - minor code changes
 * - remove __ from guards (__ should only be used by compiler)
 *
 * Revision 1.7  2003/09/08 12:31:24  tim
 * no message
 *
 * Revision 1.6  2003/09/08 11:46:54  tim
 * - added Simulation::getSensorportMinValue
 * - added Simulation::getSensorportMaxValue
 *
 * Revision 1.5  2003/09/06 21:20:22  roefer
 * Memory leaks in error case removed
 *
 * Revision 1.4  2003/09/05 20:20:24  tim
 * - added background color for scene
 *
 * Revision 1.3  2003/09/05 19:35:38  tim
 * - error handling for points, triples, surfaces and point lists
 *
 * Revision 1.2  2003/09/04 13:34:21  tim
 * - better parsing of numbers
 * - fixed macro bug
 * - better integration of macros in the object tree
 * - added getObjectReference() to Simulation
 * - faster object look-up in Simulation
 * - added changed log
 *
 */