/**
 * @file SAX2Handler.cpp
 * 
 * Implemention of class SAX2Handler
 *
 * @author <A href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</A>
 */ 

#include <xercesc/util/XMLUniDefs.hpp>
#include <xercesc/sax2/Attributes.hpp>
#include <xercesc/sax/Locator.hpp>
#include <xercesc/framework/LocalFileInputSource.hpp>
#include <string>
#include <cstdlib>
#include <cmath>
#include <cassert>
#include <ctype.h>

#include "Platform/Globals.h"
#include "SAX2Parser.h"
#include "Polyeder.h"
#include "Camera.h"
#include "Whisker.h"
#include "Bumper.h"
#include "InteractiveButton.h"
#include "Joint.h"
#include "SimMacro.h"
#include "MovableObject.h"
#include "Sphere.h"
#include "Cylinder.h"
#include "Surface.h"
#include "Errors.h"
#include "Simulation.h"


SAX2Handler::SAX2Handler(const char* const encodingName, Simulation* sim,
                         SimObject* objectTree, 
                         ErrorManager* errorManager,
                         std::vector<Sensorport*>* sensorportList,
                         std::vector<Actuatorport*>* actuatorportList,
                         std::vector<Actuator*>* actuatorList,
                         std::vector<Surface*>* surfaces)
{
  this->sim = sim;
  this->errorManager = errorManager;
  currentNode = objectTree;
  rootNode = objectTree;
  this->sensorportList = sensorportList;
  this->actuatorportList = actuatorportList;
  this->actuatorList = actuatorList;
  this->surfaces = surfaces;
}

SAX2Handler::~SAX2Handler()
{
  for(std::vector<SimMacro*>::iterator iter = macros.begin(); iter != macros.end(); ++iter)
    delete *iter;
}

// ---------------------------------------------------------------------------
//  SAX2Handler: Overrides of the SAX ErrorHandler interface
// ---------------------------------------------------------------------------
void SAX2Handler::error(const SAXParseException& e)
{
  errorManager->addError("Error parsing file "+XMLChTranscoder(e.getSystemId()).getValue(),
                         XMLChTranscoder(e.getMessage()).getValue(),
                         e.getLineNumber(), e.getColumnNumber());
}

void SAX2Handler::fatalError(const SAXParseException& e)
{
  errorManager->addError("Fatal error parsing file "+XMLChTranscoder(e.getSystemId()).getValue(),
                         XMLChTranscoder(e.getMessage()).getValue(),
                         e.getLineNumber(), e.getColumnNumber());
}

void SAX2Handler::warning(const SAXParseException& e)
{
  errorManager->addError("Error parsing file "+XMLChTranscoder(e.getSystemId()).getValue(),
                         XMLChTranscoder(e.getMessage()).getValue(),
                         e.getLineNumber(), e.getColumnNumber());
}

// ---------------------------------------------------------------------------
//  SAX2Handler: Overrides of the SAX DTDHandler interface
// ---------------------------------------------------------------------------
void SAX2Handler::unparsedEntityDecl(const XMLCh* const name,
                                     const XMLCh* const publicId,
                                     const XMLCh* const systemId,
                                     const XMLCh* const notationName)
{}

void SAX2Handler::notationDecl(const XMLCh* const name,
                               const XMLCh* const publicId,
                               const XMLCh* const systemId)
{}

// ---------------------------------------------------------------------------
//  SAX2Handler: Overrides of the SAX EntityResolver interface
// ---------------------------------------------------------------------------
/*InputSource*  SAX2Handler::resolveEntity( const XMLCh *const publicId,
		                                      const XMLCh *const systemId)
{
  return new LocalFileInputSource(XMLString::transcode("simrobot.dtd"));
}*/

// ---------------------------------------------------------------------------
//  SAX2Handler: Overrides of the SAX DocumentHandler interface
// ---------------------------------------------------------------------------
void SAX2Handler::characters(const     XMLCh* const    chars
                                  , const   unsigned int    length)
{}


void SAX2Handler::startDocument()
{}


void SAX2Handler::startElement(const XMLCh* const uri,
                               const XMLCh* const localname,
                               const XMLCh* const qname,
                               const Attributes& attributes)
{
  treeStack.push(currentNode);
  std::string currentElement(XMLChTranscoder(qname).getValue());
  if(currentElement == "scene")
  {
    parseSceneAttributes(attributes, currentNode);
  }
  else if(currentElement == "polyeder")
  {
    Polyeder* polyeder = new Polyeder();
    parsePolyederAttributes(attributes, polyeder);
    currentNode->addChildNode(polyeder, true);
    currentNode = polyeder;
  }
  else if(currentElement == "surface")
  {
    Surface* surface = new Surface();
    parseSurfaceAttributes(attributes, surface);
    surfaces->push_back(surface);
  }
  else if(currentElement == "macro")
  {
    SimMacro* macro = new SimMacro();
    parseMacroAttributes(attributes, macro);
    macros.push_back(macro);
    currentNode = macro;
  }
  else if(currentElement == "use")
  {
    SimObject* macro = 0;
    if(parseUseAttributes(attributes, macro) && (errorManager->getNumberOfErrors() == 0))
    {
      ObjectList* childNodes = macro->getPointerToChildNodes();
      ObjectList::const_iterator child;
      for(child = childNodes->begin(); child != childNodes->end(); ++child)
      {
        (*child)->setName(macro->getName());
        currentNode->addChildNode((*child), true);
      }
      currentNode = NULL;
      childNodes->clear();
    }
    delete macro;
  }
  else if(currentElement == "camera") 
  {
    Camera* camera = new Camera();
    parseCameraAttributes(attributes, camera);
    currentNode->addChildNode(camera, true);
    currentNode = camera;
  }
  else if(currentElement == "whisker")
  {
    Whisker* whisker = new Whisker();
    parseWhiskerAttributes(attributes, whisker);
    currentNode->addChildNode(whisker, true);
    currentNode = whisker;
  }
  else if(currentElement == "bumper")
  {
    Bumper* bumper = new Bumper();
    parseBumperAttributes(attributes, bumper);
    currentNode->addChildNode(bumper, true);
    currentNode = bumper;
  }
  else if(currentElement == "interactiveButton")
  {
    InteractiveButton* interactiveButton = new InteractiveButton();
    parseInteractiveButtonAttributes(attributes, interactiveButton);
    currentNode->addChildNode(interactiveButton, true);
    currentNode = interactiveButton;
  }
  else if(currentElement == "joint")
  {
    Joint* joint = new Joint();
    parseJointAttributes(attributes, joint);
    currentNode->addChildNode(joint, true);
    currentNode = joint;
  }
  else if(currentElement == "movableObject")
  {
    MovableObject* movableObject = new MovableObject();
    parseMovableObjectAttributes(attributes, movableObject);
    currentNode->addChildNode(movableObject, true);
    currentNode = movableObject;
  }
  else if(currentElement == "sphere")
  {
    Sphere* sphere = new Sphere();
    parseSphereAttributes(attributes, sphere);
    currentNode->addChildNode(sphere, true);
    currentNode = sphere;
  }
  else if(currentElement == "group")
  {
    SimObject* group = new SimObject();
    parseGroupAttributes(attributes, group);
    if(errorManager->getNumberOfErrors() == 0)
    {
      currentNode->addChildNode(group, true);
      currentNode = group;
    }
    else
    {
      delete group;
    }
  }
  else if(currentElement == "cylinder")
  {
    Cylinder* cylinder = new Cylinder();
    parseCylinderAttributes(attributes, cylinder);
    currentNode->addChildNode(cylinder, true);
    currentNode = cylinder;
  }
  else if(currentElement == "simrobot")
  {
  }
  else
  {
    errorManager->addError("Unknown Element",
                            currentElement+" is not a valid element.",
                            locator->getLineNumber(),
                            locator->getColumnNumber());
  }
}

void SAX2Handler::endDocument()
{
  std::string path="";
  currentNode = rootNode;
  postprocessObjectTree(currentNode,path);
}

void SAX2Handler::endElement(const XMLCh* const uri,
                             const XMLCh* const localname,
                             const XMLCh* const qname)
{
  currentNode = treeStack.top();
  treeStack.pop();
}

void SAX2Handler::processingInstruction(const  XMLCh* const target,
                                        const XMLCh* const data)
{
  // Not used at this time
}

// Functions for parsing attributes *****************

void SAX2Handler::parseSceneAttributes(const Attributes& attributes, SimObject* scene)
{
  for (unsigned int index = 0; index < attributes.getLength(); index++)
  {
    std::string attributeValue = XMLChTranscoder(attributes.getValue(index)).getValue();
    std::string attributeName = XMLChTranscoder(attributes.getLocalName(index)).getValue();
    if(attributeName == "name")
    {
      scene->setName(attributeValue);
    }
    else if(attributeName == "backgroundColor")
    {
      Vector3d color;
      parsePoint(attributeValue,color);
      sim->setBackgroundColor(color);
    }
    else if(attributeName == "ambientColor")
    {
      Vector3d color;
      parsePoint(attributeValue,color);
      sim->setAmbientColor(color);
    }
    else if(attributeName == "stepLength")
    {
      double stepLength;
      parseDouble(attributeValue, stepLength);
      sim->setStepLength(stepLength);
    }
    else if(attributeName == "standardLength")
    {
      double standardLength;
      parseDouble(attributeValue, standardLength);
      scene->setStandardLength(standardLength);
    }
  }
}

void SAX2Handler::parseSimObjectAttributes(const Attributes& attributes, SimObject* object)
{
  for (unsigned int index = 0; index < attributes.getLength(); index++)
  {
    std::string attributeName(XMLChTranscoder(attributes.getLocalName(index)).getValue());
    if(attributeName=="name")
    {
      object->setName(XMLChTranscoder(attributes.getValue(index)).getValue());
    }
    else if(attributeName == "pos")
    {
      Vector3d position;
      parsePoint(XMLChTranscoder(attributes.getValue(index)).getValue(), position);
      object->setPosition(position);
    }
    else if(attributeName == "rotation")
    {

      Vector3d rotation;
      parsePoint(XMLChTranscoder(attributes.getValue(index)).getValue(), rotation);
      rotation.toRad();
      Vector3d origin = object->getPosition();
      object->rotate(rotation, origin);
    }
    else if(attributeName == "invisible")
    {
      bool invisible;
      this->parseBoolean(XMLChTranscoder(attributes.getValue(index)).getValue(),invisible);
      if(invisible)
      {
        double p;
        p=5;
      }
      object->setInvisibility(invisible);
    }
  }
}

void SAX2Handler::parseMacroAttributes(const Attributes& attributes, SimMacro* macro)
{
  parseSimObjectAttributes(attributes, macro);
}

void SAX2Handler::parseJointAttributes(const Attributes& attributes, 
                                       Joint* joint)
{
  parseSimObjectAttributes(attributes, joint);
  double value;
  for (unsigned int index = 0; index < attributes.getLength(); index++)
  {
    std::string attributeValue = XMLChTranscoder(attributes.getValue(index)).getValue();
    std::string attributeName = XMLChTranscoder(attributes.getLocalName(index)).getValue();
    if(attributeName == "maxSpeed")
    {
      parseDouble(attributeValue, value);
      joint->setMaxSpeed(value);
    }
    else if(attributeName == "minValue")
    {
      parseDouble(attributeValue, value);
      joint->setMinValue(value);
    }
    else if(attributeName == "maxValue")
    {
      parseDouble(attributeValue, value);
      joint->setMaxValue(value);
    }
    else if(attributeName == "initialValue")
    {
      parseDouble(attributeValue, value);
      joint->setInitialValue(value);
    }
    else if(attributeName == "type")
    {
      if(attributeValue == "rotation")
      {
        joint->setJointType(rotationJoint);
      }
      else if(attributeValue == "translation")
      {
        joint->setJointType(translationJoint);
      }
    }
    else if(attributeName == "axis")
    {
      if(attributeValue == "X")
      {
        joint->setAxisType(xAxis);
      }
      else if(attributeValue == "Y")
      {
        joint->setAxisType(yAxis);
      }
      else if(attributeValue == "Z")
      {
        joint->setAxisType(zAxis);
      }
    }
  }
}

void SAX2Handler::parseMovableObjectAttributes(const Attributes& attributes, 
                                               MovableObject* movableObject)
{
  parseSimObjectAttributes(attributes, movableObject);
  double value;
  for (unsigned int index = 0; index < attributes.getLength(); index++)
  {
    std::string attributeValue = XMLChTranscoder(attributes.getValue(index)).getValue();
    std::string attributeName = XMLChTranscoder(attributes.getLocalName(index)).getValue();
    if(attributeName == "xMaxSpeed")
    {
      parseDouble(attributeValue, value);
      movableObject->setSpeedXMax(value);
    }
    else if(attributeName == "yMaxSpeed")
    {
      parseDouble(attributeValue, value);
      movableObject->setSpeedYMax(value);
    }
    else if(attributeName == "zMaxSpeed")
    {
      parseDouble(attributeValue, value);
      movableObject->setSpeedZMax(value);
    }
    else if(attributeName == "xMinSpeed")
    {
      parseDouble(attributeValue, value);
      movableObject->setSpeedXMin(value);
    }
    else if(attributeName == "yMinSpeed")
    {
      parseDouble(attributeValue, value);
      movableObject->setSpeedYMin(value);
    }
    else if(attributeName == "zMinSpeed")
    {
      parseDouble(attributeValue, value);
      movableObject->setSpeedZMin(value);
    }
    else if(attributeName == "xRotMaxSpeed")
    {
      parseDouble(attributeValue, value);
      movableObject->setRotationSpeedXMax(value);
    }
    else if(attributeName == "yRotMaxSpeed")
    {
      parseDouble(attributeValue, value);
      movableObject->setRotationSpeedYMax(value);
    }
    else if(attributeName == "zRotMaxSpeed")
    {
      parseDouble(attributeValue, value);
      movableObject->setRotationSpeedZMax(value);
    }
    else if(attributeName == "xRotMinSpeed")
    {
      parseDouble(attributeValue, value);
      movableObject->setRotationSpeedXMin(value);
    }
    else if(attributeName == "yRotMinSpeed")
    {
      parseDouble(attributeValue, value);
      movableObject->setRotationSpeedYMin(value);
    }
    else if(attributeName == "zRotMinSpeed")
    {
      parseDouble(attributeValue, value);
      movableObject->setRotationSpeedZMin(value);
    }
  }
}

bool SAX2Handler::parseUseAttributes(const Attributes& attributes, SimObject*& object)
{
  bool returnValue(true);
  std::string macroName;
  unsigned int i;
  for (i = 0; i < attributes.getLength(); i++)
  {
    std::string attributeName(XMLChTranscoder(attributes.getLocalName(i)).getValue());
    if(attributeName == "name")
    {
      macroName = XMLChTranscoder(attributes.getValue(i)).getValue();
      break;
    }
  }
  for(i=0; i < macros.size(); i++)
  {
    if(macros[i]->getName() == macroName)
    {
      object = macros[i]->toSimObject();
      break;
    }
  }
  if(i == macros.size())
  {
    object = new SimObject();
    object->setName("Unknown Macro: "+macroName);
    errorManager->addError("Unknown Macro",
                            macroName+" is not a macro.",
                            locator->getLineNumber(),
                            locator->getColumnNumber());
    returnValue = false;
  }
  else
  {
    for (unsigned int index = 0; index < attributes.getLength(); index++)
    {
      std::string attributeValue(XMLChTranscoder(attributes.getValue(index)).getValue());
      std::string attributeName(XMLChTranscoder(attributes.getLocalName(index)).getValue());
      if(attributeName == "pos")
      {
        Vector3d position;
        parsePoint(attributeValue, position);
        object->setPosition(position);
      }
      if(attributeName == "rotation")
      {
        Vector3d rotation;
        parsePoint(attributeValue, rotation);
        rotation.toRad();
        Matrix3d rotationMatrix(rotation);
        Vector3d origin = object->getPosition();
        object->rotate(rotationMatrix, origin);
      }
      if(attributeName == "newName")
      {
        object->setName(attributeValue);
      }
    }
  }
  return returnValue;
}

void SAX2Handler::parsePolyederAttributes(const Attributes& attributes, Polyeder* polyeder)
{
  parseSimObjectAttributes(attributes, polyeder);
  std::vector<Vector3d> points;
  std::vector< std::vector<unsigned int> > surfaces;
  points.clear();
  surfaces.clear();
  for (unsigned int index = 0; index < attributes.getLength(); index++)
  {
    std::string attributeValue = XMLChTranscoder(attributes.getValue(index)).getValue();
    std::string attributeName = XMLChTranscoder(attributes.getLocalName(index)).getValue();
    if(attributeName == "points")
    {
      parsePoints(attributeValue, points);
    }
    else if(attributeName == "surfaces")
    {
      parseSurfaces(attributeValue, surfaces);
    }
    else if(attributeName == "surface")
    {
      Surface* surface;
      getSurface(attributeValue, surface);
      polyeder->setSurface(surface);
    }
  }
  if(checkPointsAndSurfaces(points, surfaces))
  {
    polyeder->setPointsAndSurfaces(points, surfaces);
  }
}

void SAX2Handler::parseCameraAttributes(const Attributes& attributes, Camera* camera)
{
  parseSimObjectAttributes(attributes, camera);
  for (unsigned int index = 0; index < attributes.getLength(); index++)
  {
    std::string attributeValue = XMLChTranscoder(attributes.getValue(index)).getValue();
    std::string attributeName = XMLChTranscoder(attributes.getLocalName(index)).getValue();
    if(attributeName == "resolution")
    {
      int resX;
      int resY;
      parsePair(attributeValue, resX, resY);
      camera->setResolution(resX, resY);
    }
    else if(attributeName == "angles")
    {
      double angleX, angleY;
      parsePair(attributeValue, angleX, angleY);
      camera->setOpeningAngles(angleX, angleY);
    }
    else if(attributeName == "range")
    {
      double minimum, maximum;
      parsePair(attributeValue, minimum, maximum);
      camera->setRange(minimum, maximum);
    }
  }
}

void SAX2Handler::parseBumperAttributes(const Attributes& attributes, Bumper* bumper)
{
  parseSimObjectAttributes(attributes, bumper);
}

void SAX2Handler::parseInteractiveButtonAttributes(const Attributes& attributes, InteractiveButton* interactiveButton)
{
  parseSimObjectAttributes(attributes, interactiveButton);
  for (unsigned int index = 0; index < attributes.getLength(); index++)
  {
    std::string attributeName = XMLChTranscoder(attributes.getLocalName(index)).getValue();
    if(attributeName == "snapIn")
    {
      bool snapIn;
      parseBoolean(XMLChTranscoder(attributes.getValue(index)).getValue(), snapIn);
      interactiveButton->setSnapIn(snapIn);
    }
  }
}

void SAX2Handler::parseWhiskerAttributes(const Attributes& attributes, Whisker* whisker)
{
  parseSimObjectAttributes(attributes, whisker);
  for (unsigned int index = 0; index < attributes.getLength(); index++)
  {
    std::string attributeValue = XMLChTranscoder(attributes.getValue(index)).getValue();
    std::string attributeName = XMLChTranscoder(attributes.getLocalName(index)).getValue();
    if(attributeName == "minValue")
    {
      double minValue;
      parseDouble(attributeValue, minValue);
      whisker->setMinValue(minValue);
    }
    else if(attributeName == "maxValue")
    {
      double maxValue;
      parseDouble(attributeValue, maxValue);
      whisker->setMaxValue(maxValue);
    }
  }
}

void SAX2Handler::parseSphereAttributes(const Attributes& attributes, Sphere* sphere)
{
  parseSimObjectAttributes(attributes, sphere);
  for (unsigned int index = 0; index < attributes.getLength(); index++)
  {
    std::string attributeValue = XMLChTranscoder(attributes.getValue(index)).getValue();
    std::string attributeName = XMLChTranscoder(attributes.getLocalName(index)).getValue();
    if(attributeName == "surface")
    {
      Surface* surface;
      getSurface(attributeValue, surface);
      sphere->setSurface(surface);
    }
    else if(attributeName == "radius")
    {
      double radius;
      parseDouble(attributeValue, radius);
      sphere->setRadius(radius);
    }
  }
}

void SAX2Handler::parseCylinderAttributes(const Attributes& attributes, Cylinder* cylinder)
{
  parseSimObjectAttributes(attributes, cylinder);
  for (unsigned int index = 0; index < attributes.getLength(); index++)
  {
    std::string attributeValue = XMLChTranscoder(attributes.getValue(index)).getValue();
    std::string attributeName = XMLChTranscoder(attributes.getLocalName(index)).getValue();
    if(attributeName=="surface")
    {
      Surface* surface;
      getSurface(attributeValue, surface);
      cylinder->setSurface(surface);
    }
    else if(attributeName=="radius")
    {
      double radius;
      parseDouble(attributeValue, radius);
      cylinder->setRadius(radius);
    }
    else if(attributeName=="height")
    {
      double height;
      parseDouble(attributeValue, height);
      cylinder->setHeight(height);
    }
    else if(attributeName=="closed")
    {
      bool closed;
      parseBoolean(attributeValue, closed);
      cylinder->setClosed(closed);
    }
  }
}

void SAX2Handler::parseSurfaceAttributes(const Attributes& attributes, Surface*& surface)
{
  for (unsigned int index = 0; index < attributes.getLength(); index++)
  {
    std::string attributeValue = XMLChTranscoder(attributes.getValue(index)).getValue();
    std::string attributeName = XMLChTranscoder(attributes.getLocalName(index)).getValue();
    if(attributeName == "name")
    {
      surface->name = attributeValue;
    }
    else if(attributeName == "color")
    {
      Vector3d color;
      parsePoint(attributeValue, color);
      surface->setColor(color);
    }
    else if(attributeName == "style")
    {
      if(attributeValue == "color")
      {
        surface->surfaceType = SURFACE_COLOR;
      }
      else if(attributeValue == "texture")
      {
        surface->surfaceType = SURFACE_TEXTURE;
      }
      else if(attributeValue == "dual")
      {
        surface->surfaceType = SURFACE_DUAL;
      }
    }
  }
}

void SAX2Handler::parseGroupAttributes(const Attributes& attributes, SimObject* group)
{
  parseSimObjectAttributes(attributes, group);
}

inline void SAX2Handler::parsePoint(const std::string& s, Vector3d& point)
{
  std::string xString, yString, zString;
  extractNumbersFromTriple(s, xString, yString, zString);
  parseDouble(xString, point.v[0]);
  parseDouble(yString, point.v[1]);
  parseDouble(zString, point.v[2]);
}

inline void SAX2Handler::parsePair(const std::string& s, double& a, double& b)
{
  std::string aString, bString;
  extractNumbersFromPair(s, aString, bString);
  parseDouble(aString,a);
  parseDouble(bString,b);
}

inline void SAX2Handler::parsePair(const std::string& s, int& a, int& b)
{
  std::string aString, bString;
  extractNumbersFromPair(s, aString, bString);
  parseInteger(aString,a);
  parseInteger(bString,b);
}

inline void SAX2Handler::extractNumbersFromPair(const std::string& pair, 
                                   std::string& a, std::string& b)
{
  unsigned int openingBracketPos(pair.find("("));
  unsigned int closingBracketPos(pair.find(")"));
  unsigned int kommaPos(pair.find(","));
  if((openingBracketPos != std::string::npos) && 
     (closingBracketPos != std::string::npos) &&
     (kommaPos != std::string::npos) &&
     (openingBracketPos < kommaPos) && (kommaPos < closingBracketPos))
  {
    std::string x(pair.substr(openingBracketPos+1));
    x.erase(x.find(","));
    std::string y(pair.substr(kommaPos + 1));
    y.erase(y.find(")"));
    a = x;
    b = y;
  }
  else
  {
    errorManager->addError("Malformed pair",
      pair+" is a not a valid pair. Should be '(x,y)'",
      locator->getLineNumber(),
      locator->getColumnNumber());
  }
}

inline void SAX2Handler::extractNumbersFromTriple(const std::string& triple, 
                                     std::string& a, std::string& b, std::string& c)
{
  unsigned int openingBracketPos(triple.find("("));
  unsigned int closingBracketPos(triple.find(")"));
  unsigned int firstKommaPos(triple.find(","));
  unsigned int secondKommaPos(triple.find(",",firstKommaPos+1));
  if((openingBracketPos != std::string::npos) && 
     (closingBracketPos != std::string::npos) &&
     (firstKommaPos != std::string::npos) &&
     (secondKommaPos != std::string::npos) &&
     (openingBracketPos < firstKommaPos) && 
     (firstKommaPos < secondKommaPos) &&
     (secondKommaPos < closingBracketPos) &&
     onlyWhitespaces(triple,0,openingBracketPos) &&
     onlyWhitespaces(triple,closingBracketPos+1,triple.size()))
  {
    std::string x(triple.substr(openingBracketPos+1));
    x.erase(x.find(","));
    std::string y(triple.substr(firstKommaPos+1));
    y.erase(y.find(","));
    std::string z(triple.substr(secondKommaPos+1));
    z.erase(z.find(")"));
    a = x;
    b = y;
    c = z;
  }
  else
  {
    errorManager->addError("Malformed triple",
      triple+" is a not a triple. Should be '(x,y,z)'",
      locator->getLineNumber(),
      locator->getColumnNumber());
  }
}

inline void SAX2Handler::parseDouble(const std::string& s, double& d)
{
  d = atof(s.c_str());
  //Check for possibly invalid characters
  unsigned int i(0);
  while((i<s.size()) && (s[i] == ' ')) //Read away whitespaces
  {
    i++;
  }
  if((i<s.size()) && ((s[i]=='+') || (s[i]=='-'))) //check sign
  {
    i++;
  }
  while(i<s.size() && isdigit(s[i])) //Read away numbers
  {
    i++;
  }
  if((i<s.size()) && (s[i]=='.')) //check decimal point
  {
    i++;
  }
  while(i<s.size() && isdigit(s[i])) //Read away remaining numbers
  {
    i++;
  }
  while((i<s.size()) && (s[i] == ' ')) //Read away whitespaces
  {
    i++;
  }
  if(i<s.size() || s=="")
  {
    errorManager->addError("Invalid value",
      s+" is not a valid floating point value.",
      locator->getLineNumber(),
      locator->getColumnNumber());
  }
}

inline void SAX2Handler::parseInteger(const std::string& s, int& in)
{
  in = atoi(s.c_str());
  //Check for possibly invalid characters
  unsigned int i(0);
  while((i<s.size()) && (s[i] == ' ')) //Read away whitespaces
  {
    i++;
  }
  if((i<s.size()) && ((s[i]=='+') || (s[i]=='-'))) //check sign
  {
    i++;
  }
  while(i<s.size() && isdigit(s[i])) //Read away numbers
  {
    i++;
  }
  while((i<s.size()) && (s[i] == ' ')) //Read away whitespaces
  {
    i++;
  }
  if(i<s.size() || s=="")
  {
    errorManager->addError("Invalid value",
      s+" is not a valid integer value.",
      locator->getLineNumber(),
      locator->getColumnNumber());
  }
}

inline void SAX2Handler::parseBoolean(const std::string& s, bool& value)
{
  value = (s == "true");
  if(!value && (s != "false"))
  {
    errorManager->addError("Invalid value",
                            s+" is not a boolean value. Should be 'true' or 'false'",
                            locator->getLineNumber(),
                            locator->getColumnNumber());
  }
}

void SAX2Handler::parsePoints(const std::string& s, std::vector<Vector3d>& points)
{
  std::string pointList(s);
  unsigned int openingBracketPos(pointList.find("("));
  unsigned int closingBracketPos(pointList.find(")"));
  while(openingBracketPos != std::string::npos)
  {
    if(closingBracketPos != std::string::npos)
    {
      std::string pointString(pointList);
      pointString.erase(closingBracketPos+1);
      Vector3d point;
      parsePoint(pointString, point);
      points.push_back(point);
      pointList = pointList.substr(closingBracketPos+1);
      openingBracketPos = pointList.find("(");
      closingBracketPos = pointList.find(")");
    }
    else
    {
      errorManager->addError("')' missing",
      "')' missing",
      locator->getLineNumber(),
      locator->getColumnNumber());
      break;
    }
  }
  if(points.size()<3)
  {
    errorManager->addError("Not enough points",
      "This polygon has not enough points! At least three points are needed.",
      locator->getLineNumber(),
      locator->getColumnNumber());
  }
}

void SAX2Handler::parseSingleSurface(const std::string& s, 
                                     std::vector<unsigned int>& surface)
{
  int ptNumber;
  std::string surfaceString(s);
  surfaceString = surfaceString.substr(surfaceString.find("(") + 1);
  surfaceString.erase(surfaceString.find(")"));
  while(surfaceString.find(",") != std::string::npos)
  {
    std::string pt = surfaceString;
    pt.erase(pt.find(","));
    parseInteger(pt,ptNumber);
    surface.push_back(ptNumber);
    surfaceString = surfaceString.substr(surfaceString.find(",") + 1);
  }
  parseInteger(surfaceString,ptNumber);
  surface.push_back(ptNumber);
}

void SAX2Handler::parseSurfaces(const std::string& s, 
                                std::vector< std::vector<unsigned int> >& surfaces)
{
  std::string surfaceList(s);
  unsigned int parsePos(0);
  unsigned int openingBracketPos(surfaceList.find("("));
  while(openingBracketPos != std::string::npos)
  {
    if(onlyWhitespaces(surfaceList,parsePos,openingBracketPos))
    {
      std::string pointString = surfaceList;
      unsigned int closingBracketPos(pointString.find(")"));
      if(closingBracketPos != std::string::npos)
      {
        pointString.erase(closingBracketPos+1);
        pointString = pointString.substr(openingBracketPos);
        std::vector<unsigned int> surface;
        parseSingleSurface(pointString, surface);
        surfaces.push_back(surface);
        surfaceList = surfaceList.substr(surfaceList.find(")")+1);
        openingBracketPos = surfaceList.find("(");
      }
      else
      {
        errorManager->addError("')' missing",
          "')' missing at end of surface definition",
          locator->getLineNumber(),
          locator->getColumnNumber());
        break;
      }
    }
    else
    {
      errorManager->addError("Bad characters",
          "Forbidden characters in surface definition: "+surfaceList,
          locator->getLineNumber(),
          locator->getColumnNumber());
      break;
    }
  }
  if((surfaceList.size() > 0) && !(onlyWhitespaces(surfaceList,0,surfaceList.size())))
  {
    errorManager->addError("Bad characters",
          "Forbidden characters at the end of surfaces definition: "+surfaceList,
          locator->getLineNumber(),
          locator->getColumnNumber());
  }
}

void SAX2Handler::postprocessObjectTree(SimObject* element, const std::string& path)
{
  std::string name = element->getName();
  std::string fullName = path + name;
  element->setFullName(fullName);
  element->addToLists(*sensorportList, *actuatorportList, *actuatorList);
  element->computeIntersectionSphereRadius();
  for(unsigned int i=0; i<element->getNumberOfChildNodes(); i++)
  {
    postprocessObjectTree(element->getChildNode(i), fullName + ".");
  }
}

void SAX2Handler::getSurface(const std::string& name, Surface*& surface)
{
  bool surfaceFound(false);
  std::vector<Surface*>::const_iterator iter;
  for(iter = surfaces->begin(); iter != surfaces->end(); ++iter)
  {
    if((*iter)->name == name)
    {
      surface = (*iter);
      surfaceFound = true;
      break;
    }
  }
  if(!surfaceFound)
  {
    errorManager->addError("Unknown Surface",
                            name+" is not a valid surface element.",
                            locator->getLineNumber(),
                            locator->getColumnNumber());
  }
}

bool SAX2Handler::onlyWhitespaces(const std::string& s, 
                                  unsigned int firstToCheck, 
                                  unsigned int firstNotToCheck)
{
  for(unsigned int i=firstToCheck; i<firstNotToCheck; i++)
  {
    if(s[i] != ' ')
    {
      return false;
    }
  }
  return true;
}

bool SAX2Handler::checkPointsAndSurfaces(const std::vector<Vector3d>& points, 
                              const std::vector< std::vector<unsigned int> >& surfaces)
{
  unsigned int numberOfPoints(points.size());
  for(unsigned int i=0; i<surfaces.size(); i++)
  {
    for(unsigned int j=0; j<surfaces[i].size(); j++)
    {
      if(surfaces[i][j] >= numberOfPoints)
      {
        std::string pointString,numberString;
        unsignedIntToString(surfaces[i][j], pointString);
        unsignedIntToString(numberOfPoints, numberString);
        errorManager->addError("Unknown Point",
                            "The point number "+pointString+
                            ", adressed by a surface, does not exist. The polygon has only"
                            +numberString+" points",
                            locator->getLineNumber(),
                            locator->getColumnNumber());
        return false;
      }
    }
  }
  return true;
}

void SAX2Handler::unsignedIntToString(unsigned int number, std::string& str)
{
  char buf[70];
  sprintf(&buf[0],"%u",number);
  str = &buf[0];
}

/*
 * $Log: SAX2Handler.cpp,v $
 * Revision 1.1.1.1  2004/05/22 17:35:43  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.2  2003/12/09 13:40:51  roefer
 * href attribute corrected
 *
 * Revision 1.22  2003/12/09 12:38:26  roefer
 * href attribute corrected
 *
 * Revision 1.21  2003/10/20 17:11:18  roefer
 * Ambient light added
 *
 * Revision 1.20  2003/10/12 13:19:13  tim
 * - added interactive buttons
 *
 * Revision 1.19  2003/10/11 14:53:57  tim
 * - added standard length
 * - added invisibility for objects
 * - changed parser implementation
 *
 * Revision 1.18  2003/10/05 15:24:30  tim
 * - changed drag & drop visualization
 *
 * Revision 1.17  2003/10/03 21:07:50  roefer
 * camera range added
 *
 * Revision 1.16  2003/09/28 14:50:04  roefer
 * Planes changed, initialValue for joints added
 *
 * Revision 1.15  2003/09/18 11:05:13  tim
 * - fixed another parser crashing bug
 *
 * Revision 1.14  2003/09/18 10:42:01  tim
 * - fixed parser crash
 *
 * Revision 1.13  2003/09/18 01:51:14  tim
 * - added stepLength
 *
 * Revision 1.12  2003/09/17 11:20:05  tim
 * - changed newname attribute of <use> tag to newName
 *
 * Revision 1.11  2003/09/12 06:49:29  roefer
 * axis-bug fixed
 *
 * Revision 1.10  2003/09/08 22:32:08  tim
 * - removed files
 * - added some doxygen documentation
 * - added some const qualifiers
 * - partial code clean-up
 * - minor code changes
 * - remove __ from guards (__ should only be used by compiler)
 *
 * Revision 1.9  2003/09/08 12:31:25  tim
 * no message
 *
 * Revision 1.8  2003/09/08 11:49:14  tim
 * - removed DTD hack
 *
 * Revision 1.7  2003/09/08 11:46:54  tim
 * - added Simulation::getSensorportMinValue
 * - added Simulation::getSensorportMaxValue
 *
 * Revision 1.6  2003/09/06 21:20:22  roefer
 * Memory leaks in error case removed
 *
 * Revision 1.5  2003/09/06 20:53:34  roefer
 * Memory leak removed, macros are freed now
 *
 * Revision 1.4  2003/09/05 20:20:24  tim
 * - added background color for scene
 *
 * Revision 1.3  2003/09/05 19:35:38  tim
 * - error handling for points, triples, surfaces and point lists
 *
 * Revision 1.2  2003/09/04 13:34:21  tim
 * - better parsing of numbers
 * - fixed macro bug
 * - better integration of macros in the object tree
 * - added getObjectReference() to Simulation
 * - faster object look-up in Simulation
 * - added changed log
 *
 */
