/**
 * @file Polyeder.h
 * 
 * Definition of class Polyeder
 *
 * @author <A href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</A>
 */ 

#ifndef POLYEDER_H_
#define POLYEDER_H_

#include "DrawableObject.h"

/**
* @class Polyeder
*
* A class representing a polyeder in 3D space.
*/
class Polyeder : public DrawableObject
{
protected:
  /** The points of the polyeder*/
  double* vertices;
  /** The polygons*/
  std::vector<unsigned int*> polygonIDX;
  /** The number of points per polygon*/
  std::vector<unsigned int> numOfPointsPerPolygon;
  /** The number of vertices*/
  unsigned int numOfVertices;
  /** The number of points*/
  unsigned int numOfPoints;
  /** The number of polygons*/
  unsigned int numOfPolygons;
  /** The maximum number of points per polygon*/
  unsigned int maxNumOfPointsPerPolygon;
  /** The normals assigned to the polygons*/
  std::vector<Vector3d> polygonNormals;
  /** Intersection spheres for all surfaces*/
  std::vector< double > intersectionSpheres;
  /** Flag: All polygon normals are valid, if true*/
  bool normalsValid;
  /** A buffer for intersection tests*/
  std::vector<Vector3d> intersectionBuffer;

  /** Intersects the polyeder with a ray
  * @param pos The starting position of the ray
  * @param ray The ray
  * @param intersectPos The intersection point nearest to pos (if found)
  * @return true, if an intersection point has been found
  */
  virtual bool intersectWithRay(const Vector3d& pos, const Vector3d& ray, Vector3d& intersectPos);	  

public:
  /** Constructor*/
  Polyeder();

  /** Destructor*/
  ~Polyeder();

  /** Returns a string describing the kind of object
  * @return The kind
  */
  virtual std::string getKind() const {return "polyeder";}

  /** Draws the polyeder and its child nodes
  * @param pointOfView The point of view from which the scene is drawn
  * @param visParams Parameters for Visualization
  */
  virtual void draw(const Vector3d& pointOfView,
                    const VisualizationParameterSet& visParams);

  /** Sets the points and surfaces and builds internal representation
  * @param points The points
  * @param surfaces The surfaces
  */
  void setPointsAndSurfaces(const std::vector<Vector3d>& points, 
                            const std::vector< std::vector<unsigned int> >& surfaces);

  /** Sets the new position of the object (children will be translated, too)
  * @param pos The new position
  */
  virtual void setPosition(const Vector3d& pos);

  /** Translates the object and its children in 3D space
  * @param translation A vector describing the translation
  */
  virtual void translate(const Vector3d& translation);

  /** Rotates the object and its children in 3D space
  * @param rotation The rotation in matrix form
  * @param origin The point to rotate around
  */
  virtual void rotate(const Matrix3d& rotation, 
                      const Vector3d& origin);

  /** Clones the object and its child nodes
  * @return A pointer to a copy of the object
  */
  virtual SimObject* clone() const;

  /** Returns the maximum distance of any point in the subtree to a given point
  * @param base The point to compute the distance to
  * @return The distance
  */
  virtual double getMaxDistanceTo(Vector3d& base) const;

private:
  /** Updates all normal vectors*/
  void updateNormals();

  /** Returns a vector from the vertex list
  * @param pos The position of the first vector element
  * @param vec The vector
  */
  void getVectorFromVertices(unsigned int pos,
                             Vector3d& vec) const;

  /** Writes a vector to the vertex list
  * @param pos The position in the list
  * @param vec The vector
  */
  void setVerticesFromVector(unsigned int pos,
                             const Vector3d& vec);

  /** Copies vertex data to the intersection buffer
  * @param polygon The index of the polygon to be copied
  */
  void copyPolygonToIntersectionBuffer(unsigned int polygon);
};

#endif //POLYEDER_H_

/*
 * $Log: Polyeder.h,v $
 * Revision 1.1.1.1  2004/05/22 17:35:42  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.3  2003/12/09 13:40:51  roefer
 * href attribute corrected
 *
 * Revision 1.9  2003/12/09 12:38:26  roefer
 * href attribute corrected
 *
 * Revision 1.8  2003/10/26 12:09:31  tim
 * - changed polygon rendering to vertex arrays
 * - improved polygon intersection test
 * - removed backtransformation stuff
 *
 * Revision 1.7  2003/10/23 21:30:31  tim
 * - normal computation only after rotations
 * - first steps to faster geometry rendering
 *
 * Revision 1.6  2003/10/18 11:25:44  tim
 * - fixed intersection tests
 * - faster intersection test
 * - reimplementation of SimGeometry
 * - added portId for sensor calls
 * - finished sensor interfaces for joint and movableObject
 *
 * Revision 1.5  2003/09/18 01:52:07  tim
 * - changed OpenGL surface computation
 * - added stepLength
 *
 * Revision 1.4  2003/09/12 11:34:14  tim
 * - added sensor visualization framework
 * - implemented visualization for whisker
 *
 * Revision 1.3  2003/09/08 22:32:08  tim
 * - removed files
 * - added some doxygen documentation
 * - added some const qualifiers
 * - partial code clean-up
 * - minor code changes
 * - remove __ from guards (__ should only be used by compiler)
 *
 * Revision 1.2  2003/09/04 13:34:21  tim
 * - better parsing of numbers
 * - fixed macro bug
 * - better integration of macros in the object tree
 * - added getObjectReference() to Simulation
 * - faster object look-up in Simulation
 * - added changed log
 *
 */