/**
 * @file Joint.h
 * 
 * Definition of class Joint
 *
 * @author <A href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</A>
 */ 

#ifndef JOINT_H_
#define JOINT_H_

#include "Actuator.h"
#include "Sensor.h"
#include "SimObject.h"


/** The two types of joints*/
enum JointType
{
  rotationJoint,
  translationJoint
};

/** The three axis types*/
enum AxisType
{
  xAxis,
  yAxis,
  zAxis
};


/**
* @class Bumper
*
* A class representing a joint.
*/
class Joint : public Actuator, public Sensor
{
protected:
  /** The type of the joint*/
  JointType jointType;
  /** The axis of the joint*/
  AxisType axisType;
  /** The maximum value*/
  double maxValue;
  /** The minimum value*/
  double minValue;
  /** The speed of the joint*/
  double maxSpeed;
  /** The current value of the joint*/
  double currentValue;
  /** The current destination value*/
  double destValue;

public:
  /** Constructor*/
  Joint();

  /** Returns a string describing the kind of object
  * @return The kind
  */
  virtual std::string getKind() const {return "joint";}
  
  /** Adds the object to some internal lists
  * @param sensorportList A list of all sensor ports in the scene
  * @param actuatorportList A list of all actuator ports in the scene
  * @param actuatorList A list of all actuators in the scene
  */
  virtual void addToLists(std::vector<Sensorport*>& sensorportList,
                          std::vector<Actuatorport*>& actuatorportList,
                          std::vector<Actuator*>& actuatorList);

  /** Adds descriptions of the object and its children to a list
  * @param objectDescriptionTree The list of descriptions
  * @param The depth of the object in the object tree
  */
  virtual void addToDescriptions(std::vector<ObjectDescription>& objectDescriptionTree,
                                 int depth);

  /** Sets the value of one actuatorport
  * @param value The value
  * @param port The port
  */
  virtual void setValue(double value, int port);

  /** Computes the position of the joint
  * @param value The value
  * @param portId The id of the sensorport
  */
  virtual void computeValue(double& value, int portId);

  /** Let the actuator do what it is supposed to do
   * @param initial Is this the initial step?
   */
  virtual void act(bool initial);

  /** Clones the object and its child nodes
  * @return A pointer to a copy of the object
  */
  virtual SimObject* clone() const;
  
  /** Sets the maximum speed
  * @param value The maximum speed
  */
  void setMaxSpeed(double value);

  /** Sets the maximum value
  * @param value The maximum value
  */
  void setMaxValue(double value);

  /** Sets the minimum value
  * @param value The minimum value
  */
  void setMinValue(double value);

  /** Sets the initial value
  * @param value The initial value
  */
  void setInitialValue(double value);

  /** Sets the joint type
  * @param jointType The joint type
  */
  void setJointType(JointType jointType);

  /** Sets the axis type
  * @param axisType The axis type
  */
  void setAxisType(AxisType axisType);

  /** Returns the maximum value of a sensor
  * @param portId The id of the sensorport
  * @return The value
  */
  virtual double getMaxValue(int portId) const {return maxValue;}

  /** Returns the minimum value of a sensor
  * @param portId The id of the sensorport
  * @return The value
  */
  virtual double getMinValue(int portId) const {return minValue;}
};

#endif

/*
 * $Log: Joint.h,v $
 * Revision 1.1.1.1  2004/05/22 17:35:38  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.2  2003/12/09 13:40:50  roefer
 * href attribute corrected
 *
 * Revision 1.7  2003/12/09 12:38:26  roefer
 * href attribute corrected
 *
 * Revision 1.6  2003/10/18 11:25:44  tim
 * - fixed intersection tests
 * - faster intersection test
 * - reimplementation of SimGeometry
 * - added portId for sensor calls
 * - finished sensor interfaces for joint and movableObject
 *
 * Revision 1.5  2003/10/13 05:10:50  roefer
 * Progress with SimGT2004
 *
 * Revision 1.4  2003/09/28 14:50:04  roefer
 * Planes changed, initialValue for joints added
 *
 * Revision 1.3  2003/09/08 22:32:08  tim
 * - removed files
 * - added some doxygen documentation
 * - added some const qualifiers
 * - partial code clean-up
 * - minor code changes
 * - remove __ from guards (__ should only be used by compiler)
 *
 * Revision 1.2  2003/09/04 13:34:21  tim
 * - better parsing of numbers
 * - fixed macro bug
 * - better integration of macros in the object tree
 * - added getObjectReference() to Simulation
 * - faster object look-up in Simulation
 * - added changed log
 *
 */