/**
 * @file Joint.cpp
 * 
 * Implementation of class Joint
 *
 * @author <A href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</A>
 */ 

#include <cfloat>
#include "Joint.h"
#include "Sensorport.h"
#include "Actuatorport.h"

Joint::Joint() 
{
  minValue = 0.0;
  maxValue = 0.0;
  maxSpeed = DBL_MAX;
  jointType = rotationJoint;
  axisType = xAxis;
  currentValue = 0.0;
  destValue = 0.0;
}

void Joint::addToLists(std::vector<Sensorport*>& sensorportList,
                       std::vector<Actuatorport*>& actuatorportList,
                       std::vector<Actuator*>& actuatorList) 
{ 
  Actuatorport* value = new Actuatorport("value", this, 0, minValue, maxValue);
  actuatorportList.push_back(value);
  Sensorport* valueSensor = new Sensorport
        ("valueSensor", 0, doubleSensor, this, minValue, maxValue);
  sensorportList.push_back(valueSensor);
  actuatorList.push_back(this);
}

void Joint::addToDescriptions(std::vector<ObjectDescription>& objectDescriptionTree,
                              int depth) 
{
  Actuator::addToDescriptions(objectDescriptionTree, depth);  

  //Add Actuatorports:
  ObjectDescription valueDesc;
  valueDesc.name = "value";
  valueDesc.fullName = fullName + ".value";
  valueDesc.depth = depth + 1;
  valueDesc.type = OBJECT_TYPE_ACTUATORPORT;
  objectDescriptionTree.push_back(valueDesc);

  //Add Sensorports:
  valueDesc.name = "valueSensor";
  valueDesc.fullName = fullName + ".valueSensor";
  valueDesc.depth = depth + 1;
  valueDesc.type = OBJECT_TYPE_SENSORPORT;
  objectDescriptionTree.push_back(valueDesc);
}

void Joint::setValue(double value, int port)
{
  if(jointType == rotationJoint)
  {
    value *= M_PI / 180.0;
  }
  destValue = value;
}

void Joint::act(bool initial)
{
  if(destValue != currentValue)
  {
    if(jointType == rotationJoint)
    {
      double angle = destValue - currentValue;
      currentValue = destValue;

      //TODO:
      //Testen, ob der angestrebte Winkel in einem Simulationsschritt erreicht werden kann
      //Ansonsten einen Zwischenwinkel bestimmen

      Vector3d axis(0.0,0.0,0.0);
      switch(axisType)
      {
        case xAxis: axis.v[0] = 1.0; break;
        case yAxis: axis.v[1] = 1.0; break;
        case zAxis: axis.v[2] = 1.0; break;
      }
      Matrix3d relativeRotation;
      axis.rotate(rotation);
      axis.normalize();
      relativeRotation.setRotationAroundAxis(axis, angle);
      std::list<SimObject*>::iterator pos;
      for(pos = childNodes.begin(); pos != childNodes.end(); ++pos)
      {
        (*pos)->rotate(relativeRotation, position);
      }
    }
    else //rotationType == translationJoint
    {
      Vector3d trans;
      switch(axisType)
      {
        case xAxis: trans.v[0] = 1.0; break;
        case yAxis: trans.v[1] = 1.0; break;
        case zAxis: trans.v[2] = 1.0; break;
      }
      double translation = destValue - currentValue;
      currentValue = destValue;
      trans *= translation;
      trans.rotate(rotation);

      std::list<SimObject*>::iterator pos;
      for(pos = childNodes.begin(); pos != childNodes.end(); ++pos)
      {
        (*pos)->translate(trans);
      }
    }
  }
}

void Joint::computeValue(double& value, int portId)
{
  value = currentValue * 180 / M_PI;
}

SimObject* Joint::clone() const
{
  Joint* newJoint = new Joint();
  newJoint->setName(name);
  newJoint->setFullName(fullName);
  newJoint->setPosition(position);
  newJoint->rotation = rotation;
  newJoint->jointType = jointType;
  newJoint->axisType = axisType;
  newJoint->maxValue = maxValue;
  newJoint->minValue = minValue;
  newJoint->destValue = destValue;
  newJoint->maxSpeed = maxSpeed;
  newJoint->childNodes.clear();
  std::list<SimObject*>::const_iterator pos;
  for(pos = childNodes.begin(); pos != childNodes.end(); ++pos)
  {
    SimObject* childNode = (*pos)->clone();
    newJoint->addChildNode(childNode, false);
  }
  SimObject* newObject = newJoint;
  return newObject;
}

void Joint::setMaxSpeed(double value)
{
  if(jointType == rotationJoint)
  {
    value *= M_PI / 180.0;
  }
  maxSpeed = value;
}

void Joint::setMaxValue(double value)
{
  /*if(jointType == rotationJoint)
  {
    value *= M_PI / 180.0;
  }*/
  maxValue = value;
}

void Joint::setMinValue(double value)
{
  /*if(jointType == rotationJoint)
  {
    value *= M_PI / 180.0;
  }*/
  minValue = value;
}

void Joint::setInitialValue(double value)
{
  if(jointType == rotationJoint)
  {
    value *= M_PI / 180.0;
  }
  destValue = value;
}

void Joint::setJointType(JointType jointType)
{
  if(this->jointType != jointType)
  {
    this->jointType = jointType;
    if(jointType == rotationJoint)
    {
      maxValue *= M_PI / 180.0;
      minValue *= M_PI / 180.0;
      maxSpeed *= M_PI / 180.0;
    }
    /*else
    {
      maxValue /= M_PI / 180.0;
      minValue /= M_PI / 180.0;
      maxSpeed /= M_PI / 180.0;
    }*/
  }
}

void Joint::setAxisType(AxisType axisType)
{
  this->axisType = axisType;
}

/*
 * $Log: Joint.cpp,v $
 * Revision 1.1.1.1  2004/05/22 17:35:38  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.3  2003/12/09 13:40:50  roefer
 * href attribute corrected
 *
 * Revision 1.7  2003/12/09 12:38:26  roefer
 * href attribute corrected
 *
 * Revision 1.6  2003/10/26 12:09:31  tim
 * - changed polygon rendering to vertex arrays
 * - improved polygon intersection test
 * - removed backtransformation stuff
 *
 * Revision 1.5  2003/10/18 11:25:44  tim
 * - fixed intersection tests
 * - faster intersection test
 * - reimplementation of SimGeometry
 * - added portId for sensor calls
 * - finished sensor interfaces for joint and movableObject
 *
 * Revision 1.4  2003/09/28 14:50:04  roefer
 * Planes changed, initialValue for joints added
 *
 * Revision 1.3  2003/09/08 22:32:08  tim
 * - removed files
 * - added some doxygen documentation
 * - added some const qualifiers
 * - partial code clean-up
 * - minor code changes
 * - remove __ from guards (__ should only be used by compiler)
 *
 * Revision 1.2  2003/09/04 13:34:21  tim
 * - better parsing of numbers
 * - fixed macro bug
 * - better integration of macros in the object tree
 * - added getObjectReference() to Simulation
 * - faster object look-up in Simulation
 * - added changed log
 *
 */