/**
 * @file InteractiveButton.h
 * 
 * Definition of class InteractiveButton
 *
 * @author <A href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</A>
 */ 

#ifndef INTERACTIVE_BUTTON_H_
#define INTERACTIVE_BUTTON_H_

#include "Sensor.h"


/**
* @class InteractiveButton
*
* A class representing a button which can be
* clicked by the user in any scene view.
*/
class InteractiveButton : public Sensor
{
protected:
  /** The current value of the button (true/false)*/
  bool value;
  /** Flag: Button keeps activated after click, if true*/
  bool snapIn;

public:
  /** Constructor */
  InteractiveButton();

  /** Returns a string describing the kind of object
  * @return The kind
  */
  virtual std::string getKind() const {return "interactiveButton";}
  
  /** Returns the current value of the button
  * @param value The value
  * @param portId The id of the sensorport
  */
  virtual void computeValue(bool& value, int portId);
  
  /** Adds the object to some internal lists
  * @param sensorportList A list of all sensor ports in the scene
  * @param actuatorportList A list of all actuator ports in the scene
  * @param actuatorList A list of all actuators in the scene
  */
  virtual void addToLists(std::vector<Sensorport*>& sensorportList,
    std::vector<Actuatorport*>& actuatorportList,
    std::vector<Actuator*>& actuatorList);
  
  /** Adds descriptions of the object and its children to a list
  * @param objectDescriptionTree The list of descriptions
  * @param The depth of the object in the object tree
  */
  virtual void addToDescriptions(std::vector<ObjectDescription>& objectDescriptionTree,
    int depth);

  /** Clones the object and its child nodes
  * @return A pointer to a copy of the object
  */
  virtual SimObject* clone() const;
  
  /** Returns true, since this object is clickable
  * @param nameOfObject The name of this object will be copied to this variable
  * @param testClickable Flag: Also test for interactive buttons, if true
  * @return Always true
  */
  virtual bool isMovableOrClickable(std::string& nameOfObject,
                                    bool testClickable) const;

  /** Sets the value of the snapIn flag
  * @param snapIn The new value of snapIn 
  */
  void setSnapIn(bool snapIn) {this->snapIn = snapIn;}

  /** Informs the button about being clicked*/
  void hasBeenClicked();

  /** Informs the button about being released*/
  void hasBeenReleased();
};

#endif //INTERACTIVE_BUTTON_H_

/*
 * $Log: InteractiveButton.h,v $
 * Revision 1.1.1.1  2004/05/22 17:35:38  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.2  2003/12/09 13:40:50  roefer
 * href attribute corrected
 *
 * Revision 1.5  2003/12/09 12:38:26  roefer
 * href attribute corrected
 *
 * Revision 1.4  2003/10/18 20:04:00  roefer
 * Switch is bool sensor
 *
 * Revision 1.3  2003/10/18 11:25:44  tim
 * - fixed intersection tests
 * - faster intersection test
 * - reimplementation of SimGeometry
 * - added portId for sensor calls
 * - finished sensor interfaces for joint and movableObject
 *
 * Revision 1.2  2003/10/17 22:16:25  roefer
 * New files in GT2004 added
 *
 * Revision 1.1  2003/10/12 13:21:32  tim
 * - added interactive buttons
 *
 */