/**
 * @file Errors.h
 * 
 * Definition of class ErrorManager
 *
 * @author <A href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</A>
 */ 

#ifndef ERRORS_H_
#define ERRORS_H_

#include "APIDatatypes.h"
#include "Platform/Globals.h"
#include <deque>
#include <vector>

/**
* @class ErrorManager
*
* A class for storing errors which occur at runtime
*/
class ErrorManager
{
private:
  /** The error queue*/
  std::deque<ErrorDescription> errors;

public:
  /** Destructor*/
  ~ErrorManager() {errors.clear();}

  /** Returns the number of stored errors
  * @return The number of errors
  */
  unsigned int getNumberOfErrors() const {return errors.size();}

  /** Returns the first (oldest) error
  * @param error The error to be returned
  */
  void getFirstError(ErrorDescription& error) const {error = errors.front();}

  /** Returns a list of all errors
  * @param errors The errors to be returned
  */
  void getAllErrors(std::vector<ErrorDescription>& errors);

  /** Deletes the first (oldest) error*/
  void deleteFirstError() {errors.pop_front();}

  /** Deletes all errors*/
  void deleteAllErrors() {errors.clear();}

  /** Adds a new error at the end of the queue
  * @param error The error
  */
  void addError(const ErrorDescription& error) {errors.push_back(error);}

  /** Adds a new error at the end of the queue
  * (line and column will be set to 0, error.position to false)
  * @param title The title of the error
  * @param text Detailed description
  */
  void addError(const std::string& title, const std::string& text);

  /** Adds a new error at the end of the queue
  * (error.position will be set to true)
  * @param title The title of the error
  * @param text Detailed description
  * @param line The line in the XML file in which the error occured
  * @param column The column in the XML file in which the error occured
  */
  void addError(const std::string& title, const std::string& text,
                int line, int column);
};

#endif //ERRORS_H_

/*
 * $Log: Errors.h,v $
 * Revision 1.1.1.1  2004/05/22 17:35:38  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.2  2003/12/09 13:40:50  roefer
 * href attribute corrected
 *
 * Revision 1.5  2003/12/09 12:38:26  roefer
 * href attribute corrected
 *
 * Revision 1.4  2003/09/10 10:09:14  tim
 * - added comments
 * - small interface changes
 *
 * Revision 1.3  2003/09/08 22:32:07  tim
 * - removed files
 * - added some doxygen documentation
 * - added some const qualifiers
 * - partial code clean-up
 * - minor code changes
 * - remove __ from guards (__ should only be used by compiler)
 *
 * Revision 1.2  2003/09/04 13:34:21  tim
 * - better parsing of numbers
 * - fixed macro bug
 * - better integration of macros in the object tree
 * - added getObjectReference() to Simulation
 * - faster object look-up in Simulation
 * - added changed log
 *
 */