/**
 * @file Camera.h
 * 
 * Definition of class Camera
 *
 * @author <A href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</A>
 */ 

#ifndef CAMERA_H_
#define CAMERA_H_

#include "Sensor.h"

class OffscreenRenderer;


/**
* @class Camera
*
* A class representing a camera.
*/
class Camera : public Sensor
{
protected:
  /** The image width*/
  int resolutionX;
  /** The image height*/
  int resolutionY;
  /** The horizontal opening angle*/
  double angleX;
  /** The vertical opening angle*/
  double angleY;
  /** The minimum range*/
  double minRange;
  /** The maximum range*/
  double maxRange;
  /** A pointer to the image*/
  unsigned char* image;
  /** The size of the image*/
  unsigned int imageSize;
  /** Parameters for visualization*/
  VisualizationParameterSet visParams;
  /** A pointer to an offscreen renderer*/
  OffscreenRenderer* osRenderer;
  
public:
  /** Constructor*/
  Camera();
  /** Destructor */
  ~Camera();

  /** Returns a string describing the kind of object
  * @return The kind
  */
  virtual std::string getKind() const {return "camera";}
  
  /** Sets the resolution and initializes the image
  * @param x The new image width
  * @param y The new image height
  */
  void setResolution(int x, int y);

  /** Sets the opening angles of the camera in degrees
  * @param x The horizontal opening angle
  * @param y The vertical opening angle
  */
  void setOpeningAngles(double x, double y) {angleX = x; angleY = y;}

  /** Sets the distance range of the camera
  * @param minRange The minimum distance recognized
  * @param maxRange The maximum distance recognized
  */
  void setRange(double minimum, double maximum) {minRange = minimum; maxRange = maximum;}

  /** Adds the object to some internal lists
  * @param sensorportList A list of all sensor ports in the scene
  * @param actuatorportList A list of all actuator ports in the scene
  * @param actuatorList A list of all actuators in the scene
  */
  virtual void addToLists(std::vector<Sensorport*>& sensorportList,
    std::vector<Actuatorport*>& actuatorportList,
    std::vector<Actuator*>& actuatorList);

  /** Adds descriptions of the object and its children to a list
  * @param objectDescriptionTree The list of descriptions
  * @param The depth of the object in the object tree
  */
  virtual void addToDescriptions(std::vector<ObjectDescription>& objectDescriptionTree,
    int depth);

  /** Clones the object and its child nodes
  * @return A pointer to a copy of the object
  */
  virtual SimObject* clone() const;

  /** Computes the image
  * @param value A pointer to be set to the image
  * @param portId The id of the sensorport
  */
  void computeValue(unsigned char*& value, int portId);

  /** Executes the rendering of the image, called by offscreen renderer*/
  void renderingInstructions() const;

  /** Draws the camera opening and all child nodes
  * @param pointOfView The point of view from which the scene is drawn
  * @param visParams Parameters for Visualization
  */
  virtual void draw(const Vector3d& pointOfView,
                    const VisualizationParameterSet& visParams);
};

#endif //CAMERA_H_

/*
 * $Log: Camera.h,v $
 * Revision 1.1.1.1  2004/05/22 17:35:37  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.2  2003/12/09 13:40:49  roefer
 * href attribute corrected
 *
 * Revision 1.9  2003/12/09 12:38:26  roefer
 * href attribute corrected
 *
 * Revision 1.8  2003/10/18 11:25:44  tim
 * - fixed intersection tests
 * - faster intersection test
 * - reimplementation of SimGeometry
 * - added portId for sensor calls
 * - finished sensor interfaces for joint and movableObject
 *
 * Revision 1.7  2003/10/05 16:57:02  tim
 * - added simple visualization for camera opening
 *
 * Revision 1.6  2003/10/03 21:07:50  roefer
 * camera range added
 *
 * Revision 1.5  2003/09/18 01:52:06  tim
 * - changed OpenGL surface computation
 * - added stepLength
 *
 * Revision 1.4  2003/09/08 22:32:07  tim
 * - removed files
 * - added some doxygen documentation
 * - added some const qualifiers
 * - partial code clean-up
 * - minor code changes
 * - remove __ from guards (__ should only be used by compiler)
 *
 * Revision 1.3  2003/09/08 11:46:54  tim
 * - added Simulation::getSensorportMinValue
 * - added Simulation::getSensorportMaxValue
 *
 * Revision 1.2  2003/09/04 13:34:21  tim
 * - better parsing of numbers
 * - fixed macro bug
 * - better integration of macros in the object tree
 * - added getObjectReference() to Simulation
 * - faster object look-up in Simulation
 * - added changed log
 *
 */
