
#include "Platform/OpenGL.h"
#include "Camera.h"
#include "Platform/OffScreenRenderer.h"
#include "Sensorport.h"



Camera::Camera()
{
  image = 0;
  resolutionX = 0;
  resolutionY = 0;
  minRange = 0.01;
  maxRange = 100;
  osRenderer = new OffscreenRenderer();
  visParams.surfaceStyle = VisualizationParameterSet::SMOOTH_SHADING;
}

Camera::~Camera()
{
  delete osRenderer;
  delete image;
}

void Camera::addToLists(std::vector<Sensorport*>& sensorportList,
                        std::vector<Actuatorport*>& actuatorportList,
                        std::vector<Actuator*>& actuatorList) 
{
  std::vector<int> dimensions;
  dimensions.clear();
  dimensions.push_back(resolutionX);
  dimensions.push_back(resolutionY);
  dimensions.push_back(3);
  Sensorport* cameraPort = new Sensorport
     ("image", 0, cameraSensor, this, 0.0, 255.0, dimensions);
  sensorportList.push_back(cameraPort);
}

void Camera::addToDescriptions(std::vector<ObjectDescription>& objectDescriptionTree,
                               int depth) 
{
  Sensor::addToDescriptions(objectDescriptionTree, depth);  

  //Add Sensorport:
  ObjectDescription sensorportDesc;
  sensorportDesc.name = "image";
  sensorportDesc.fullName = fullName + ".image";
  sensorportDesc.depth = depth + 1;
  sensorportDesc.type = OBJECT_TYPE_SENSORPORT;
  objectDescriptionTree.push_back(sensorportDesc);
}

void Camera::setResolution(int x, int y)
{
  if(image != 0)
  {
    delete image;
  }
  resolutionX = x;
  resolutionY = y;
  imageSize = x * y * 3;
  image = new unsigned char[imageSize];
}

void Camera::renderingInstructions() const
{
  glViewport(0, 0, resolutionX,resolutionY);
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective(angleY, tan(angleX * M_PI / 360) / tan(angleY * M_PI / 360), minRange, maxRange);
  
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  glTranslatef(0.0,0.0,0.0);
  glClearColor((float) backgroundColor.v[0],
               (float) backgroundColor.v[1],
               (float) backgroundColor.v[2],0);
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);			
  
  glShadeModel(GL_SMOOTH);
  glEnable(GL_POLYGON_SMOOTH); 
  glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);
  glEnable(GL_LIGHTING);
  GLfloat LightAmbient[]= 
  { 
    float(ambientColor.v[0]),
    float(ambientColor.v[1]),
    float(ambientColor.v[2]),
    1.0f 
  };  // Ambient Light Values ( NEW )
  GLfloat LightDiffuse[]= 
  { 
    float(1 - ambientColor.v[0]),
    float(1 - ambientColor.v[1]),
    float(1 - ambientColor.v[2]),
    1.0f 
  }; // Diffuse Light Values ( NEW )
  GLfloat LightPosition[]= { 0.0f, 0.0f, 10.0f, 1.0f }; // Light Position ( NEW )
  glLightfv(GL_LIGHT1, GL_AMBIENT, LightAmbient); // Setup The Ambient Light
  glLightfv(GL_LIGHT1, GL_DIFFUSE, LightDiffuse); // Setup The Diffuse Light
  glLightfv(GL_LIGHT1, GL_POSITION,LightPosition); // Position The Light
  glEnable(GL_LIGHT1); // Enable Light One
  glEnable(GL_DEPTH_TEST);
  glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
  glEnable(GL_COLOR_MATERIAL);
  glLightModeli(GL_LIGHT_MODEL_TWO_SIDE,1);
  
  Vector3d forwardVec(1.0,0.0,0.0);
  Vector3d upVec(0.0,0.0,1.0);
  forwardVec.rotate(rotation);
  upVec.rotate(rotation);
  gluLookAt(position.v[0], position.v[1], position.v[2], 
            position.v[0] + forwardVec.v[0], position.v[1] + forwardVec.v[1], 
            position.v[2] + forwardVec.v[2],
            upVec.v[0], upVec.v[1], upVec.v[2]);
  rootNode->setDrawForCamera(true);
  rootNode->draw(position,visParams);
  rootNode->setDrawForCamera(false);
  glFlush();
}

void Camera::computeValue(unsigned char*& value, int portId)
{	
  if(simulationStep>lastComputationStep)
  {
    osRenderer->prepareRendering(image, resolutionX, resolutionY);
    renderingInstructions();
    osRenderer->finishRendering(image, resolutionX, resolutionY);
    lastComputationStep = simulationStep;
  }
  value = image;
}

SimObject* Camera::clone() const
{
  Camera* newCamera= new Camera();
  newCamera->setName(name);
  newCamera->setFullName(fullName);
  newCamera->setPosition(position);
  newCamera->rotation=rotation;
  newCamera->resolutionX = resolutionX;
  newCamera->resolutionY = resolutionY;
  newCamera->imageSize = imageSize;
  newCamera->angleX = angleX;
  newCamera->angleY = angleY;
  newCamera->minRange = minRange;
  newCamera->maxRange = maxRange;
  newCamera->childNodes.clear();
  newCamera->image = new unsigned char[resolutionX * resolutionY * 3];
  memcpy(newCamera->image, image, sizeof(unsigned char) * resolutionX * resolutionY * 3);
  std::list<SimObject*>::const_iterator pos;
  for(pos = childNodes.begin(); pos != childNodes.end(); ++pos)
  {
    SimObject* childNode = (*pos)->clone();
    newCamera->addChildNode(childNode, false);
  }
  SimObject* newObject = newCamera;
  return newObject;
}

void Camera::draw(const Vector3d& pointOfView,
                  const VisualizationParameterSet& visParams)
{
  if(visualizeSensors && !drawForCamera)
  {
    double dist(standardLength);
    double yRot((angleY/2.0) * (M_PI/180.0));
    double zRot((angleX/2.0) * (M_PI/180.0));
    Vector3d camPos = position;
    //Compute four points: Upper left, Upper right, Lower left and lower right
    Vector3d ptUL(dist,0.0,0.0);
    ptUL.rotateY(yRot);
    Vector3d ptUR(ptUL);
    ptUL.rotateZ(zRot);
    ptUR.rotateZ(-zRot);

    Vector3d ptLL(dist,0.0,0.0);
    ptLL.rotateY(-yRot);
    Vector3d ptLR(ptLL);
    ptLL.rotateZ(zRot);
    ptLR.rotateZ(-zRot);
    //Transform points in scene
    ptUL.rotate(rotation);
    ptUL+=camPos;
    ptUR.rotate(rotation);
    ptUR+=camPos;
    ptLL.rotate(rotation);
    ptLL+=camPos;
    ptLR.rotate(rotation);
    ptLR+=camPos;
    //Draw camera opening
    glColor3f(0.0,0.0,1.0);
    glBegin(GL_LINES);
      glVertex3d (camPos.v[0], camPos.v[1], camPos.v[2]);
      glVertex3d (ptUL.v[0], ptUL.v[1], ptUL.v[2]);
    glEnd();
    glBegin(GL_LINES);
      glVertex3d (camPos.v[0], camPos.v[1], camPos.v[2]);
      glVertex3d (ptUR.v[0], ptUR.v[1], ptUR.v[2]);
    glEnd();
    glBegin(GL_LINES);
      glVertex3d (camPos.v[0], camPos.v[1], camPos.v[2]);
      glVertex3d (ptLL.v[0], ptLL.v[1], ptLL.v[2]);
    glEnd();
    glBegin(GL_LINES);
      glVertex3d (camPos.v[0], camPos.v[1], camPos.v[2]);
      glVertex3d (ptLR.v[0], ptLR.v[1], ptLR.v[2]);
    glEnd();
    glBegin(GL_LINE_LOOP);
      glVertex3d (ptUL.v[0], ptUL.v[1], ptUL.v[2]);
      glVertex3d (ptUR.v[0], ptUR.v[1], ptUR.v[2]);
      glVertex3d (ptLR.v[0], ptLR.v[1], ptLR.v[2]);
      glVertex3d (ptLL.v[0], ptLL.v[1], ptLL.v[2]);
    glEnd();
  }
  SimObject::draw(pointOfView, visParams);
}


/*
 * $Log: Camera.cpp,v $
 * Revision 1.1.1.1  2004/05/22 17:35:37  cvsadm
 * created new repository GT2004_WM
 *
 * Revision 1.3  2003/12/09 13:40:49  roefer
 * href attribute corrected
 *
 * Revision 1.21  2003/12/03 18:12:37  roefer
 * Compatibility with VC2003.NET, GUI does still not work completely there
 *
 * Revision 1.20  2003/10/20 17:11:18  roefer
 * Ambient light added
 *
 * Revision 1.19  2003/10/18 11:25:44  tim
 * - fixed intersection tests
 * - faster intersection test
 * - reimplementation of SimGeometry
 * - added portId for sensor calls
 * - finished sensor interfaces for joint and movableObject
 *
 * Revision 1.18  2003/10/11 14:53:57  tim
 * - added standard length
 * - added invisibility for objects
 * - changed parser implementation
 *
 * Revision 1.17  2003/10/09 23:13:37  roefer
 * Bug in offscreen renderng fixed
 *
 * Revision 1.16  2003/10/07 08:16:37  roefer
 * angle ratio replace by aspect ratio
 *
 * Revision 1.15  2003/10/07 08:13:21  roefer
 * angle ratio replace by aspect ratio
 *
 * Revision 1.14  2003/10/05 16:57:02  tim
 * - added simple visualization for camera opening
 *
 * Revision 1.13  2003/10/05 15:24:30  tim
 * - changed drag & drop visualization
 *
 * Revision 1.12  2003/10/03 21:07:50  roefer
 * camera range added
 *
 * Revision 1.11  2003/09/28 14:50:04  roefer
 * Planes changed, initialValue for joints added
 *
 * Revision 1.10  2003/09/18 01:52:06  tim
 * - changed OpenGL surface computation
 * - added stepLength
 *
 * Revision 1.9  2003/09/17 02:39:26  roefer
 * GL_FRONT_AND_BACK used everywhere
 *
 * Revision 1.8  2003/09/12 11:34:14  tim
 * - added sensor visualization framework
 * - implemented visualization for whisker
 *
 * Revision 1.7  2003/09/11 22:54:05  tim
 * - background color now in camera images, too
 *
 * Revision 1.6  2003/09/11 17:26:00  tim
 * - fixed perspective/resolution bug
 *
 * Revision 1.5  2003/09/08 22:32:07  tim
 * - removed files
 * - added some doxygen documentation
 * - added some const qualifiers
 * - partial code clean-up
 * - minor code changes
 * - remove __ from guards (__ should only be used by compiler)
 *
 * Revision 1.4  2003/09/08 15:45:54  tim
 * - adapted rendering to changes in Simulation::draw
 *
 * Revision 1.3  2003/09/07 17:50:07  roefer
 * Camera returns pointer to image
 *
 * Revision 1.2  2003/09/04 13:34:21  tim
 * - better parsing of numbers
 * - fixed macro bug
 * - better integration of macros in the object tree
 * - added getObjectReference() to Simulation
 * - faster object look-up in Simulation
 * - added changed log
 *
 */