// srSensor.cpp : implementation file
//

#include "stdafx.h"
#include <float.h>
#include <winnls.h>
#include "SimRobXP.h"
#include "srDoc.h"
#include "srSensor.h"
#define _NOLINK
#include "Controller/Controller.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CSensor

IMPLEMENT_DYNCREATE(CSensor, CView)

CSensor::CSensor()
{
}

CSensor::~CSensor()
{
  if(m_pBitmap)
    delete m_pBitmap;
  if(m_pBuffer)
    delete [] m_pBuffer;
  if(m_nBufSize)
    delete [] m_pData;
}

void CSensor::WriteLayout()
{
  CDoc* pDoc = GetDocument();
  WINDOWPLACEMENT wp;
  wp.length = sizeof(wp);
  GetParent()->GetWindowPlacement(&wp);
  pDoc->NewSection("sensorFrame");
  pDoc->WriteInt("flags",wp.flags);
  pDoc->WriteInt("show",wp.showCmd);
  pDoc->WriteInt("top",wp.rcNormalPosition.top);
  pDoc->WriteInt("bottom",wp.rcNormalPosition.bottom);
  pDoc->WriteInt("left",wp.rcNormalPosition.left);
  pDoc->WriteInt("right",wp.rcNormalPosition.right);
  pDoc->WriteInt("minX",wp.ptMinPosition.x);
  pDoc->WriteInt("minY",wp.ptMinPosition.y);
  pDoc->WriteInt("maxX",wp.ptMaxPosition.x);
  pDoc->WriteInt("maxY",wp.ptMaxPosition.y);

  pDoc->NewSection("sensor");
  pDoc->WriteString("sensor",m_sSensor);
  pDoc->WriteInt("view",m_nView);
  pDoc->WriteInt("brightening",m_nBright);
  pDoc->WriteInt("grid",m_bGrid);
}

bool CSensor::RestoreLayout()
{
  CDoc* pDoc = GetDocument();
  if(!pDoc->IsRestoringLayout())
    return false;
  WINDOWPLACEMENT wp;
  wp.length = sizeof(wp);
  pDoc->NewSection("sensorFrame");
  wp.flags = pDoc->ReadInt("flags");
  wp.showCmd = pDoc->ReadInt("show");
  wp.rcNormalPosition.top = pDoc->ReadInt("top");
  wp.rcNormalPosition.bottom = pDoc->ReadInt("bottom");
  wp.rcNormalPosition.left = pDoc->ReadInt("left");
  wp.rcNormalPosition.right = pDoc->ReadInt("right");
  wp.ptMinPosition.x = pDoc->ReadInt("minX");
  wp.ptMinPosition.y = pDoc->ReadInt("minY");
  wp.ptMaxPosition.x = pDoc->ReadInt("maxX");
  wp.ptMaxPosition.y = pDoc->ReadInt("maxY");
  GetParent()->SetWindowPlacement(&wp);

  pDoc->NewSection("sensor");
  m_sSensor = pDoc->ReadString("sensor");
  m_nView = pDoc->ReadInt("view");
  m_nBright = pDoc->ReadInt("brightening");
  m_bGrid = pDoc->ReadInt("grid") != 0;
  return true;
}

BEGIN_MESSAGE_MAP(CSensor, CView)
  //{{AFX_MSG_MAP(CSensor)
  ON_WM_RBUTTONUP()
  ON_COMMAND(ID_VIEW_LINE, OnViewLine)
  ON_UPDATE_COMMAND_UI(ID_VIEW_LINE, OnUpdateViewLine)
  ON_COMMAND(ID_VIEW_COLUMN, OnViewColumn)
  ON_UPDATE_COMMAND_UI(ID_VIEW_COLUMN, OnUpdateViewColumn)
  ON_COMMAND(ID_VIEW_MONO, OnViewMono)
  ON_UPDATE_COMMAND_UI(ID_VIEW_MONO, OnUpdateViewMono)
  ON_COMMAND(ID_VIEW_COLOR, OnViewColor)
  ON_UPDATE_COMMAND_UI(ID_VIEW_COLOR, OnUpdateViewColor)
  ON_COMMAND(ID_VIEW_STEREO, OnViewStereo)
  ON_UPDATE_COMMAND_UI(ID_VIEW_STEREO, OnUpdateViewStereo)
  ON_COMMAND(ID_VIEW_GRID, OnViewGrid)
  ON_UPDATE_COMMAND_UI(ID_VIEW_GRID, OnUpdateViewGrid)
  ON_COMMAND(ID_BRIGHT0, OnBright)
  ON_UPDATE_COMMAND_UI(ID_BRIGHT0, OnUpdateBright)
  ON_WM_ERASEBKGND()
  ON_COMMAND(ID_EDIT_COPY, OnEditCopy)
  ON_COMMAND(ID_BRIGHT1, OnBright)
  ON_COMMAND(ID_BRIGHT2, OnBright)
  ON_COMMAND(ID_BRIGHT3, OnBright)
  ON_COMMAND(ID_BRIGHT4, OnBright)
  ON_COMMAND(ID_BRIGHT5, OnBright)
  ON_UPDATE_COMMAND_UI(ID_BRIGHT1, OnUpdateBright)
  ON_UPDATE_COMMAND_UI(ID_BRIGHT2, OnUpdateBright)
  ON_UPDATE_COMMAND_UI(ID_BRIGHT3, OnUpdateBright)
  ON_UPDATE_COMMAND_UI(ID_BRIGHT4, OnUpdateBright)
  ON_UPDATE_COMMAND_UI(ID_BRIGHT5, OnUpdateBright)
  ON_UPDATE_COMMAND_UI(ID_EDIT_COPY, OnUpdateEditCopy)
  //}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CSensor drawing

void CSensor::OnDraw(CDC* pDC)
{
  CRect rect;
  GetClientRect(&rect);
  if(!GetParent()->IsIconic() && rect.right && rect.bottom)
  {
    if(m_sp == -1)
    {
      CBrush br(GetSysColor(COLOR_WINDOW));
      pDC->FillRect(&rect,&br);
    }
    else
    {
      Simulation* pSim = GetDocument()->GetSimulation();
      switch(pSim->getSensorportType(m_sp))
      {
        case boolSensor:
        {
          bool boolValue;
          pSim->getSensorportValue(m_sp,boolValue);
          m_data = boolValue ? 255 : 0;
          m_pData = &m_data;
          break;
        }
        case intSensor:
        {
          bool intValue;
          pSim->getSensorportValue(m_sp,intValue);
          m_data = ToChar(intValue);
          m_pData = &m_data;
          break;
        }
        case doubleSensor:
          double doubleValue;
          pSim->getSensorportValue(m_sp,doubleValue);
          m_data = ToChar(doubleValue);
          m_pData = &m_data;
          break;
        case cameraSensor:
        {
          pSim->getSensorportValue(m_sp,m_pData);
          break;
        }
        case doubleFieldSensor:
        {
          double* pDouble;
          pSim->getSensorportValue(m_sp,pDouble);
          for(int i = 0; i < m_nBufSize; ++i)
            m_pData[i] = ToChar(pDouble[i]);
        }
      }

      if(m_nView <= 1 || m_nView == 5)
        DrawMemDC(*pDC,rect);
      else
        DrawBitmap(*pDC,rect);
    }
  }
}

void CSensor::DrawBitmap(CDC& dc,CRect& rect)
{
  if(!m_pBuffer || (m_nView == 4 && rect != m_rect))
  {
    m_rect = rect;
    if(m_pBuffer)
      delete [] m_pBuffer;
    if(m_nView == 2)
    {
      m_xSize = m_nDimSize[0];
      m_ySize = m_nDimSize[1];
    }
    else if(m_nView == 3)
    {
      m_xSize = m_nDim > 1 ? m_nDimSize[0] : 1,
      m_ySize = m_nDim > 2 ? m_nDimSize[1] : 1;
    }
    else
    {
      m_xSize = rect.right;
      m_ySize = rect.bottom;
    }
    m_pBuffer = (BITMAPINFOHEADER*) new BYTE[sizeof(BITMAPINFOHEADER) + 
                                            ((m_xSize * 3 + 3) & 0xfffffffc) * m_ySize];
    m_pBuffer->biSize          = sizeof(BITMAPINFOHEADER);
    m_pBuffer->biWidth         = m_xSize;
    m_pBuffer->biHeight        = m_ySize;
    m_pBuffer->biPlanes        = 1;
    m_pBuffer->biBitCount      = 24;
    m_pBuffer->biCompression   = BI_RGB;
    m_pBuffer->biSizeImage     = 0;
    m_pBuffer->biXPelsPerMeter = 0;
    m_pBuffer->biYPelsPerMeter = 0;
    m_pBuffer->biClrUsed       = 0;
    m_pBuffer->biClrImportant  = 0;
    m_bRepaint = true;
  }
  if(m_bRepaint)
  {
    m_bRepaint = false;
    switch(m_nView)
    {
    case 2:
      DrawMono();
      break;
    case 3:
      DrawColor();
      break;
    case 4:
      DrawStereo();
    }
  }
  if(m_nView == 4)
    SetDIBitsToDevice(dc.m_hDC,0,0,rect.right,rect.bottom,0,0,0,rect.bottom,
                      m_pBuffer+1,(BITMAPINFO*) m_pBuffer,DIB_RGB_COLORS);
  else
  {
    dc.SetStretchBltMode(HALFTONE);
    dc.SetBrushOrg(0,0);
    CPalette pal;
    pal.CreateHalftonePalette(&dc);
    CPalette* pPal = dc.SelectPalette(&pal,false);
    dc.RealizePalette();
    StretchDIBits(dc.m_hDC,0,0,rect.right,rect.bottom,0,0,m_xSize,m_ySize,
                  m_pBuffer+1,(BITMAPINFO*) m_pBuffer,DIB_RGB_COLORS,SRCCOPY);
    dc.SelectPalette(pPal,false);
  }
}

void CSensor::DrawMono()
{
  double dBright = 1.0 / (m_nBright * 0.5 + 1.0);
  unsigned char scale[256];
  for(int i = 0; i < 256; ++i)
    scale[i] = (unsigned char) (pow(i / 255.0,dBright) * 255);

  unsigned char* pSrc = m_pData,
               * pDest = (BYTE*) (m_pBuffer+1) + m_ySize * ((m_xSize * 3 + 3) & 0xfffffffc);
  for(int y = 0; y < m_ySize; ++y)
  {
    pDest -= (m_xSize * 3 + 3) & 0xfffffffc;
    unsigned char* pDest2 = pDest;
    for(int x = 0; x < m_xSize; ++x)
    {
      unsigned char value = scale[*pSrc++];
      *pDest2++ = value;
      *pDest2++ = value;
      *pDest2++ = value;
    }
  }
}

void CSensor::DrawColor()
{
  double dBright = 1.0 / (m_nBright * 0.5 + 1.0);
  unsigned char scale[256];
  for(int i = 0; i < 256; ++i)
    scale[i] = (unsigned char) (pow(i / 255.0,dBright) * 255);

  int zMax = m_nDimSize[m_nDim-1];

  unsigned char* pSrc = m_pData,
               * pDest = (BYTE*) (m_pBuffer+1) + m_ySize * ((m_xSize * 3 + 3) & 0xfffffffc);
  for(int y = 0; y < m_ySize; ++y)
  {
    pDest -= (m_xSize * 3 + 3) & 0xfffffffc;
    unsigned char* pDest2 = pDest;
    for(int x = 0; x < m_xSize; ++x)
    {
      for(int z = 0; z < zMax; ++z)
        *pDest2++ = scale[*pSrc++];
      while(z < 3)
        *pDest2++ = 0;
    }
  }
}

void CSensor::Constrains(unsigned char* pDepth,int* pConstrain,
                         bool* pVisible,double xDepthStep)
{
  double  E = 1.0/250,
          z,
          zMax = DBL_MIN;
  int nPos = m_xSize;

  for(int x = m_xSize-1; x >= 0; x--)
  {
    pConstrain[x] = x;
    z = pDepth[(int) (x * xDepthStep)];
    if(z < 0)
    {
      z = 0;
      pDepth[(int) (x * xDepthStep)] = 0;
    }
    else if(z > 1)
    {
      z = 1;
      pDepth[(int) (x * xDepthStep)] = 255;
    }
    if(z + (nPos-x) * E * (12 - 2 * z) > zMax)
    {
      nPos = x;
      zMax = z;
      pVisible[x] = true;
    }
    else
      pVisible[x] = false;
  }

  zMax = DBL_MIN;
  nPos = -1;
  for(x = 0; x < m_xSize; x++)
  {
    if(pVisible[x])
    {
      z = pDepth[(int) (x * xDepthStep)];
      if(z + (x-nPos) * E * (12 - 2 * z) > zMax)
      {
        nPos = x;
        zMax = z;
        int s = (int) ((3 - z) / (6 - z) / E + 0.5),
            left = x - (s >> 1),
            right = left + s;
        if(0 <= left && right < m_xSize)
        {
          int l = pConstrain[left];
          while(l != left && l != right)
          {
            if(l < right)
            {
              left = l;
              l = pConstrain[left];
            }
            else
            {
              pConstrain[left] = right;
              left = right;
              right = l;
              l = pConstrain[left];
            }
          }
          pConstrain[left] = right;
        }
      }
    }
  }
}

void CSensor::DrawStereo()
{
  int* pConstrain = new int[m_xSize];
  bool* pVisible = new bool[m_xSize];
  double xDepthStep = (double) m_nDimSize[0] / m_xSize,
         yDepthStep = (double) m_nDimSize[1] / m_ySize;
  int yd = -1,
      ydo;
  srand(0);
  int zMax = m_nDimSize[m_nDim-1];

  unsigned char* pSrc = m_pData,
               * pDest = (BYTE*) (m_pBuffer+1) + m_ySize * ((m_xSize * 3 + 3) & 0xfffffffc);
  for(int y = 0; y < m_ySize; ++y)
  {
    pDest -= (m_xSize * 3 + 3) & 0xfffffffc;
    unsigned char* pDest2 = pDest;

    ydo = yd;
    yd = (int) (y * yDepthStep);
    if(yd != ydo)
      Constrains(m_pData + yd * m_nDimSize[0],pConstrain,
                 pVisible,xDepthStep);
    for(int x = m_xSize-1; x >= 0; x--)
    {
      bool bPixel = rand() & 1;
      if(pConstrain[x] == x)
        pVisible[x] = bPixel;
      else
        pVisible[x] = pVisible[pConstrain[x]];
    }
    for(x = 0; x < m_xSize; x++)
    {
      unsigned char value = pVisible[x] ? 255 : 0;
      *pDest2++ = value;
      *pDest2++ = value;
      *pDest2++ = value;
    }
  }
  delete [] pVisible;
  delete [] pConstrain;
}

void CSensor::DrawMemDC(CDC& dc,CRect& rect)
{
  if(!m_pBitmap || rect != m_rect)
  {
    m_rect = rect;
    if(m_pBitmap)
      delete m_pBitmap;
    m_pBitmap = new CBitmap;
    m_pBitmap->CreateCompatibleBitmap(&dc,rect.right,rect.bottom);
    m_bRepaint = true;
  }
  CDC dcMem;
  dcMem.CreateCompatibleDC(&dc);
  CBitmap* pBitmap = dcMem.SelectObject(m_pBitmap);
  if(m_bRepaint)
  {
    m_bRepaint = false;
    dcMem.SaveDC();
    CBrush br(GetSysColor(COLOR_WINDOW));
    dcMem.FillRect(&rect,&br);
    dcMem.SetMapMode(MM_ANISOTROPIC);
    dcMem.SetViewportExt(rect.right,rect.bottom);
    dcMem.SetViewportOrg(0,0);
    switch(m_nView)
    {
    case 0:
      DrawLine(dcMem);
      break;
    case 1:
      DrawColumn(dcMem);
      break;
    case 5:
      DrawDirect(dcMem);
    }
    dcMem.RestoreDC(-1);
  }
  dc.BitBlt(0,0,rect.right,rect.bottom,&dcMem,0,0,SRCCOPY);
  dcMem.SelectObject(pBitmap);
}

POINT CSensor::Project2D(int x,int y,double z)
{
  POINT pt;
  if (m_ySize)
  {
    pt.x = x*2+m_ySize-y;
    pt.y = 500 + (int) ((long) m_yAdd * y / m_ySize) - (int) (z * 500 + 0.5);
  }
  else
  {
    pt.x = x*2;
    pt.y = 500 - (int) (z * 500 + 0.5);
  }
  return pt;
}

void CSensor::Grid3D(CDC& dc)
{
  if(m_bGrid)
  {
    CPen pen(PS_SOLID,0,GetSysColor(COLOR_BTNSHADOW)),
        *pPen = dc.SelectObject(&pen);
    for(int y = 0; y <= m_ySize ; y++)
    {
      dc.MoveTo(Project2D(0,y,1));
      dc.LineTo(Project2D(0,y,0));
      dc.LineTo(Project2D(m_xSize,y,0));
    }
    for(int x = 0; x <= m_xSize; x++)
    {
      dc.MoveTo(Project2D(x,0,1));
      dc.LineTo(Project2D(x,0,0));
      dc.LineTo(Project2D(x,m_ySize,0));
    }
    for(double z = 0; z < 1.05; z += 0.1)
    {
      dc.MoveTo(Project2D(0,m_ySize,z));
      dc.LineTo(Project2D(0,0,z));
      dc.LineTo(Project2D(m_xSize,0,z));
    }
    dc.SelectObject(pPen);
  }
}

void CSensor::DrawLine(CDC& dc)
{
  m_xSize = m_nDimSize[0]-1;
  m_ySize = m_nDimSize[1]-1;
  m_yAdd = (int) (500L * m_ySize / m_xSize / 2);
  dc.SetWindowExt(m_xSize * 2 +  m_ySize,500 + m_yAdd);
  Grid3D(dc);
  if(m_nDimSize[1] == 1)
    for(int x = 0; x < m_nDimSize[0]; x++)
    {
      CPoint pt = Project2D(x,0,m_pData[x] / 255.0);
      if(x)
        dc.LineTo(pt);
      else
        dc.MoveTo(pt);
    }
  else
    for (int y = 0; y < m_nDimSize[1]-1; y++)
      for (int x = 0; x < m_nDimSize[0]-1; x++)
      {
        POINT pt[4];
        pt[0] = Project2D(x,y,m_pData[x + m_nDimSize[0] * y] / 255.0);
        pt[1] = Project2D(x+1,y,m_pData[x+1 + m_nDimSize[0] * y] / 255.0);
        pt[2] = Project2D(x+1,y+1,m_pData[x+1 + m_nDimSize[0] * (y+1)] / 255.0);
        pt[3] = Project2D(x,y+1,m_pData[x + m_nDimSize[0] * (y+1)] / 255.0);
        dc.Polygon(pt,4);
      }
}

void CSensor::DrawColumn(CDC& dc)
{
  m_xSize = m_nDimSize[0];
  m_ySize = m_nDimSize[1];
  m_yAdd = (int) (500L * m_ySize / m_xSize / 2);
  dc.SetWindowExt(m_xSize * 2 +  m_ySize,500 + m_yAdd);
  Grid3D(dc);
  CBrush brGray(GetSysColor(COLOR_BTNFACE)),
         brWhite(GetSysColor(COLOR_BTNHIGHLIGHT)),
			   brDark(GetSysColor(COLOR_BTNSHADOW)),
        *pBrush = dc.SelectObject(&brWhite);
  for(int y = 0; y < m_nDimSize[1]; y++)
    for(int x = 0; x < m_nDimSize[0]; x++)
    {
      double r = m_pData[x + m_nDimSize[0] * y] / 255.0;
      POINT pt[4];
      pt[0] = Project2D(x,y+1,r);
      pt[1] = Project2D(x+1,y+1,r);
      pt[2] = Project2D(x+1,y,r);
      pt[3] = Project2D(x,y,r);
      dc.SelectObject(&brWhite);
      dc.Polygon(pt,4);
      double r2 = y == m_nDimSize[1]-1
                     ? 0
                     : m_pData[x + m_nDimSize[0] * (y+1)] / 255.0;
      if(r2 < r)
      {
        pt[2] = Project2D(x+1,y+1,r2);
        pt[3] = Project2D(x,y+1,r2);
        dc.SelectObject(&brGray);
        dc.Polygon(pt,4);
      }
      r2 = x == m_nDimSize[0]-1 ? 0
                  : m_pData[x+1 + m_nDimSize[0] * y] / 255.0;
      if (r2 < r)
      {
        pt[0] = Project2D(x+1,y,r);
        pt[2] = Project2D(x+1,y+1,r2);
        pt[3] = Project2D(x+1,y,r2);
        dc.SelectObject(&brDark);
        dc.Polygon(pt,4);
      }
    }
  dc.SelectObject(pBrush);
}

void CSensor::DrawDirect(CDC& dc)
{
  ((DirectView*) GetDocument()->GetSimulation()->getView(m_sp))->draw(dc);
}

/////////////////////////////////////////////////////////////////////////////
// CSensor clipboard support

void CSensor::SetClipboardGraphics()
{
  switch(m_nView)
  {
    case 0:
    case 1:
    case 5:
    {
      CMetaFileDC dc;
      dc.Create();
      dc.SetMapMode(MM_ANISOTROPIC);
      switch(m_nView)
      {
      case 0:
        DrawLine(dc);
        break;
      case 1:
        DrawColumn(dc);
        break;
      case 5:
        DrawDirect(dc);
      }
      HMETAFILE hmf = dc.Close();
      GLOBALHANDLE hGMem = GlobalAlloc(GHND,sizeof(METAFILEPICT));
      LPMETAFILEPICT lpMFP = (LPMETAFILEPICT) GlobalLock(hGMem);
      lpMFP->mm = MM_ANISOTROPIC;
      lpMFP->xExt = 6000;
      lpMFP->yExt = 6000;
      lpMFP->hMF = hmf;
      GlobalUnlock(hGMem);
      SetClipboardData(CF_METAFILEPICT,hGMem);
      break;
    }
    case 2:
    case 3:
    case 4:
    {
      int nSize = sizeof(BITMAPINFOHEADER) + 
                  ((m_xSize * 3 + 3) & 0xfffffffc) * m_ySize;
      GLOBALHANDLE hGMem = GlobalAlloc(GHND,nSize);
      memcpy(GlobalLock(hGMem),m_pBuffer,nSize);
      GlobalUnlock(hGMem);
      SetClipboardData(CF_DIB,hGMem);
    }
  }
}

void CSensor::SetClipboardText()
{
  double doubleValue,
        *pDouble;
  Simulation* pSim = GetDocument()->GetSimulation();
  switch(pSim->getSensorportType(m_sp))
  {
    case boolSensor:
    {
      bool boolValue;
      pSim->getSensorportValue(m_sp,boolValue);
      doubleValue = boolValue ? 1 : 0;
      pDouble = &doubleValue;
      break;
    }
    case intSensor:
    {
      bool intValue;
      pSim->getSensorportValue(m_sp,intValue);
      doubleValue = intValue;
      pDouble = &doubleValue;
      break;
    }
    case doubleSensor:
      pSim->getSensorportValue(m_sp,doubleValue);
      pDouble = &doubleValue;
      break;
    case cameraSensor:
    {
      pDouble = new double[m_nDimSize[0] * m_nDimSize[1] * m_nDimSize[2]];
      for(int i = m_nDimSize[0] * m_nDimSize[1] * m_nDimSize[2] - 1; i >= 0; --i)
        pDouble[i] = m_pData[i];
      break;
    }
    case doubleFieldSensor:
      pSim->getSensorportValue(m_sp,pDouble);
  }
  char sDecimal[10];
  GetLocaleInfo(LOCALE_SYSTEM_DEFAULT,LOCALE_SDECIMAL,sDecimal,sizeof(sDecimal));
  char sBuffer[30];
  double* pData = pDouble;
  int nCount = 0;
  for(int y = 0; y < m_nDimSize[1] * m_nDimSize[2]; y++)
    for(int x = 0; x < m_nDimSize[0]; x++)
    {
      double d = *pData;
      sprintf(sBuffer,"%lg",d);
      nCount += strlen(sDecimal)+strlen(sBuffer);
      pData++;
    }
  GLOBALHANDLE hGMem = GlobalAlloc(GHND,nCount);
  if(hGMem)
  {
    char* pBuffer = (char*) ::GlobalLock(hGMem),
        * pChar = pBuffer;
    pData = pDouble;
    for(y = 0; y < m_nDimSize[1] * m_nDimSize[2]; y++)
      for(int x = 0; x < m_nDimSize[0]; x++)
      {
        double d = *pData;
        sprintf(sBuffer,"%lg",d);
        int i = 0;
        while(sBuffer[i] && sBuffer[i] != '.')
          *pChar++ = sBuffer[i++];
        if(sBuffer[i])
        {
          strcpy(pChar,sDecimal);
          pChar += strlen(pChar);
          strcpy(pChar,sBuffer+i+1);
        }
        pChar += strlen(pChar);
        *pChar++ = x + 1 == m_nDimSize[0] ? '\n' : '\t';
        pData++;
      }
    *--pChar = 0;
    GlobalUnlock(hGMem);
    SetClipboardData(CF_TEXT,hGMem);
  }
  if(pSim->getSensorportType(m_sp) == cameraSensor)
    delete [] pDouble;
}

/////////////////////////////////////////////////////////////////////////////
// CSensor diagnostics

#ifdef _DEBUG
void CSensor::AssertValid() const
{
  CView::AssertValid();
}

void CSensor::Dump(CDumpContext& dc) const
{
  CView::Dump(dc);
}

CDoc* CSensor::GetDocument() // non-debug version is inline
{
  ASSERT(m_pDocument->IsKindOf(RUNTIME_CLASS(CDoc)));
  return (CDoc*)m_pDocument;
}
#endif //_DEBUG

void CSensor::Repaint()
{
  m_bRepaint = true;
  Invalidate(false);
}

/////////////////////////////////////////////////////////////////////////////
// CSensor message handlers

void CSensor::OnInitialUpdate() 
{
  if(!RestoreLayout())
  {
    m_sSensor = GetDocument()->GetSelectedObject();
    m_bGrid = 1;
    m_nBright = 0;
    m_nView = 2;
  }
  CString s = "Sensor - ";
  GetParent()->ModifyStyle(FWS_ADDTOTITLE,0);
  GetParent()->SetWindowText(s + m_sSensor);
  m_menuPopup.LoadMenu(IDP_SENSOR);
  m_rect = CRect(0,0,0,0);
  m_pBitmap = 0;
  m_pBuffer = 0;
  m_nBufSize = 0;
  CView::OnInitialUpdate();
}

void CSensor::OnUpdate(CView* pSender, LPARAM lHint, CObject* pHint) 
{
  if(lHint != UPDATE_ACTORS && lHint!= UPDATE_OBJECTS)
  {
    if(lHint != UPDATE_SENSORS)
    {
      if(GetDocument()->GetType(m_sSensor) == OBJECT_TYPE_SENSORPORT)
      {
        Simulation* pSim = GetDocument()->GetSimulation();
        m_sp = pSim->getSensorportId((const char*) m_sSensor);
        m_minValue = pSim->getSensorportMinValue(m_sp);
        m_maxValue = pSim->getSensorportMaxValue(m_sp);
        if(m_pBitmap)
        {
          delete m_pBitmap;
          m_pBitmap = 0;
        }
        if(m_pBuffer)
        {
          delete [] m_pBuffer;
          m_pBuffer = 0;
        }
        if(m_nBufSize)
        {
          delete [] m_pData;
          m_pData = 0;
        }
        const std::vector<int>& dims = pSim->getSensorDimensions(m_sp);
        m_nDim = 0;
        m_nDimSize[0] = m_nDimSize[1] = m_nDimSize[2] = 1;
        m_nBufSize = 1;
        for(unsigned int i = 0; i < 3; i++)
          if(i < dims.size() && dims[i] > 1)
          {
            m_nDimSize[m_nDim] = dims[i];
            m_nBufSize *= dims[i];
            m_nDim++;
          }
        if(pSim->getSensorportType(m_sp) == doubleFieldSensor)
          m_pData = new unsigned char[m_nBufSize];
        else
          m_nBufSize = 0;
        if(pSim->getSensorportType(m_sp) == viewSensor)
          m_nView = 5;
        else if(m_nView == 5)
          m_nView = 2;
        if(m_nView == 3 && (!m_nDim || m_nDimSize[m_nDim-1] > 3))
          m_nView = 2;
        switch (m_nView)
        {
          case 0:
            if(!m_nDim)
              m_nView = 1;
          case 1:
          case 2:
          case 4:
            if(m_nDim > 2)
              m_nView = 3;
          case 3:
            if(m_nDimSize[2] > 3 && m_nDimSize[1] == 2)
              m_nView = 5;
        }
      }
      else
        m_sp = -1;
    }
    Repaint();
  }
}

void CSensor::OnRButtonUp(UINT nFlags, CPoint point) 
{
  ClientToScreen(&point);
  m_menuPopup.GetSubMenu(0)
    ->TrackPopupMenu(TPM_LEFTALIGN | TPM_LEFTBUTTON | TPM_RIGHTBUTTON,
                     point.x,point.y,AfxGetMainWnd());
}

void CSensor::OnViewLine() 
{
  if(m_sp != -1 && m_nDim < 3 && m_nDimSize[0] > 1 && m_nView != 5)
  {
    m_nView = 0;
    Repaint();
  }
}

void CSensor::OnUpdateViewLine(CCmdUI* pCmdUI) 
{
  pCmdUI->Enable(m_sp != -1 && m_nDim < 3 && m_nDimSize[0] > 1 && m_nView != 5);
  pCmdUI->SetCheck(m_nView == 0);
}

void CSensor::OnViewColumn() 
{
  if(m_sp != -1 && m_nDim < 3 && m_nView != 5)
  {
    m_nView = 1;
    Repaint();
  }
}

void CSensor::OnUpdateViewColumn(CCmdUI* pCmdUI) 
{
  pCmdUI->Enable(m_sp != -1 && m_nDim < 3 && m_nView != 5);
  pCmdUI->SetCheck(m_nView == 1);
}

void CSensor::OnViewMono() 
{
  if(m_sp != -1 && m_nDim < 3 && m_nView != 5)
  {
    if(m_pBuffer)
    {
      delete m_pBuffer;
      m_pBuffer = 0;
    }
    m_nView = 2;
    Repaint();
  }
}

void CSensor::OnUpdateViewMono(CCmdUI* pCmdUI) 
{
  pCmdUI->Enable(m_sp != -1 && m_nDim < 3 && m_nView != 5);
  pCmdUI->SetCheck(m_nView == 2);
}

void CSensor::OnViewColor() 
{
  if(m_sp != -1 && m_nDim && m_nDimSize[m_nDim-1] <= 3 && m_nView != 5)
  {
    if(m_pBuffer)
    {
      delete m_pBuffer;
      m_pBuffer = 0;
    }
    m_nView = 3;
    Repaint();
  }
}

void CSensor::OnUpdateViewColor(CCmdUI* pCmdUI) 
{
  pCmdUI->Enable(m_sp != -1 && m_nDim && m_nDimSize[m_nDim-1] <= 3 && m_nView != 5);
  pCmdUI->SetCheck(m_nView == 3);
}

void CSensor::OnViewStereo() 
{
  if(m_sp != -1 && m_nDim < 3 && m_nView != 5)
  {
    if(m_pBuffer)
    {
      delete m_pBuffer;
      m_pBuffer = 0;
    }
    m_nView = 4;
    Repaint();
  }
}

void CSensor::OnUpdateViewStereo(CCmdUI* pCmdUI)
{
  pCmdUI->Enable(m_sp != -1 && m_nDim < 3 && m_nView != 5);
  pCmdUI->SetCheck(m_nView == 4);
}

void CSensor::OnViewGrid() 
{
  if(m_sp != -1 && m_nView < 2)
  {
    m_bGrid ^= 1;
    Repaint();
  }
}

void CSensor::OnUpdateViewGrid(CCmdUI* pCmdUI) 
{
  pCmdUI->Enable(m_sp != -1 && m_nView < 2);
  pCmdUI->SetCheck(m_bGrid);
}

void CSensor::OnBright() 
{
  if(m_sp != -1 && (m_nView == 2 || m_nView == 3))
  {
    m_nBright = LOWORD(GetCurrentMessage()->wParam) - ID_BRIGHT0;
    Repaint();
  }
}

void CSensor::OnUpdateBright(CCmdUI* pCmdUI) 
{
  pCmdUI->Enable(m_sp != -1 && (m_nView == 2 || m_nView == 3));
  pCmdUI->SetCheck((int) pCmdUI->m_nID == m_nBright + ID_BRIGHT0);
}

BOOL CSensor::OnEraseBkgnd(CDC* pDC) 
{
  return true;
}

void CSensor::OnEditCopy() 
{
  if(m_sp != -1)
  {
    OpenClipboard();
    EmptyClipboard();
    SetClipboardGraphics();
    if(m_nDimSize[0] * m_nDimSize[1] * m_nDimSize[2] < 10000 && m_nView != 5)
    {
      SetClipboardText();
    }
    CloseClipboard();
  }
}

void CSensor::OnUpdateEditCopy(CCmdUI* pCmdUI) 
{
  pCmdUI->Enable(m_sp != -1);
}
