/**
* @file RobotRemote.h
*
* Definition of the RobotRemote application
*
* @author Martin Ltzsch
*/

#ifndef __RobotRemote_h_
#define __RobotRemote_h_

#include "StdAfx.h"
#include "resource.h"		
#include "Tools/MessageQueue/MessageQueue.h"
#include "Platform/Win32/Thread.h"
#include "Platform/Win32Linux/TCPConnection.h"
#include "Representations/JoystickData.h"
#include "Representations/Motion/MotionRequest.h"
#include "Representations/Perception/Image.h"
#include <map>
#include <mmsystem.h>

/// class prototypes to avoid inclusion of RobotRemoteDlg.h 
class CRobotRemoteDlg1; 
class CRobotRemoteDlg2; 

/**
* @class CRobotRemoteWLanThread
*
* A thread that manages the communication between the queues and the Wireless Lan
*
* @author Martin Ltzsch
*/
class CRobotRemoteWLanThread : public Thread<CRobotRemoteWLanThread>
{
public:
  DECLARE_SYNC; /**< Make this object synchronizable. */
  
  /** Constructor */
  CRobotRemoteWLanThread();
  
  /** Destructor */
  ~CRobotRemoteWLanThread();
  
  /** A queue for messages to be sent via the WLAN to the robot */
  MessageQueue queueToRobot;
  
  /** A queue for messages to be received from the WLAN */
  MessageQueue queueFromRobot;
  
  /** Returns whether WLAN connection is established */
  bool wlanIsConnected();
  
  /** This function establishes the connection over the WLan interface and must be called before using it, of course **/
  void wlanConnect(const unsigned long remoteIP, int remotePort);
  
  /** This function terminates an WLan connection **/
  void wlanDisconnect();
  
private:  
  /** The thread main function */
  void communicate();
  
  /** a TCP/IP client class handling the WLan interface **/
  TcpConnection* tcpConnection;
  unsigned long remoteIP;
  int remotePort;
  bool wlanNeedsToBeConnected;
};


/** 
* @class CRobotRemoteApp
* 
* The application class of RobotRemote
*/
class CRobotRemoteApp : public CWinApp, public MessageHandler
{
public:
  DECLARE_SYNC; /**< Make this object synchronizable. */

  /** Constructor */
	CRobotRemoteApp();

  /** The first configuration dialog */
  CRobotRemoteDlg1* dlg1;

  /** The second main dialog */
  CRobotRemoteDlg2* dlg2;

  /** updates the visible of the both dialogs */
  void updateDialogs();

  /** A message queue for messages to the robot */
  MessageQueue toRobot;

  /** A message queue for messages from the robot */
  MessageQueue fromRobot;

  /** The ip address of the robot */
  unsigned long ip;

  /** A WLAN thread */
  CRobotRemoteWLanThread wlanThread;

  /** Idle Processing */
  void OnIdle();

  /** Determines if a joystick is connected */
  bool joystickIsConnected;

  /** Determines if the WLAN connection is established */
  bool wlanIsConnected;

  /** the last received image */
  Image image;

  /** sends the joystick data to the robot */
  void sendJoystickData(JOYINFOEX& ji);

  /** the current joystick */
  unsigned char joystick;

  /** is called when joystick changed */
  void onChangeJoystick();

  /** different configurations */
  enum Configuration { noImages, someImages, allImages };

  /** The current configuration */
  enum Configuration configuration;

  /** is called when configuration changed */
  void onChangeConfiguration();

  /** The special-actions mapped on joystick-buttons **/
  std::map<int,SpecialActionRequest::SpecialActionID> buttonMapping;
  
  /** The configuration for manipulating axis **/
  bool invertZ;
  bool switchYandZ;
  
  //{{AFX_VIRTUAL(CRobotRemoteApp)
	public:
	virtual BOOL InitInstance();
	virtual int ExitInstance();
	virtual BOOL OnIdle(LONG lCount);
	//}}AFX_VIRTUAL


	//{{AFX_MSG(CRobotRemoteApp)
	//}}AFX_MSG
	DECLARE_MESSAGE_MAP()

  /** 
  * Called from a MessageQueue to distribute messages.
  * Use message.getMessageID to decide if the message is relavant for 
  * the MesssageHandler derivate.
  * Use message.bin, message.text or message.config as In streams to get the data from.
  * @param message The message that can be read.
  * @return true if the message was read (handled).
  */
  virtual bool handleMessage(InMessage& message);

};

/** Returns a pointer to the CRobotRemoteApp */
CRobotRemoteApp* getRobotRemoteApp();

#endif // __RobotRemote_h_

/** 
* Change Log:
*
* $Log: RobotRemote.h,v $
* Revision 1.4  2004/06/14 14:40:29  thomas
* added config-parameters to invert z-axis or switch y- and z-axis
* added stand in joystick-behaviour for small values
* updated all special-actions for robot-remote after renaming
*
* Revision 1.3  2004/06/03 12:49:37  thomas
* updated to changes made in motion-request
*
* Revision 1.2  2004/05/28 15:51:55  thomas
* added support for multiple joysticks in robotremote
* added solution gt2004-joystick-controlled
*
* Revision 1.3  2004/05/24 15:40:41  thomas
* added combo-box to support for multiple joysticks
*
* Revision 1.2  2004/04/28 16:12:35  thomas
* switched y- and z-axis in joystick-behavior
* added restoring of solution from module.cfg before disconnect
* remove model-specific image-settings
*
* Revision 1.1  2004/04/26 15:58:59  thomas
* added new project RobotRemote based on ATHAiboControl
*
* Revision 1.2  2004/01/26 13:35:02  loetzsch
* improved
*
* Revision 1.1  2004/01/24 14:55:28  loetzsch
* created ATH AiboControl
*
*/
