/**
* @file RobotRemote.cpp
*
* Implementation of the RobotRemote application
*
* @author Martin Ltzsch
*/

#include "stdafx.h"
#include "RobotRemote.h"
#include "RobotRemoteDlg.h"
#include "Platform/GTAssert.h"
#include "Tools/Debugging/QueueFillRequest.h"
#include "Tools/Debugging/DebugKeyTable.h"
#include "Representations/JoystickData.h"
#include "Representations/Motion/MotionRequest.h"
#include "Representations/Perception/CameraParameters.h"
#include "Tools/Debugging/DebugKeyTable.h"
#include "Tools/Debugging/QueueFillRequest.h"
#include "Tools/Location.h"
#include "Tools/Module/SolutionRequest.h"
#include <iterator>

#define HAVE_BOOLEAN
#include "Representations/Perception/JPEGImage.h"

BEGIN_MESSAGE_MAP(CRobotRemoteApp, CWinApp)
//{{AFX_MSG_MAP(CRobotRemoteApp)
//}}AFX_MSG
ON_COMMAND(ID_HELP, CWinApp::OnHelp)
END_MESSAGE_MAP()


CRobotRemoteApp::CRobotRemoteApp()
: joystickIsConnected(false), wlanIsConnected(false),
dlg1(0), dlg2(0), joystick(0), configuration(noImages)
{
}

CRobotRemoteApp theApp;

CRobotRemoteApp* getRobotRemoteApp()
{
  return &theApp;
}

void CRobotRemoteApp::onChangeConfiguration()
{
  DebugKeyTable debugKeyTable;
  switch(configuration)
  {
  case allImages:
    debugKeyTable.set(DebugKeyTable::sendJPEGImage,DebugKey::always);
    break;
  case someImages:
    debugKeyTable.set(DebugKeyTable::sendJPEGImage,DebugKey::every_n_ms,500);
    break;
  case noImages:
    debugKeyTable.set(DebugKeyTable::sendJPEGImage,DebugKey::disabled);
    break;
  }
  toRobot.out.bin << debugKeyTable;
  toRobot.out.finishMessage(idDebugKeyTable);

  SolutionRequest setting;
  char section[20];
  strcpy(section, "Modules");
  InConfigFile configFile(getLocation().getModelFilename("robotremote.cfg"),section);
  if (!configFile.exists() || configFile.eof()) {
    //OUTPUT(idText,text,"RobotRemote : Error, config-file not found");
  } else {
    configFile >> setting;
    // allow specifying only used modules in config-file
    setting.errorWhileReading = false;
//    if (setting.errorWhileReading) {
//      PRINT("RobotRemote : Error, robotremote.cfg contained errors. Please check.");
//    }
  }
  toRobot.out.bin << setting;
  toRobot.out.finishMessage(idSolutionRequest);

  // joystick-button-mapping
  // - disabled buttons are skipped
  // - specified special-actions are directly executed
  // - else joystick-button is given to the behaviour
  buttonMapping.clear();
  strcpy(section, "Buttons");
  InConfigFile configFileButtons(getLocation().getModelFilename("robotremote.cfg"),section);
  if (!configFileButtons.exists() || configFileButtons.eof()) {
    //OUTPUT(idText,text,"RobotRemote : Error, config-file not found");
  } else {
    int button;
    char action[80];
    while(!configFileButtons.eof())
    {
      configFileButtons >> button;
      if(configFileButtons.eof()) break;
      configFileButtons >> action;
      if(strcmp(action,"disabled")==0) {
        buttonMapping[button] = SpecialActionRequest::numOfSpecialAction;
      } else {
        buttonMapping[button] = SpecialActionRequest::getSpecialActionIDFromName(action);
      }
    }
  }

  // axis-configuration
  strcpy(section, "Axis");
  InConfigFile configFileAxis(getLocation().getModelFilename("robotremote.cfg"),section);
  if (!configFileAxis.exists() || configFileAxis.eof()) {
    //OUTPUT(idText,text,"RobotRemote : Error, config-file not found");
  } else {
    char key[20];
    int value;
    while(!configFileAxis.eof())
    {
      configFileAxis >> key;
      if(configFileAxis.eof()) break;
      configFileAxis >> value;
      if(strcmp(key,"invertZ")==0) {
        invertZ =  value == 1;
      } else if(strcmp(key,"switchYandZ")==0) {
        switchYandZ =  value == 1;
      } else {
        //OUTPUT(idText,text,"RobotRemote : Error, unrecognized key in section axis");
      }
    }
  }

  QueueFillRequest defaultQueueFillRequest;
  defaultQueueFillRequest.mode = QueueFillRequest::overwriteOlder;
  toRobot.out.bin << defaultQueueFillRequest;
  toRobot.out.finishMessage(idQueueFillRequest);

/*  CameraParameters defaultCameraParameters;
  defaultCameraParameters.theGain = CameraParameters::gain_high;
  defaultCameraParameters.theWhiteBalance = CameraParameters::wb_fl_mode;
  defaultCameraParameters.theShutterSpeed = CameraParameters::shutter_slow;
  toRobot.out.bin << defaultCameraParameters;
  toRobot.out.finishMessage(idCameraParameters);*/
}

BOOL CRobotRemoteApp::InitInstance()
{
  if (!AfxSocketInit())
  {
    AfxMessageBox(IDP_SOCKETS_INIT_FAILED);
    return FALSE;
  }
  
  SetRegistryKey(_T("GermanTeam"));

  ip = GetProfileInt("","ip",0);
  configuration = (Configuration)GetProfileInt("","configuration",configuration);

  // Mit der nchsten Zeile kann man das Aussehen des 2. Dialoges testen,
  // ohne dass das WLan verbunden sein muss.
  // dlg2 = new CRobotRemoteDlg2(); dlg2->DoModal(); delete dlg2; dlg2 = 0;

  while(1)
  {
    onChangeConfiguration();
    dlg1 = new CRobotRemoteDlg1();
    if (dlg1->DoModal() == IDCANCEL) return FALSE;
    delete dlg1;
    dlg1 = 0;
    
    if (wlanIsConnected && joystickIsConnected)
    {
      dlg2 = new CRobotRemoteDlg2();
      dlg2->DoModal();
      delete dlg2;
      dlg2 = 0;
      wlanThread.wlanDisconnect();
    }
  }
  return FALSE;
}

bool CRobotRemoteApp::handleMessage(InMessage& message)
{
  switch(message.getMessageID())
  {
  case idText:
    {
      ASSERT(message.getMessageSize());
      char* buf = (char*)malloc(message.getMessageSize()*2+1);
      message.text.readAll(buf);
      PRINT(buf);
      delete[](buf);
      return true;
    }
  case idJPEGImage:
    {
      JPEGImage jpegImage;
      message.bin >> jpegImage;
      jpegImage.toImage(image);
      updateDialogs();
      return true;
    }
  case idImage:
    {
      message.bin >> image;
      updateDialogs();
    }
  default:
    return false;
  }
}

BOOL CRobotRemoteApp::OnIdle(LONG lCount) 
{
  this->OnIdle();

  return CWinApp::OnIdle(lCount);
}

void CRobotRemoteApp::updateDialogs()
{
  if (dlg1 != 0) dlg1->update();
  if (dlg2 != 0) dlg2->update();
}

void CRobotRemoteApp::OnIdle() 
{
  JOYINFOEX ji;
  //MMRESULT joyresult;

  //joyresult=joyGetPosEx(joystick,&ji);
  ji.dwSize = sizeof(ji);
  ji.dwFlags = JOY_RETURNALL;
  if(joyGetPosEx(JOYSTICKID1,&ji) == 0)
  //if (joyresult == 0)
  {
    if (!joystickIsConnected) 
    {
      joystickIsConnected = true;
      updateDialogs();
    }
    sendJoystickData(ji);
  }
  else
  {
    if (joystickIsConnected)
    {
      joystickIsConnected = false;
      updateDialogs();
    }
  }

  {  
    SYNC_WITH(wlanThread);
    
    if (wlanThread.isRunning())
    {
      if (wlanThread.wlanIsConnected())
      {
        if (!wlanIsConnected)
        {
          wlanIsConnected = true;
          updateDialogs();
        }
        toRobot.moveAllMessages(wlanThread.queueToRobot);
      }
      else
      {
        if (wlanIsConnected)
        {
          wlanIsConnected = false;
          updateDialogs();
        }
      }
    }
  }
  {
    SYNC; // now no other threads can access the handled queues
    
    fromRobot.handleAllMessages(*this);
    fromRobot.clear();
  }

  if (wlanIsConnected && joystickIsConnected)
  {
    if (dlg1!=0) dlg1->EndModalLoop(0);
  }

}

void CRobotRemoteApp::sendJoystickData(JOYINFOEX& ji)
{
  JoystickData d;

  d.x = -1.0*((double)ji.dwYpos-32767.0)/32768.0;
  d.y = -1.0*((double)ji.dwXpos-32767.0)/32768.0;
  d.z = -1.0*((double)ji.dwRpos-32767.0)/32768.0;
  d.accel = -1.0*((double)ji.dwZpos-32767.0)/32768.0;
  d.button = ji.dwButtons;
  d.directSpecialAction = false;
  d.coolie = (ji.dwPOV==65535?-1:ji.dwPOV/4500) + 1;

  if(d.x < -1.0 || d.x > 1.0) d.x = 0.0;
  if(d.y < -1.0 || d.y > 1.0) d.y = 0.0;
  if(d.z < -1.0 || d.z > 1.0) d.z = 0.0;
  if(d.accel < -1.0 || d.accel > 1.0) d.accel = 0.0;

  if(invertZ) d.z = -1.0 * d.z;
  if(switchYandZ)
  {
    double tmp = d.y;
    d.y = d.z;
    d.z = tmp;
  }

  if (d.button > 0
    && d.button <= (1 << (numOfJoystickButtons-1)))
  {
    // get lowest pressed button-number from bit-array
    int lowestPressed, mask = 1;
    for(lowestPressed=1;lowestPressed<=numOfJoystickButtons;lowestPressed++)
    {
      if(mask & d.button) break;
      mask *= 2;
    }
    std::map<int,SpecialActionRequest::SpecialActionID>::iterator it = buttonMapping.find(lowestPressed);
    if(it!=buttonMapping.end())
    {
      if(buttonMapping[lowestPressed]!=SpecialActionRequest::numOfSpecialAction)
      {
        d.directSpecialAction = true;
        d.specialActionID = buttonMapping[lowestPressed];
      }
      // reset button if mapped directly to a special-action or disabled
      d.button = 0;
    }
  }

  toRobot.out.bin << d;
  toRobot.out.finishMessage(idJoystickData);
  toRobot.removeRepetitions();
}

int CRobotRemoteApp::ExitInstance() 
{
  wlanThread.wlanDisconnect();
  WriteProfileInt("","ip",ip);
  WriteProfileInt("","configuration",configuration);

  return CWinApp::ExitInstance();
}

CRobotRemoteWLanThread::CRobotRemoteWLanThread()
{
  tcpConnection = 0;
  wlanNeedsToBeConnected = false;
  start(this,communicate);
}

void CRobotRemoteWLanThread::communicate()
{
  while(isRunning())
  {
    if(wlanNeedsToBeConnected)
    {
      if(tcpConnection)
        delete tcpConnection;
      char strRemoteIP[32];
      sprintf(strRemoteIP, "%i.%i.%i.%i", remoteIP >> 24,
        remoteIP >> 16 & 255,
        remoteIP >> 8 & 255,
        remoteIP & 255);
      tcpConnection = new TcpConnection(strRemoteIP, remotePort);
      
      if(wlanIsConnected())
      {
        char s[200];
        sprintf(s,"CRobotRemoteWLanThread : connected to %s:%i.",strRemoteIP,remotePort);
        PRINT(s);
      }
      else
      {
        char s[200];
        sprintf(s,"CRobotRemoteWLanThread : could not connect to %s:%i.",strRemoteIP,remotePort);
        PRINT(s);
      }
      wlanNeedsToBeConnected = false;
    }
    
    if(wlanIsConnected())
    {
      char* sendData,
        * receivedData;
      int sendSize = 0,
        receivedSize = 0;
      
      sendData = 0;
      
      // If there is something to send, prepare a package
      {
        SYNC;
        if(!queueToRobot.isEmpty())
        {
          OutBinarySize size;
          size << queueToRobot;
          sendSize = size.getSize();
          sendData = new char[sendSize];
          OutBinaryMemory stream(sendData);
          stream << queueToRobot;
        }
      }
      
      // exchange data with the router
      if(tcpConnection->sendAndReceive(sendData,sendSize,receivedData,receivedSize))
        queueToRobot.clear();
      
      // If a package was prepared, remove it
      if(sendSize)
        delete [] sendData;
      
      // If a package was received from the router program, add it to receiver queue
      if(receivedSize > 0)
      {
        InBinaryMemory stream(receivedData,receivedSize);
        stream >> queueFromRobot;
        delete [] receivedData;
        
        // move all messages to the queue from physical robots
        
        SYNC_WITH(*getRobotRemoteApp());
        queueFromRobot.moveAllMessages(getRobotRemoteApp()->fromRobot);
        
      }
    }
    Sleep(10);
  }
}

bool CRobotRemoteWLanThread::wlanIsConnected()
{
  if(tcpConnection && !tcpConnection->isConnected())
  {
    delete tcpConnection;
    tcpConnection = 0;
  }
  return tcpConnection != 0;
}

/** This function establishes the connection over the WLan interface and must be called before using it, of course **/
void CRobotRemoteWLanThread::wlanConnect(const unsigned long theRemoteIP, int theRemotePort)
{
  SYNC;
  remoteIP = theRemoteIP;
  remotePort = theRemotePort;
  wlanNeedsToBeConnected = true;
  char strRemoteIP[32];
  sprintf(strRemoteIP, "%i.%i.%i.%i", remoteIP >> 24,
    remoteIP >> 16 & 255,
    remoteIP >> 8 & 255,
    remoteIP & 255);
  char s[200];
  sprintf(s,"CRobotRemoteWLanThread : trying to connect to %s:%i.", strRemoteIP, remotePort);
  PRINT(s);
}

/** This function terminates an WLan connection **/
void CRobotRemoteWLanThread::wlanDisconnect()
{
  SolutionRequest setting;
  setting.setDefaultSolutions();
  queueToRobot.out.bin << setting;
  queueToRobot.out.finishMessage(idSolutionRequest);

  {
    SYNC;
    if(wlanIsConnected())
    {
    /** send empty DebugKeyTable and immediateReadWrite to prevent useless full queues on robot
    * @todo this might be not usefull when in QueueFillRequest::overwriteOlder mode
      */
      QueueFillRequest queueFillRequest;
      queueToRobot.out.bin << queueFillRequest;
      queueToRobot.out.finishMessage(idQueueFillRequest);
      DebugKeyTable debugKeyTable;
      queueToRobot.out.bin << debugKeyTable;
      queueToRobot.out.finishMessage(idDebugKeyTable);
    }  
  }
  Sleep(800);
  SYNC;
  if(wlanIsConnected())
  {
    delete tcpConnection;
    tcpConnection = 0;
    PRINT("CRobotRemoteWLanThread : disconnected");
  }
}

CRobotRemoteWLanThread::~CRobotRemoteWLanThread()
{
}


/*
* Change Log:
* 
* $Log: RobotRemote.cpp,v $
* Revision 1.6  2004/06/14 14:40:29  thomas
* added config-parameters to invert z-axis or switch y- and z-axis
* added stand in joystick-behaviour for small values
* updated all special-actions for robot-remote after renaming
*
* Revision 1.5  2004/06/03 12:49:37  thomas
* updated to changes made in motion-request
*
* Revision 1.4  2004/05/28 18:07:48  thomas
* updated robotremote for new locations
*
* Revision 1.3  2004/05/28 15:51:55  thomas
* added support for multiple joysticks in robotremote
* added solution gt2004-joystick-controlled
*
* Revision 1.3  2004/05/24 15:40:41  thomas
* added combo-box to support for multiple joysticks
*
* Revision 1.2  2004/04/28 16:12:35  thomas
* switched y- and z-axis in joystick-behavior
* added restoring of solution from module.cfg before disconnect
* remove model-specific image-settings
*
* Revision 1.1  2004/04/26 15:58:59  thomas
* added new project RobotRemote based on ATHAiboControl
*
* Revision 1.5  2004/01/26 23:03:16  loetzsch
* RobotRemote disables Vision and Selflocalization, as they are too slow
* on non-SuperCore ERS210
*
* Revision 1.4  2004/01/26 17:47:42  loetzsch
* bug fix
*
* Revision 1.3  2004/01/26 13:35:02  loetzsch
* improved
*
* Revision 1.2  2004/01/24 20:18:40  loetzsch
* improved ATH AiboControl
*
* Revision 1.1  2004/01/24 14:55:28  loetzsch
* created ATH AiboControl
*
*/

