/** 
* @file PaintMethodsWin32.cpp
* Implementation of class PaintMethodsWin32.
*
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Jngel</A>
*/

#include "StdAfx.h"

#include "PaintMethodsWin32.h"
#include "Tools/Actorics/RobotDimensions.h"


void PaintMethodsWin32::paintDebugDrawingToCDC(
 const DebugDrawing& debugDrawing,
  CDC &dc
 )
{
  for(List<DebugDrawing::Element*>::Pos i = debugDrawing.elements.getFirst(); i; ++i)
    switch(debugDrawing.elements[i]->type)
  {
      case DebugDrawing::Element::POLYGON:
        {
          const DebugDrawing::Polygon& element = *(const DebugDrawing::Polygon*) debugDrawing.elements[i];
          CBrush coloredBrush;
          CBrush* oldBrush;
          coloredBrush.CreateSolidBrush(RGB(
            element.fillColor.red,
            element.fillColor.green,
            element.fillColor.blue
            ));
          oldBrush = dc.SelectObject(&coloredBrush);
          
          int penStyle;
          switch (element.penStyle)
          {
          case Drawings::ps_solid : penStyle = PS_SOLID; break;
          case Drawings::ps_dash  : penStyle = PS_DASH; break;
          case Drawings::ps_dot   : penStyle = PS_DOT; break;
          case Drawings::ps_null  : penStyle = PS_NULL; break;
          default:penStyle = PS_SOLID;
          }
          
          CPen coloredPen;
          CPen* oldPen;
          coloredPen.CreatePen(penStyle, element.width, RGB(
            element.penColor.red,
            element.penColor.green,
            element.penColor.blue
            ));
          oldPen = dc.SelectObject(&coloredPen);
          
          //copy vector2 to points
          POINT points[16];
          for(int n = 0; n < element.nCount; n++)
          {
            points[n].x = element.points[n].x;
            points[n].y = element.points[n].y;
          }
          if(element.fillStyle == Drawings::bs_null)
          {
            dc.Polyline(points, element.nCount);
            dc.MoveTo(points[0].x,points[0].y);
            dc.LineTo(points[element.nCount - 1]);
          }
          else
          {
            dc.Polygon(points, element.nCount);
          }
          
          dc.SelectObject(oldBrush);
          dc.SelectObject(oldPen);
          coloredBrush.DeleteObject();
          coloredPen.DeleteObject();
          break;
        }
      case DebugDrawing::Element::ELLIPSE:
        {
          const DebugDrawing::Ellipse& element = *(const DebugDrawing::Ellipse*) debugDrawing.elements[i];
          CBrush coloredBrush;
          CBrush* oldBrush;
          LOGBRUSH logbrush;
          logbrush.lbColor = RGB(
            element.fillColor.red,
            element.fillColor.green,
            element.fillColor.blue
            );
          logbrush.lbHatch = 0;
          switch(element.fillStyle)
          {
          case Drawings::bs_solid: logbrush.lbStyle = BS_SOLID; break;
          case Drawings::bs_null: logbrush.lbStyle = BS_HOLLOW; break;
          default:logbrush.lbStyle = BS_HOLLOW;
          }
          
          coloredBrush.CreateBrushIndirect(&logbrush);
          oldBrush = dc.SelectObject(&coloredBrush);
          
          int penStyle;
          switch (element.penStyle)
          {
          case Drawings::ps_solid : penStyle = PS_SOLID; break;
          case Drawings::ps_dash  : penStyle = PS_DASH; break;
          case Drawings::ps_dot   : penStyle = PS_DOT; break;
          case Drawings::ps_null  : penStyle = PS_NULL; break;
          default:penStyle = PS_SOLID;
          }
          
          CPen coloredPen;
          CPen* oldPen;
          coloredPen.CreatePen(penStyle, element.width, RGB(
            element.penColor.red,
            element.penColor.green,
            element.penColor.blue
            ));
          oldPen = dc.SelectObject(&coloredPen);
          
          dc.Ellipse(
            (int)(element.left),
            (int)(element.top),
            (int)(element.right),
            (int)(element.bottom)
            );
          
          dc.SelectObject(oldBrush);
          dc.SelectObject(oldPen);
          coloredBrush.DeleteObject();
          coloredPen.DeleteObject();
          break;
        }
      case DebugDrawing::Element::LINE:
        {
          const DebugDrawing::Line& element = *(const DebugDrawing::Line*) debugDrawing.elements[i];
          int penStyle;
          switch (element.penStyle)
          {
          case Drawings::ps_solid : penStyle = PS_SOLID; break;
          case Drawings::ps_dash  : penStyle = PS_DASH; break;
          case Drawings::ps_dot   : penStyle = PS_DOT; break;
          case Drawings::ps_null  : penStyle = PS_NULL; break;
          default:penStyle = PS_SOLID;
          }
          
          CPen coloredPen;
          CPen* oldPen;
          coloredPen.CreatePen(penStyle, element.width, RGB(
            element.penColor.red,
            element.penColor.green,
            element.penColor.blue
            ));
          oldPen = dc.SelectObject(&coloredPen);
          
          dc.MoveTo((int)(element.xStart), (int)(element.yStart) );
          dc.LineTo((int)(element.xEnd), (int)(element.yEnd) );
          
          dc.SelectObject(oldPen);
          coloredPen.DeleteObject();
          break;
        }
    }
}

void PaintMethodsWin32::paintColorClassUV2CDC
(
 BITMAPINFOHEADER* pBuffer,
 ColorTable& colorTable
 )
{
  int xSize = 3*64;
  int ySize = 3*64;
  
  // initializes the bitmap with black
  memset((pBuffer + 1), 0, ((xSize * 3 + 3) & 0xfffffffc) * ySize);
  
  unsigned char y,u,v;
  
  bool alreadyPainted[64][64];
  
  for (u = 0; u < 64; u++)
  {
    for (v = 0; v < 64; v++)
    {
      alreadyPainted[u][v] = false;
    }
  }
  
  char *p; // pointer for navigation in the bitmap
  
  for (u = 0; u < 64; u++)
  {
    for (v = 0; v < 64; v++)
    {
      for (y = 63; y > 0; y--)
      {
        if (colorTable.getColorClass(y << 2, u << 2, v << 2) != noColor && !alreadyPainted[u][v])
        {
          int r,g,b;
          int rgb = ColorClasses::colorClassToRGB((colorClass)colorTable.getColorClass(y << 2, u << 2, v << 2));
          r = rgb & 0xff;
          rgb >>= 8;
          g = rgb & 0xff;
          rgb >>= 8;
          b = rgb & 0xff;
          
          //sets 9 pixels with the color of the color class.
          p = (char*)(pBuffer + 1) + (int)(3 * xSize * 3 * v + 3 * 3 * u);
          *p++ = b; *p++ = g; *p++ = r;
          *p++ = b; *p++ = g; *p++ = r;
          *p++ = b; *p++ = g; *p++ = r;
          
          p = (char*)(pBuffer + 1) + (int)(3 * xSize * (3 * v + 1) + 3 * 3 * u);
          *p++ = b; *p++ = g; *p++ = r;
          *p++ = b; *p++ = g; *p++ = r;
          *p++ = b; *p++ = g; *p++ = r;
          
          p = (char*)(pBuffer + 1) + (int)(3 * xSize * (3 * v + 2) + 3 * 3 * u);
          *p++ = b; *p++ = g; *p++ = r;
          *p++ = b; *p++ = g; *p++ = r;
          *p++ = b; *p++ = g; *p++ = r;
          
          alreadyPainted[u][v] = true;
          //leaves the inner loop
          y = 1;
        }
      }
    }
  }
}

void PaintMethodsWin32::paintColorClassUY2CDC
(
 BITMAPINFOHEADER* pBuffer,
 ColorTable& colorTable
 )
{
  int xSize = 3*64;
  int ySize = 3*64;
  
  // initializes the bitmap with black
  memset((pBuffer + 1), 0, ((xSize * 3 + 3) & 0xfffffffc) * ySize);
  
  bool alreadyPainted[64][64];
  
  unsigned char y,u,v;
  for (u = 0; u < 64; u++)
  {
    for (y = 0; y < 64; y++)
    {
      alreadyPainted[u][y] = false;
    }
  }
  
  char *p; //pointer for navigation in the bitmap
  
  for (u = 0; u < 64; u++)
  {
    for (y = 0; y < 64; y++)
    {
      for (v = 63; v > 0; v--)
      {
        if (colorTable.getColorClass(y << 2, u << 2, v << 2) != noColor && !alreadyPainted[u][y])
        {
          int r,g,b;
          int rgb = ColorClasses::colorClassToRGB((colorClass)colorTable.getColorClass(y << 2, u << 2, v << 2));
          r = rgb & 0xff;
          rgb >>= 8;
          g = rgb & 0xff;
          rgb >>= 8;
          b = rgb & 0xff;
          
          //sets 9 pixels with the color of the color class.
          p = (char*)(pBuffer + 1) + (int)(3 * xSize * 3 * y + 3 * 3 * u);
          *p++ = b; *p++ = g; *p++ = r;
          *p++ = b; *p++ = g; *p++ = r;
          *p++ = b; *p++ = g; *p++ = r;
          
          p = (char*)(pBuffer + 1) + (int)(3 * xSize * (3 * y + 1) + 3 * 3 * u);
          *p++ = b; *p++ = g; *p++ = r;
          *p++ = b; *p++ = g; *p++ = r;
          *p++ = b; *p++ = g; *p++ = r;
          
          p = (char*)(pBuffer + 1) + (int)(3 * xSize * (3 * y + 2) + 3 * 3 * u);
          *p++ = b; *p++ = g; *p++ = r;
          *p++ = b; *p++ = g; *p++ = r;
          *p++ = b; *p++ = g; *p++ = r;
          
          alreadyPainted[u][y] = true;
          //leaves the inner loop
          v = 1;
        }
      }
    }
  }
}

void PaintMethodsWin32::paintColorClassUV2CDC
(
 CDC* dcMem,
 unsigned char slice,
 bool init,
 ColorTable* pColorTable
 )
{
  CBrush coloredBrush[numOfColors];
  createColoredPens(&coloredBrush[0]);
  CBrush* oldBrush; 
  oldBrush = dcMem->SelectObject(&coloredBrush[0]);
  if (init)
  {
    dcMem->Rectangle(0, 0, 64 * 3, 64 * 3);
  }
  unsigned char y = slice;
  for (unsigned char u = 0; u < 64; u++)
  {
    for (unsigned char v = 0; v < 64; v++)
    {
      if (pColorTable->getColorClass(y << 2, u << 2, v << 2) != noColor)
      {
        dcMem->SelectObject(&coloredBrush[pColorTable->getColorClass(y << 2, u << 2, v << 2)]);
        dcMem->Rectangle(u * 3, 64 * 3 - v * 3 - 5, u * 3 + 5, 64 * 3 - v * 3);
      }
    }
  }
  dcMem->SelectObject(oldBrush);
  deleteColoredPens(&coloredBrush[0]);
}

void PaintMethodsWin32::paintColorClassUY2CDC
(
 CDC* dcMem,
 unsigned char slice,
 bool init,
 ColorTable* pColorTable
 )
{
  CBrush coloredBrush[numOfColors];
  createColoredPens(&coloredBrush[0]);
  CBrush* oldBrush; 
  oldBrush = dcMem->SelectObject(&coloredBrush[0]);
  if (init)
  {
    dcMem->Rectangle(0, 0, 64 * 3, 64 * 3);
  }
  unsigned char v = slice;
  for (unsigned char y = 0; y < 64; y++)
  {
    for (unsigned char u = 0; u < 64; u++)
    {
      if (pColorTable->getColorClass(y << 2, u << 2, v << 2) != noColor)
      {
        dcMem->SelectObject(&coloredBrush[pColorTable->getColorClass(y << 2, u << 2, v << 2)]);
        dcMem->Rectangle(u * 3, 64 * 3 - y * 3 - 4, u * 3 + 4, 64 * 3 - y * 3);
      }
    }
  }
  dcMem->SelectObject(oldBrush);
  deleteColoredPens(&coloredBrush[0]);
}

void PaintMethodsWin32::createColoredPens(CBrush* coloredBrush)
{
  for (int color = 0; color < numOfColors; color++)
  {
    coloredBrush[color].CreateSolidBrush(ColorClasses::colorClassToRGB((colorClass)color));
  }
}

void PaintMethodsWin32::deleteColoredPens(CBrush* coloredBrush)
{
  for (int color = 0; color < numOfColors; color++)
  {
    coloredBrush[color].DeleteObject();
  }
}

void PaintMethodsWin32::paintCoordinateSystemToCDC(CDC& dc)
{
  PaintMethodsWin32::paintCoordinateSystemToCDC(dc, 3100, -600, 2100, -2100, 100);
}

void PaintMethodsWin32::paintCoordinateSystemForKickSelectionToCDC(CDC& dc)
{
  PaintMethodsWin32::paintCoordinateSystemToCDC(dc,  
    KickSelectionTable::xRange * 10, 0, 
    KickSelectionTable::yRange * 10 / 2, -KickSelectionTable::yRange * 10 / 2, 
    10);
}

void PaintMethodsWin32::paintCoordinateSystemToCDC
(
 CDC& dc, 
 double maxX,
 double minX,
 double maxY,
 double minY,
 double spacing
 )
{
  CPen pen1, pen2, pen3;
  CPen* oldPen = 0;
  pen1.CreatePen(PS_SOLID, 1, RGB(190, 190, 190));
  pen2.CreatePen(PS_SOLID, 1, RGB(0, 0, 0));
  pen3.CreatePen(PS_SOLID, 2, RGB(0, 0, 0));
  oldPen = dc.SelectObject(&pen1);
  
  for (int x=(int)(minX / spacing); x<= (int)(maxX / spacing); x++)
  {
    if(x%5 == 0) dc.SelectObject(&pen2);
    else dc.SelectObject(&pen1);
    dc.MoveTo((int)(x*spacing),(int)(minY));
    dc.LineTo((int)(x*spacing),(int)(maxY));
  }
  
  for (int y=(int)(minY / spacing); y<= (int)(maxY / spacing); y++)
  {
    if(y%5 == 0) dc.SelectObject(&pen2);
    else dc.SelectObject(&pen1);
    dc.MoveTo((int)(minX),(int)(y*spacing));
    dc.LineTo((int)(maxX),(int)(y*spacing));
  }
  
  maxX += 2*spacing; minX -= spacing;
  maxY += 2*spacing; minY -= spacing;
  dc.SelectObject(&pen3);
  // x axis
  dc.MoveTo((int)(maxX),0);
  dc.LineTo((int)(minX),0);
  dc.MoveTo((int)(maxX),0);
  dc.LineTo((int)((maxX-1*spacing)),(int)(0.3*spacing));
  dc.MoveTo((int)(maxX),0);
  dc.LineTo((int)((maxX-1*spacing)),(int)(-0.3*spacing));
  
  // y axis
  dc.MoveTo(0,(int)(minY));
  dc.LineTo(0,(int)(maxY));
  dc.MoveTo(0,(int)(maxY));
  dc.LineTo((int)(0.3*spacing),(int)((maxY-1*spacing)));
  dc.MoveTo(0,(int)(maxY));
  dc.LineTo((int)(-0.3*spacing),(int)((maxY-1*spacing)));
  
  dc.SelectObject(oldPen);
  pen1.DeleteObject();
  pen2.DeleteObject();
  pen3.DeleteObject();
}

void PaintMethodsWin32::paintKickSegmentsToCDC
(
 CDC& dc, 
 int currentWidth,
 int currentHeight,
 int numberOfSegments,
 int indexOfHighlightedSegment,
 int indexOfSelectedSegment,
 int frameSize)
{
  CPen pen1, pen2, pen3, pen4;
  CPen* oldPen = 0;
  pen1.CreatePen(PS_SOLID, 1, RGB(0, 0, 0));
  pen2.CreatePen(PS_SOLID, 10, RGB(100, 100, 200));
  pen3.CreatePen(PS_SOLID, 2, RGB(0, 0, 0));
  pen4.CreatePen(PS_NULL, 0, RGB(0, 0, 0));
  oldPen = dc.SelectObject(&pen4);
  
  double innerCircle = 0.95;
  
  dc.Rectangle(-currentHeight / 4 * 1, -currentWidth / 2, -currentHeight / 4 * 1 + frameSize, currentWidth / 2);
  dc.Rectangle(currentHeight / 4 * 3, currentWidth / 2, currentHeight / 4 * 3 - frameSize, -currentWidth / 2);
  
  dc.Rectangle(-currentHeight / 4 * 1, -currentWidth / 2, currentHeight / 4 * 3, -currentWidth / 2 + frameSize);
  dc.Rectangle(currentHeight / 4 * 3, currentWidth / 2, -currentHeight / 4 * 1, currentWidth / 2 - frameSize);
  
  dc.SelectObject(&pen3);
  dc.MoveTo(-currentHeight / 4 * 1 + frameSize, -currentWidth / 2 + frameSize);
  dc.LineTo(-currentHeight / 4 * 1 + frameSize, +currentWidth / 2 - frameSize);
  dc.LineTo(+currentHeight / 4 * 3 - frameSize, +currentWidth / 2 - frameSize);
  dc.LineTo(+currentHeight / 4 * 3 - frameSize, -currentWidth / 2 + frameSize);
  dc.LineTo(-currentHeight / 4 * 1 + frameSize, -currentWidth / 2 + frameSize);
  
  for (int segmentIndex = 0; segmentIndex < numberOfSegments; segmentIndex++)
  {
    double radiusForWidth, radiusForHeight, length, outerRadius, innerRadius;
    double angle = -(segmentIndex + 0.5) * pi2 / numberOfSegments + pi_2;
    radiusForWidth = fabs(currentWidth / 2 / cos(angle));
    if(segmentIndex < numberOfSegments / 4 || segmentIndex > numberOfSegments / 4 * 3)
      radiusForHeight = fabs(currentHeight / (4.0 / 3.0) / sin(angle));
    else
      radiusForHeight = fabs(currentHeight / (4) / sin(angle));
    if(radiusForWidth < radiusForHeight) length = fabs(frameSize / cos(angle));
    else length = fabs(frameSize / sin(angle));
    outerRadius = min(radiusForWidth, radiusForHeight);
    innerRadius = outerRadius - length;
    
    dc.SelectObject(&pen3);
    innerRadius = 100;
    dc.MoveTo((int)(sin(angle)*outerRadius),(int)(cos(angle)*outerRadius));
    dc.LineTo((int)(sin(angle)*innerRadius),(int)(cos(angle)*innerRadius));
  }
  
  dc.SelectObject(&pen2);
  double radiusForWidth, radiusForHeight, length, outerRadius, innerRadius;
  double angle = -(indexOfSelectedSegment + 0.0) * pi2 / numberOfSegments + pi_2;
  radiusForWidth = fabs(currentWidth / 2 / cos(angle));
  if(indexOfSelectedSegment < numberOfSegments / 4 || indexOfSelectedSegment > numberOfSegments / 4 * 3)
    radiusForHeight = fabs(currentHeight / (4.0 / 3.0) / sin(angle));
  else
    radiusForHeight = fabs(currentHeight / (4) / sin(angle));
  if(radiusForWidth < radiusForHeight) length = fabs(frameSize / cos(angle));
  else length = fabs(frameSize / sin(angle));
  outerRadius = min(radiusForWidth, radiusForHeight);
  innerRadius = outerRadius - length;
  
  dc.MoveTo((int)(sin(angle)*outerRadius),(int)(cos(angle)*outerRadius));
  dc.LineTo((int)(sin(angle)*innerRadius),(int)(cos(angle)*innerRadius));
  
  dc.SelectObject(&pen3);
  angle = -(indexOfHighlightedSegment + 0.0) * pi2 / numberOfSegments + pi_2;
  radiusForWidth = fabs(currentWidth / 2 / cos(angle));
  if(indexOfHighlightedSegment < numberOfSegments / 4 || indexOfHighlightedSegment > numberOfSegments / 4 * 3)
    radiusForHeight = fabs(currentHeight / (4.0 / 3.0) / sin(angle));
  else
    radiusForHeight = fabs(currentHeight / (4) / sin(angle));
  if(radiusForWidth < radiusForHeight) length = fabs(frameSize / cos(angle));
  else length = fabs(frameSize / sin(angle));
  outerRadius = min(radiusForWidth, radiusForHeight);
  innerRadius = outerRadius - length;
  
  dc.MoveTo((int)(sin(angle)*outerRadius),(int)(cos(angle)*outerRadius));
  dc.LineTo((int)(sin(angle)*innerRadius),(int)(cos(angle)*innerRadius));
  
  dc.SelectObject(oldPen);
  pen1.DeleteObject();
  pen2.DeleteObject();
  pen3.DeleteObject();
  pen4.DeleteObject();
}

void PaintMethodsWin32::paintFrameForScalingToCDC(CDC& dc)
{
  double maxX =  4000;
  double minX = -1000;
  double maxY =  2500;
  double minY = -2500;
  
  CPen pen;
  CPen* oldPen = 0;
  pen.CreatePen(PS_SOLID, 1, RGB(0, 0, 0));
  oldPen = dc.SelectObject(&pen);
  
  dc.MoveTo((int)(minX),(int)(minY));
  dc.LineTo((int)(minX),(int)(maxY));
  dc.LineTo((int)(maxX),(int)(maxY));
  dc.LineTo((int)(maxX),(int)(minY));
  dc.LineTo((int)(minX),(int)(minY));
  
  dc.SelectObject(oldPen);
  pen.DeleteObject();
}

void PaintMethodsWin32::paintRobotToCDC(CDC& dc)
{
  double maxX =  75;
  double minX =  -150;
  double maxY =  60;
  double minY =  -60;
  
  CPen pen;
  CPen* oldPen = 0;
  pen.CreatePen(PS_SOLID, 2, RGB(0, 0, 0));
  oldPen = dc.SelectObject(&pen);
  
  dc.Rectangle((int)(minX), (int)(minY), (int)(maxX), (int)(maxY));
  dc.MoveTo((int)(maxX), 0);
  dc.LineTo(0,(int)(minY));
  dc.LineTo(0,(int)(maxY));
  dc.LineTo((int)(maxX), 0);
  
  dc.SelectObject(oldPen);
  pen.DeleteObject();
}


void PaintMethodsWin32::paintBallPerceptToCDC
(
 BallPercept& ballPercept, 
 bool bearingBased,
 bool ballAsCross,
 bool lineToBall,
 CDC& dc)
{
  CPen orangePen1, orangePen2, redPen1, redPen2;
  CPen* oldPen = 0;
  orangePen1.CreatePen(PS_SOLID, 1, RGB(255, 127, 63));
  orangePen2.CreatePen(PS_SOLID, 2, RGB(255, 127, 63));
  redPen1.CreatePen(PS_SOLID, 1, RGB(255, 0, 0));
  redPen2.CreatePen(PS_SOLID, 2, RGB(255, 0, 0));
  oldPen = dc.SelectObject(&orangePen1);
  
  Vector2<double> ballOffset;
  
  // size based
  if(!bearingBased)
  {
    ballPercept.getOffsetSizeBased(ballOffset);
        
    dc.SelectObject(&orangePen1);
    if(lineToBall)
    {
      dc.MoveTo(0,0);
      dc.LineTo((int)(ballOffset.x), (int)(ballOffset.y) );
    }
    dc.SelectObject(&orangePen2);
    if(ballAsCross)
    {
      dc.MoveTo((int)(ballOffset.x - ballRadius / 3), (int)(ballOffset.y));
      dc.LineTo((int)(ballOffset.x + ballRadius / 3), (int)(ballOffset.y));
      dc.MoveTo((int)(ballOffset.x), (int)(ballOffset.y - ballRadius / 3));
      dc.LineTo((int)(ballOffset.x), (int)(ballOffset.y + ballRadius / 3));
    }
    else
    {
      dc.Ellipse(
        (int)(ballOffset.x - ballRadius), (int)(ballOffset.y - ballRadius),
        (int)(ballOffset.x + ballRadius), (int)(ballOffset.y + ballRadius));
    }
    
  }
  else
  {
    // bearing based
    ballPercept.getOffsetBearingBased(ballOffset);
    
    dc.SelectObject(&redPen1);
    if(lineToBall)
    {
      dc.MoveTo(0,0);
      dc.LineTo((int)(ballOffset.x), (int)(ballOffset.y) );
    }
    dc.SelectObject(&redPen2);
    if(ballAsCross)
    {
      dc.MoveTo((int)(ballOffset.x - ballRadius / 3), (int)(ballOffset.y));
      dc.LineTo((int)(ballOffset.x + ballRadius / 3), (int)(ballOffset.y));
      dc.MoveTo((int)(ballOffset.x), (int)(ballOffset.y - ballRadius / 3));
      dc.LineTo((int)(ballOffset.x), (int)(ballOffset.y + ballRadius / 3));
    }
    else
    {
      dc.Ellipse(
        (int)(ballOffset.x - ballRadius), (int)(ballOffset.y - ballRadius),
        (int)(ballOffset.x + ballRadius), (int)(ballOffset.y + ballRadius));
    }
  }
  dc.SelectObject(oldPen);
  orangePen1.DeleteObject();
  orangePen2.DeleteObject();
  redPen1.DeleteObject();
  redPen2.DeleteObject();
}

void PaintMethodsWin32::paintBallModelToCDC
(
 BallModel& ballModel, 
 RobotPose& robotPose,
 bool ballAsCross,
 bool lineToBall,
 CDC& dc
)
{
  CPen orangePen1, orangePen2, redPen1, redPen2;
  CPen* oldPen = 0;

  CBrush coloredBrush;
  CBrush* oldBrush;
  coloredBrush.CreateSolidBrush(RGB(255, 127, 63));
  oldBrush = dc.SelectObject(&coloredBrush);

  orangePen1.CreatePen(PS_SOLID, 1, RGB(255, 127, 63));
  orangePen2.CreatePen(PS_SOLID, 2, RGB(255, 127, 63));
  redPen1.CreatePen(PS_SOLID, 1, RGB(255, 0, 0));
  redPen2.CreatePen(PS_SOLID, 2, RGB(255, 0, 0));
  oldPen = dc.SelectObject(&orangePen1);
  
  Vector2<double> ballOffset;
  
  double distanceToBall = Geometry::distanceTo(robotPose.getPose(), ballModel.seen);
  double angleToBall = Geometry::angleTo(robotPose.getPose(), ballModel.seen);
  ballOffset.x = distanceToBall * cos(angleToBall);
  ballOffset.y = distanceToBall * sin(angleToBall);
  
  dc.SelectObject(&orangePen1);
  if(lineToBall)
  {
    dc.MoveTo(0,0);
    dc.LineTo((int)(ballOffset.x), (int)(ballOffset.y) );
  }
  dc.SelectObject(&orangePen2);
  if(ballAsCross)
  {
    dc.MoveTo((int)(ballOffset.x - ballRadius / 3), (int)(ballOffset.y));
    dc.LineTo((int)(ballOffset.x + ballRadius / 3), (int)(ballOffset.y));
    dc.MoveTo((int)(ballOffset.x), (int)(ballOffset.y - ballRadius / 3));
    dc.LineTo((int)(ballOffset.x), (int)(ballOffset.y + ballRadius / 3));
  }
  else
  {
    dc.Ellipse(
      (int)(ballOffset.x - ballRadius), (int)(ballOffset.y - ballRadius),
      (int)(ballOffset.x + ballRadius), (int)(ballOffset.y + ballRadius));
  }
  
  dc.MoveTo((int)(ballOffset.x), (int)(ballOffset.y) );
  dc.LineTo( (int)(ballOffset.x + ballModel.seen.speed.x), (int)(ballOffset.y + ballModel.seen.speed.y) );
  
  double maxX =  75;
  double minX =  -150;
  double maxY =  60;
  double minY =  -60;
  
  if (ballModel.ballState.ballRollsByLeft) {
    dc.MoveTo((int)(maxX), 0);
    dc.LineTo(0,(int)(maxY));
  } else 
  if (ballModel.ballState.ballRollsByRight) {
    dc.MoveTo((int)(maxX), 0);
    dc.LineTo(0,(int)(minY));
  } else 
  if (ballModel.ballState.ballRollsFast) {
    dc.Rectangle((int)(minX), (int)(minY), (int)(maxX), (int)(maxY));
  }
  
  dc.SelectObject(oldPen);
  orangePen1.DeleteObject();
  orangePen2.DeleteObject();
  redPen1.DeleteObject();
  redPen2.DeleteObject();
  dc.SelectObject(oldBrush);
  coloredBrush.DeleteObject();
}






void PaintMethodsWin32::paintCameraMatrixToCDC(
                                               const CameraMatrix& cameraMatrix, 
                                               const CameraInfo& cameraInfo, 
  CDC& dc
)
{
  double distanceToScreen = 100;
  double x = distanceToScreen * tan(cameraInfo.openingAngleWidth / 2.0);
  double y = distanceToScreen * tan(cameraInfo.openingAngleHeight / 2.0);
  
  Vector3<double> camera[4], world[4];
  
  camera[0].x = distanceToScreen;
  camera[0].y = -x;
  camera[0].z = y;
  
  camera[1].x = distanceToScreen;
  camera[1].y = x;
  camera[1].z = y;
  
  camera[2].x = distanceToScreen;
  camera[2].y = x;
  camera[2].z = -y;
  
  camera[3].x = distanceToScreen;
  camera[3].y = -x;
  camera[3].z = -y;
  
  for(int i = 0; i < 4; i++)
  {
    world[i] = cameraMatrix.rotation * camera[i] + cameraMatrix.translation;
    dc.MoveTo((int)(cameraMatrix.translation.x),(int)(cameraMatrix.translation.y));
    dc.LineTo((int)world[i].x, (int)world[i].y);
  }
  dc.MoveTo((int)world[0].x, (int)world[0].y);
  dc.LineTo((int)world[1].x, (int)world[1].y);
  dc.LineTo((int)world[2].x, (int)world[2].y);
  dc.LineTo((int)world[3].x, (int)world[3].y);
  dc.LineTo((int)world[0].x, (int)world[0].y);
}

void PaintMethodsWin32::paintLandmarksPerceptToCDC(LandmarksPercept& landmarksPercept, CDC& dc)
{
  for (int j=0;j<landmarksPercept.numberOfGoals;j++)
  {
    const Goal& goal = landmarksPercept.goals[j];
    
    /*    if(!goal.isOnBorder(goal.x.max))
    {
    Vector2<double> gpleft;
    gpleft.x=cos(goal.x.max)*goal.distance*1.2;
    gpleft.y=sin(goal.x.max)*goal.distance*1.2;
    
      dc.MoveTo(0,0);
      dc.LineTo((int)gpleft.x,(int)gpleft.y);
      }
      if(!goal.isOnBorder(goal.x.min))
      {
      Vector2<double> gpright;
      gpright.x=cos(goal.x.min)*goal.distance*1.2;
      gpright.y=sin(goal.x.min)*goal.distance*1.2;
      
        dc.MoveTo(0,0);
        dc.LineTo((int)gpright.x,(int)gpright.y);
        }
    */  
    Pose2D gp;
    gp.translation.x = cos(goal.angle)*goal.distance;
    gp.translation.y = sin(goal.angle)*goal.distance;
    
    dc.MoveTo(0,0);
    dc.LineTo((int) gp.translation.x,(int) gp.translation.y);
    
    Pose2D leftCorner(gp);
    leftCorner.rotation = goal.angle + goal.rotation;
    leftCorner = leftCorner + Pose2D(0,yPosLeftGoal);
    
    Pose2D rightCorner(gp);
    rightCorner.rotation = goal.angle + goal.rotation;
    rightCorner = rightCorner + Pose2D(0,yPosRightGoal);
    
    dc.MoveTo((int)leftCorner.translation.x, (int)leftCorner.translation.y);
    dc.LineTo((int)rightCorner.translation.x, (int)rightCorner.translation.y);
  }
}

void PaintMethodsWin32::paintLinesPerceptToCDC(LinesPercept& linesPercept, CDC& dc)
{
  CPen coloredPen;
  CPen* oldPen = 0;
  COLORREF color[7] = {
    RGB(100,100,100), // gray
      RGB(  0,255,  0), // green
      RGB(230,230,  0), // yellow
      RGB(  0,255,255), // skyblue
      RGB(255,  0,  0), // red
      RGB(  0,  0,255), // blue
      RGB(255,127,  0) // orange
  };
  for(int i = 0; i < LinesPercept::numberOfTypes; ++i)
  {
    coloredPen.CreatePen(PS_SOLID, 2, color[i]);
    oldPen = dc.SelectObject(&coloredPen);
    for(int j = 0; j < linesPercept.numberOfPoints[i]; j++)
    {
      POINT points[4];
      points[0].x = (int)(linesPercept.points[i][j].x) - 2; points[0].y = (int)(linesPercept.points[i][j].y) - 2;
      points[1].x = (int)(linesPercept.points[i][j].x) + 2; points[1].y = (int)(linesPercept.points[i][j].y) - 2;
      points[2].x = (int)(linesPercept.points[i][j].x) + 2; points[2].y = (int)(linesPercept.points[i][j].y) + 2;
      points[3].x = (int)(linesPercept.points[i][j].x) - 2; points[3].y = (int)(linesPercept.points[i][j].y) + 2;
      dc.Polygon(points, 4);
    }
    dc.SelectObject(oldPen);
    coloredPen.DeleteObject();
  }
}

void PaintMethodsWin32::paintObstaclesModelToCDC(ObstaclesModel& obstaclesModel, CDC& dc)
{
  CPen grayPen, redPen, greenPen, blackPen, bluePen, thinBlackPen;
  CPen* oldPen = 0;
  grayPen.CreatePen(PS_SOLID, 3, RGB(127, 127, 127));
  redPen.CreatePen(PS_SOLID, 3, RGB(255, 0, 0));
  greenPen.CreatePen(PS_SOLID, 3, RGB(0, 255, 0));
  bluePen.CreatePen(PS_SOLID, 3, RGB(0, 0, 255));
  blackPen.CreatePen(PS_SOLID, 3, RGB(0, 0, 0));
  thinBlackPen.CreatePen(PS_SOLID, 1, RGB(0, 0, 0));
  oldPen = dc.SelectObject(&blackPen);
  
  Vector2<double> point, point2, lastPoint;
  double angle, openingAngle;
  angle = ObstaclesModel::getAngleOfSector(ObstaclesModel::numOfSectors-1);
  openingAngle = tan(pi / ObstaclesModel::numOfSectors);
  
  lastPoint.x = cos(angle) * obstaclesModel.distance[ObstaclesModel::numOfSectors-1]
    - sin(angle) * obstaclesModel.distance[ObstaclesModel::numOfSectors-1]*openingAngle;
  lastPoint.y = sin(angle) * obstaclesModel.distance[ObstaclesModel::numOfSectors-1]
    + cos(angle) * obstaclesModel.distance[ObstaclesModel::numOfSectors-1]*openingAngle;
  
  for(int i = 0; i < ObstaclesModel::numOfSectors; i++)
  {
    angle = ObstaclesModel::getAngleOfSector(i);
    
    point.x = cos(angle) * obstaclesModel.distance[i]
      - sin(angle) * obstaclesModel.distance[i]*openingAngle;
    point.y = sin(angle) * obstaclesModel.distance[i]
      + cos(angle) * obstaclesModel.distance[i]*openingAngle;
    
    point2.x = cos(angle) * obstaclesModel.distance[i]
      + sin(angle) * obstaclesModel.distance[i]*openingAngle;
    point2.y = sin(angle) * obstaclesModel.distance[i]
      - cos(angle) * obstaclesModel.distance[i]*openingAngle;
    
    switch(obstaclesModel.obstacleType[i])
    {
    case ObstaclesPercept::unknown: dc.SelectObject(blackPen); break;
    case ObstaclesPercept::opponent: dc.SelectObject(redPen); break;
    case ObstaclesPercept::teammate: dc.SelectObject(greenPen); break;
    case ObstaclesPercept::goal: dc.SelectObject(bluePen); break;
    case ObstaclesPercept::border: dc.SelectObject(grayPen); break;
    }
    
    dc.MoveTo((int) point2.x, (int)point2.y);
    dc.LineTo((int) point.x, (int)point.y);
    
    dc.SelectObject(thinBlackPen);
    dc.MoveTo((int) point2.x,(int)point2.y);
    dc.LineTo((int) lastPoint.x,(int) lastPoint.y);
    lastPoint = point;
  }
  
  dc.SelectObject(oldPen);
  redPen.DeleteObject();
  greenPen.DeleteObject();
  bluePen.DeleteObject();
  blackPen.DeleteObject();
  thinBlackPen.DeleteObject();
  grayPen.DeleteObject();
  /*  
  // Free parts of goal:
  DebugDrawing::Color freePartColor[2];
  freePartColor[0] = DebugDrawing::Color(127, 0 , 0);
  freePartColor[1] = DebugDrawing::Color(0, 127 , 0);
  
    for(i = 0; i < 2; i++)
    {
    if(obstaclesModel.angleToFreePartOfGoalWasDetermined[i])
    {
    Vector2<double> vLeft = (
    Pose2D(obstaclesModel.angleToFreePartOfGoal[i] + obstaclesModel.widthOfFreePartOfGoal[i] / 2.0) + 
        Pose2D(Vector2<double>(obstaclesModel.distanceToFreePartOfGoal[i] , 0))
    ).translation;
    
      Vector2<double> vRight = (
      Pose2D(obstaclesModel.angleToFreePartOfGoal[i] - obstaclesModel.widthOfFreePartOfGoal[i] / 2.0) + 
        Pose2D(Vector2<double>(obstaclesModel.distanceToFreePartOfGoal[i], 0))
      ).translation;
      
        dc.MoveTo(0,0);
        dc.LineTo(int(vLeft.x), int(vLeft.y));
        
          dc.MoveTo(0,0);
          dc.LineTo(int(vRight.x), int(vRight.y));
          }
          }
          
            // Next free teammate
            if(obstaclesModel.angleToNextFreeTeammateWasDetermined)
            {
            Vector2<double> v = (
            Pose2D(obstaclesModel.angleToNextFreeTeammate) + 
      Pose2D(Vector2<double>(obstaclesModel.distanceToNextFreeTeammate, 0))
            ).translation;
            
              dc.MoveTo(0,0);
              dc.LineTo(int(v.x), int(v.y));
              }
  */
}
void PaintMethodsWin32::paintObstaclesPerceptToCDC(ObstaclesPercept& obstaclesPercept, CDC& dc)
{
  CPen groundPen, darkGreenPen;
  CPen grayPen, redPen, greenPen, blackPen, bluePen, thinBlackPen;
  CPen* oldPen = 0;
  grayPen.CreatePen(PS_SOLID, 2, RGB(127, 127, 127));
  redPen.CreatePen(PS_SOLID, 2, RGB(255, 0, 0));
  greenPen.CreatePen(PS_SOLID, 2, RGB(0, 255, 0));
  bluePen.CreatePen(PS_SOLID, 2, RGB(0, 0, 255));
  blackPen.CreatePen(PS_SOLID, 2, RGB(0, 0, 0));
  thinBlackPen.CreatePen(PS_SOLID, 1, RGB(0, 0, 0));
  groundPen.CreatePen(PS_SOLID, 1, RGB(0, 255, 0));
  darkGreenPen.CreatePen(PS_SOLID, 3, RGB(0, 127, 0));
  
  oldPen = dc.SelectObject(&groundPen);
  for(int j = 0; j < obstaclesPercept.numberOfPoints; j++)
  {
    dc.SelectObject(&groundPen);
    dc.MoveTo((int)(obstaclesPercept.nearPoints[j].x), (int)(obstaclesPercept.nearPoints[j].y));
    dc.LineTo((int)(obstaclesPercept.farPoints[j].x), (int)(obstaclesPercept.farPoints[j].y));
    
    switch(obstaclesPercept.obstacleType[j])
    {
    case ObstaclesPercept::unknown: dc.SelectObject(blackPen); break;
    case ObstaclesPercept::opponent: dc.SelectObject(redPen); break;
    case ObstaclesPercept::teammate: dc.SelectObject(greenPen); break;
    case ObstaclesPercept::goal: dc.SelectObject(bluePen); break;
    case ObstaclesPercept::border: dc.SelectObject(grayPen); break;
    }
    dc.Ellipse(
      (int)(obstaclesPercept.farPoints[j].x) - 4, (int)(obstaclesPercept.farPoints[j].y - 4),
      (int)(obstaclesPercept.farPoints[j].x) + 4, (int)(obstaclesPercept.farPoints[j].y + 4)
      );
    
    
    if(obstaclesPercept.farPointIsOnImageBorder[j])
    {
      dc.SelectObject(&darkGreenPen);
      dc.Ellipse(
        (int)(obstaclesPercept.farPoints[j].x) - 2, (int)(obstaclesPercept.farPoints[j].y - 2),
        (int)(obstaclesPercept.farPoints[j].x) + 2, (int)(obstaclesPercept.farPoints[j].y + 2)
        );
    }
  }
  
  dc.SelectObject(oldPen);
  darkGreenPen.DeleteObject();
  groundPen.DeleteObject();
  redPen.DeleteObject();
  greenPen.DeleteObject();
  bluePen.DeleteObject();
  blackPen.DeleteObject();
  thinBlackPen.DeleteObject();
  grayPen.DeleteObject();
}

void PaintMethodsWin32::paintPSDPerceptToCDC(PSDPercept& psdPercept, CDC& dc)
{
  CPen greenPen, redPen;
  CPen* oldPen = 0;
  greenPen.CreatePen(PS_SOLID, 2, RGB(0, 255, 0));
  redPen.CreatePen(PS_SOLID, 2, RGB(255, 0, 0));
  oldPen = dc.SelectObject(&redPen);
  
  for(int i = 0; i < psdPercept.numOfPercepts; i++)
  {
    if(psdPercept[i].tooFarAway) dc.SelectObject(&greenPen);
    else dc.SelectObject(&redPen);
    dc.MoveTo(0,0);
    dc.LineTo((int)(psdPercept[i].x), (int)(psdPercept[i].y) );
  }
  dc.SelectObject(oldPen);
  redPen.DeleteObject();
  greenPen.DeleteObject();
}

void PaintMethodsWin32::paintImageProjectionOnGroundToCDC(
                                                          const CameraMatrix& cameraMatrix, 
                                                          const CameraInfo& cameraInfo, 
                                                          const Image& image,
  CDC& dc
)
{
  Image rgbImage;
  rgbImage.convertFromYUVToRGB(image);
  
  CPen noPen;
  CPen* oldPen = 0;
  noPen.CreatePen(PS_NULL, 0, RGB(0,0,0));
  oldPen = dc.SelectObject(&noPen);
  CBrush coloredBrush;
  CBrush* oldBrush;
  for(int i = 0; i < image.cameraInfo.resolutionWidth; i++)
  {
    for(int j = 0; j < image.cameraInfo.resolutionHeight; j++)
    {
      Vector2<int> quadCorners[4];
      if(
        Geometry::calculatePointOnField(i,     j, cameraMatrix, image.cameraInfo, quadCorners[0]) &&
        Geometry::calculatePointOnField(i + 1, j, cameraMatrix, image.cameraInfo, quadCorners[1]) &&
        Geometry::calculatePointOnField(i + 1, j + 1, cameraMatrix, image.cameraInfo, quadCorners[2]) &&
        Geometry::calculatePointOnField(i,     j + 1, cameraMatrix, image.cameraInfo, quadCorners[3]) 
        )
      {
        POINT points[4];
        points[0].x = (long)(quadCorners[0].x); points[0].y = (long)(quadCorners[0].y);
        points[1].x = (long)(quadCorners[1].x); points[1].y = (long)(quadCorners[1].y);
        points[2].x = (long)(quadCorners[2].x); points[2].y = (long)(quadCorners[2].y);
        points[3].x = (long)(quadCorners[3].x); points[3].y = (long)(quadCorners[3].y);
        
        coloredBrush.CreateSolidBrush(RGB(rgbImage.image[j][0][i], rgbImage.image[j][1][i], rgbImage.image[j][2][i]));
        oldBrush = dc.SelectObject(&coloredBrush);
        
        dc.Polygon(points, 4);
        
        dc.SelectObject(oldBrush);
        coloredBrush.DeleteObject();
      }
    }
  }
  dc.SelectObject(oldPen);
  noPen.DeleteObject();
}


/*
* Change log :
* 
* $Log: PaintMethodsWin32.cpp,v $
* Revision 1.3  2004/06/20 14:27:24  kindler
* - Removed arkward scale-parameter in paint functions, all drawings are now done in world coordinates.
* - It's easy now to calculate the inverse transformation for interactive DebugDrawings (like the CeilingCamView)
*
* Revision 1.2  2004/05/27 18:49:17  kerdels
* added a small 5 frames sliding average for the relative ballspeed,
* added new ballState Representation and adjusted the apropriate files
*
* Revision 1.1.1.1  2004/05/22 17:30:13  cvsadm
* created new repository GT2004_WM
*
* Revision 1.11  2004/04/07 13:00:44  risler
* ddd checkin after go04 - second part
*
* Revision 1.3  2004/03/31 11:35:09  mkunz
* warnings removed
*
* Revision 1.2  2004/03/30 17:32:31  mkunz
* draw ball speed and ball status
*
* Revision 1.1.1.1  2004/03/29 08:28:43  Administrator
* initial transfer from tamara
*
* Revision 1.10  2004/03/25 21:21:49  juengel
* Implemented paintBallModelToCDC.
*
* Revision 1.9  2004/03/16 14:00:23  juengel
* Integrated Improvments from "Gnne"
* -ATH2004ERS7Behavior
* -ATHHeadControl
* -KickSelectionTable
* -KickEditor
*
* Revision 1.9  2004/03/11 17:28:42  juengel
* Increased penWidth for segments.
*
* Revision 1.8  2004/03/10 22:02:46  juengel
* Made frame size variable.
*
* Revision 1.7  2004/03/10 20:56:52  juengel
* Moved origin of KickEditor.
*
* Revision 1.6  2004/03/10 16:01:51  juengel
* Improved kick segment drawing.
*
* Revision 1.5  2004/03/10 14:52:57  juengel
* Improved kick segment drawing.
*
* Revision 1.4  2004/03/09 18:42:18  juengel
* Moved PaintStyle to PaintMethodsWin32
*
* Revision 1.3  2004/03/09 13:41:41  juengel
* Added parameters to paintCoordinateSystem.
*
* Revision 1.2  2004/03/09 01:15:14  juengel
* Visualization of kick sectors.
*
* Revision 1.8  2004/03/01 11:44:40  juengel
* visualization of obstacle types
*
* Revision 1.7  2004/02/28 13:59:31  juengel
* Added ObstacleType to ObstaclesModel.
*
* Revision 1.6  2004/02/12 14:11:42  juengel
* goals
*
* Revision 1.5  2004/02/07 15:48:23  juengel
* Added buffer edit box to RadarViewer.
* Added some different modes of view for the BallPercept
*
* Revision 1.4  2004/02/05 14:32:28  juengel
* Added visualization of obstacles percept and bearing based ball offset.
*
* Revision 1.3  2004/02/02 10:09:49  juengel
* Redesign of RadarViewer.
*
* Revision 1.2  2003/10/30 22:46:28  roefer
* DebugDrawings now preserve sequence of drawing instructions
*
* Revision 1.1  2003/10/07 10:11:08  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:27  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.5  2003/03/31 03:52:12  osterhues
* changed ColorTable64 to ColorTable
* optimized rgb to r, g, b conversion
*
* Revision 1.4  2002/11/12 10:49:02  juengel
* New debug drawing macros - second step
* -moved /Tools/Debugging/PaintMethods.h and . cpp
*  to /Visualization/DrawingMethods.h and .cpp
* -moved DebugDrawing.h and .cpp from /Tools/Debugging/
*  to /Visualization
*
* Revision 1.3  2002/10/10 11:46:12  juengel
* Scalable version for paintDebugDrawingToCDC(...) added.
*
* Revision 1.2  2002/09/22 18:40:51  risler
* added new math functions, removed GTMath library
*
* Revision 1.1  2002/09/10 15:49:12  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
*
* Revision 1.1.1.1  2002/05/10 12:40:27  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.3  2002/02/11 00:53:33  loetzsch
* ::Paint Methoden in ::paint umbenannt
*
* Revision 1.2  2002/01/22 14:51:38  juengel
* TestDataGenerator eingefhrt
*
* Revision 1.1  2002/01/04 14:36:00  juengel
* no message
*
*
*/
